/*
**  Library file GnuDraw.ox
**
**  Purpose:
**    Contain routines for drawing plots, with high level of
**    compatibility with Ox drawing routines
**
**  Date:
**    17/1/2002
**
**  Version:
**    3.03      Support for Ox V2.20 dropped
**    4.00      Changed the structure again; included support for
**              tsmod executable of wgnuplot
**    4.1       Get rid of execgnu.bat, and of nag-screen in the
**              windows version when plotting a graph.
**    4.2       Version which places all kind of files in the
**              tmp directory, also on Windows
**    4.3       Allowing for plotting of intra-day time series data
**    4.4       Extended possibilities of DrawText to intra-day,
**              extensive internal maintenance
**    4.5       Introduced further compatibility with Ox 4.02; for
**              compatibility with Ox 3.40, comment in the timeofday
**              function below.
**    5.0       Changed structure for time series plots
**    5.1       Including support for font size, and shifting colors
**    5.2       Improved fan-plot support
**    5.3       Improved asymmetric ZMODE support, added DrawSymbol
**    5.4       Altered 3d support, better fan plots
**    5.5       Further compatibility with OxDraw
**    5.6       Allowing better support for log-axes, and for time at
**              y-axis; more robust checking of inputs
**    6.0       Improved output, robuster options
**    6.1       Implemented ADJ_AREA_3D, and adapted DrawHistogram
**    6.2       Added ADJ_PAPERSCALE, ADJ_COLORMODEL
**    6.3       Improved DrawText; turned DrawXYZ around, such
**              that it is compatible with Ox7
**    7.0       Upgraded version, compilation of c-code for Ox7
**
**  Author:
**    (c) 2003-2013 Charles Bos
**    email: c.s.bos at vu dot nl
**    http://personal.vu.nl/c.s.bos/
**
**  Based on:
**    oxdraw plotting package by Jurgen Doornik.
*/
#pragma array_base(0)

#include <oxfloat.h>
#include <oxprob.h>
#include <arma.h>

// #include <lib/densest.ox>
#import <lib/densest>
#include "packages/gnudraw/lib/libkern.ox"
#include "packages/gnudraw/lib/gnudraw_lib.ox"

/*
**  SaveDrawWindow(const sFilename, ...)
**  SaveDrawWindow(const sFilename, const iFont)
**  SaveDrawWindow(const sFilename, const dXSize, const dYSize)
**  SaveDrawWindow(const sFilename, const iFont, const dXSize, const dYSize)
**
**  sFilename
**      in: valid file name, or a file handle
**  iFont
**      in  integer, font size
**  dXSize
**      in: double, indicating x-size
**  dYSize
**      in: double, indicating y-size
**
**  No return value.
**
**  Saves the current graph to a file with the same basename as sFilename,
**  and extension .plt (containing information on final output type) or .plb
**  (no information on output type included). The ASCII file is
**  prepared to be translated by GnuPlot to a file with the specified
**  extension in the corresponding format. The following formats are supported:
**      .eps/epc  Encapsulated PostScript
**      .gif/png  png format, for inclusion on Web pages
**      .aux/tex  PSLaTeX output
**      .svg      SVG output
**      .fig      xfig output
**      .pdf      PDF output
**      .plb      General, output type can be specified later using a
**                script
**      .scr      No final output, except screen output (deprecated)
**      .tmp      No final output, except screen output. File is deleted
**                after use  (deprecated)
**      .win      Windows screen output. File is deleted after use
**      .x11      X11 screen output. File is deleted after use
*/
SaveDrawWindow(const sFilename, ...)
{
  decl ir, ava, fh, sDirectory, sBasename, iOuttype, i, j, iN,
       inLoc, aSetOld, aaPlot, sCommand, bSurf, bCont,
       sLine, iCnt, vContour, vMiss, vFont, vSize,
       iNPlots, sFont, iSize, vL, vSpec, vRGB, bEmpty;

  if (sizeof(s_Fig) == 0)
    return;

  ava= va_arglist();
  if ((sizeof(ava) == 1) || (sizeof(ava) == 3))
    {
      if (ava[0] != -1)
        // Standard font of size 300, times 10
        SetDraw(SET_FONT, M_NAN, 30*ava[0], M_NAN);
      ava= sizeof(ava) > 1 ? ava[1:] : {};
    }
  if (sizeof(ava) == 2)
    DrawAdjust(ADJ_PAPERSCALE, ava[0], ava[1]);

  // Prepare font
  vFont= s_Fig[F_GLOB][G_DEFFONT];
  iNPlots= s_Fig[F_NPLOTS];
  vSize= s_Fig[F_SIZES];

  vFont[1]*= s_Fig[F_SIZES][1];   // Scale font with paperscale...

  if (!iNPlots)
    return;

  ir= s_ParseFilename(sFilename, &sDirectory, &sBasename, &iOuttype) &&
      // s_AdaptSizeLabels(&vSize) &&
      s_InitOutfile(sDirectory~sBasename, iOuttype, &fh);

  #ifdef GNUDRAW_DEBUG
    println ("Trying to save ", sFilename);
    if (!ir)
      println ("\nTrouble with filename or initialisation ir=", ir);
  #endif

  if (!ir)
    return;

  bEmpty= s_DropMissingLines();

  vRGB= !ismissing(s_Fig[F_PLOT][0][P_PAPERCOLOR])
    ? s_Fig[F_PLOT][0][P_PAPERCOLOR]
    : s_Fig[F_GLOB][G_PAPERCOLOR];
  if (!s_SetTerm(fh, sBasename, vSize, s_Fig[F_PLOT][0][P_COLORMODEL],
          s_Fig[F_GLOB][G_TITLE][0],
          vRGB, iOuttype))
    return;
  s_SetFont(fh, vFont, iNPlots, iOuttype);

  // Set the color scheme
  s_SetColors(fh, s_Fig[F_GLOB][G_COLORS]);

  // Write the tables
  bCont= s_WriteTables(fh, sBasename);

  inLoc= 0;
  aSetOld= {};
  for (i= 0; i < iNPlots; ++i)
    {
      aaPlot= &(s_Fig[F_PLOT][i]);
      bSurf= aaPlot[0][P_TYPE][T_XYZ];

      [sFont, iSize]= s_GetFontTypeSize(aaPlot[0][P_AXISLABEL][1:2], 1,
                                        iOuttype, FALSE);
// println ("Font ", sFont, " of size ", iSize, " for type ", iOuttype);

      // Set the range key, and surface settings, into P_SET
      s_DateSpec(&vL, &vSpec, aaPlot);
      s_SetRange(aaPlot, vSpec);
      s_SetKey(aaPlot);
      s_SetSurf(aaPlot);
      s_SetAxis(aaPlot, aaPlot[0][P_AXISLABEL], sFont, iSize);
      s_SetScale(aaPlot, vL, vSpec);
      s_SetText(aaPlot, iOuttype);

      // Keep track of the plots with specified location; put
      //   only other plots in standard slots. Originally I used
      //   i-inLoc in call to PrepareMult, to put other plots in slots
      //   0..(nPlot-nLoc). Now I put the other plots in slots
      //   i, counting the replaced plots as well.
      inLoc+=
        s_SetMultPlot(aaPlot, i, iNPlots, vSize, bEmpty);

      // Prepare all settings into the file
      s_WriteSet(fh, aSetOld, aaPlot[0][P_SET]);

      if (aaPlot[0][P_NLINES] > 0)
        {
          vContour= s_WriteFilePlot(fh, aaPlot[0], TRUE,
            sprint(sBasename, i), -1,
            (iOuttype != <O_TEX, O_ETEX, O_WIN>) ? iSize : -1,
            int(sumr(vSpec)));
          s_WriteLines(fh, aaPlot[0], vSpec, vContour);
        }
      // Save resets
      aSetOld= aaPlot[0][P_SET];
    }
  s_CloseOutfile(&fh, iNPlots, sDirectory, sBasename, iOuttype);
}

/*
**  CloseDrawWindow()
**
**  Purpose:
**    Clean up settings on lines, keys and others
*/
CloseDrawWindow()
{
  if (sizeof(s_Fig))
    s_Fig[F_GLOB][G_INITFIGURE]= TRUE;
  s_InitFigure();
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  Draw(const iArea, const mYt);
**  Draw(const iArea, const mYt, const asKey);
**  Draw(const iArea, const mYt, const dXfirst, const dXstep);
**
**      iArea
**          in: int, area index
**      mYt
**          in: m x T matrix with m rows of data
**      asKey
**          in: (optional) array of strings, names
**      dXfirst
**          in: (optional) double, X-value of first observation, x, default is 1
**      dXstep
**          in: (optional) double, gap between X-values, dx, default is 1
**
**  No return value.
**
**  This function draws m variables against an X variable, where the X variable consists of
**  evenly spaced observations x, x+dx, x+2dx, x+3dx, .... Each variable is drawn by linking up
**  the points. The first line index is 2.
*/
Draw(const iArea, const mYt, ...)
{
  decl ava, vX, iT, dXFirst, dXStep, asKey;

  s_InitFigure();
  if (iArea < 0)
    return;

  ava= va_arglist();
  asKey= (sizeof(ava) > 0) && (isarray(ava[0]) || isstring(ava[0])) ? ava[0] : "";
  dXFirst= (sizeof(ava) > 0) && (isint(ava[0]) || isdouble(ava[0])) ? ava[0] : 1;
  dXStep= (sizeof(ava) > 1) && (isint(ava[1]) || isdouble(ava[1])) ? ava[1] : 1;

  iT= columns(mYt);
  vX= range(0, iT-1)*dXStep + dXFirst;
  DrawXMatrix(iArea, mYt, asKey, vX, "", 0, 2);
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawAbout();
**
**  Purpose:
**    Give version information on GnuDraw on screen
*/
DrawAbout()
{
  println ("GnuDraw version 7.1 -- 14/3/2014");
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawAcf(const iArea, const vY, const sY, const cLag, ...);
**  DrawAcf(const iArea, const mY, const sY, const cLag, const fAcf,
**          const fPacf, const fErrorBand, int iIndex, const fBar,
**          const fJoin);
**
**  iArea
**      in: int, area index
**  mY
**      in: k x T matrix, each row is a new plot
**  sY
**      in: string, variable name, or array of strings (k > 1)
**  cLag
**      in: int, highest lag to be used in the ACF
**  fAcf
**      in: int, TRUE: draw ACF (optional argument, drawn by default)
**  fPacf
**      in: int, TRUE: draw PACF (optional argument, not drawn by default)
**  fErrorBand
**      in: int, TRUE: draw error bands (optional argument, not drawn by default)
**  iIndex
**      in: int, line index, see default line attributes, (optional argument,
**      default is 2).
**  iBar
**      in: int, 1: draw bar plot, 0:draw index plot, 2: use lines (optional argument,
**      using bars by default)
**  fJoin
**      in: int, TRUE: draw acf's together in one plot, else draw separately
**      (optional argument, drawing separate plots by default)
**
**  Return value:
**    none
*/
DrawAcf(const iArea, const mY, const sY, const cLag, ...)
{
  decl ava, asY, fAcf, fPacf, fErrorBand, iIndex, iBar, fJoin, nDim, nP,
       dD, dM, mACF, mPACF, ilArea, ilIndex, vIndex, ir, mX, sKey, i, j,
       dErrBand, sType, sTit, dOffset, iN, vYL;

  s_InitFigure();
  if (iArea < 0)
    return;

  ava= va_arglist();
  fAcf= (sizeof(ava) > 0) ? int(ava[0]) : TRUE;
  fPacf= (sizeof(ava) > 1) ? int(ava[1]) : FALSE;
  fErrorBand= (sizeof(ava) > 2) ? int(ava[2]) : FALSE;
  iIndex= (sizeof(ava) > 3) ? vecr(ava[3]) : s_Fig[F_GLOB][G_DEFLINE][0];
  iBar= (sizeof(ava) > 4) ? int(ava[4]) : TRUE;
  fJoin= (sizeof(ava) > 5) ? int(ava[5]) : s_Fig[F_GLOB][G_JOIN];
  asY= isarray(sY) ? sY : {sY};

  if (max(fAcf, fPacf) == 0)
    {
      if (binand(s_Fig[F_GLOB][G_WARN], 1))
        oxwarning("No ACF/PACF requested");
      return;
    }

  ilArea= iArea;
  ilIndex= iIndex;

  nDim= rows(mY);
  nP= fJoin ? nDim * (fAcf + fPacf) : (fAcf + fPacf);
  dD= (nP > 1) && (iBar < 2) ? 1/(nP+0.5) : 0;

  // Get sufficient colors for the lines
  vIndex= sizerc(iIndex) < nP
          ? vecr(iIndex)|range(iIndex[0]+sizerc(iIndex), iIndex[0]+nP-1)'
          : vecr(iIndex)';

  // Use filled boxes by default
  sType= iBar == 1 ? sprint("boxes fs solid ", "%3.1f", s_Fig[F_GLOB][G_DEFFILL]) :
         iBar == 0 ? "impulses" :
         "lines";

  dErrBand= 2.0 ./ sqrt(columns(mY));
  dOffset= iBar < 2 ? -0.5*(nP-1) : 0;
  dM= 0;
  ir= 0;

  for (i= j= 0; i < nDim; ++i)
    {
      // For security:
      j= imod(j, nP);

      sKey= (i < sizeof(asY)) ? asY[i] : "";
      vYL= deletec(mY[i][]);
      mACF= sizerc(vYL) ? acf(vYL', cLag)'
                        : constant(M_NAN, 1, cLag+1);
      mX= range(0, cLag);

      if (fAcf)
        {
          sTit= sizeof(sKey) || fPacf ? "ACF "~sKey : "";
          ir= s_CheckSize(mX, mACF, 0, 1, "DrawACF", &iN)
              && s_AppendLine(ilArea, mX[:iN-1]+dOffset*dD,
                              mACF[][:iN-1], <>, <>, sType,
                              vIndex[j++], M_NAN, M_NAN, sTit);
          dM= min(dM, mACF);
        }
      if (fPacf)
        {
          sTit= sizeof(sKey) || fAcf ? "PACF "~sKey : "";
          mPACF= pacf(mACF')';
          ir= s_CheckSize(mX, mPACF, 0, 1, "DrawACF", &iN)
              && s_AppendLine(ilArea, mX[:iN-1]+(dOffset+fAcf)*dD, mPACF[][:iN-1],
                              <>, <>, sType,
                              vIndex[j++], M_NAN, M_NAN, sTit);
          // Original color: ilIndex+fAcf
          dM= min(dM, mPACF);
        }
      s_Fig[F_GLOB][G_LASTPLOT]= ilArea;

      if (fJoin)
        {
          dOffset+= fAcf + fPacf;
          ilIndex+= fAcf + fPacf;
        }

      if (fErrorBand && ((i==nDim-1) || !fJoin))
        {
          mX= 0.5~(iN+dOffset*dD);
          mX= 0.5~(iN+dOffset*dD);
          ir= ir && s_AppendLine(ilArea, mX, constant(-dErrBand, mX), <>, <>,
                           "lines", 0, M_NAN, M_NAN, "")
              && s_AppendLine(ilArea, mX, constant(dErrBand, mX), <>, <>,
                           "lines", 0, M_NAN, M_NAN, "");
          dM= min(dM, -dErrBand);
        }

      if ((i==nDim-1) || !fJoin)
        {
          dM= floor(dM*20)/20;
// No longer using minimum, sometimes troublesome when combining
//   acf plots.
//          DrawAdjust(ADJ_AREA_Y, dM, 1);
          DrawAdjust(ADJ_AREA_Y, M_NAN, 1);
          DrawAdjust(ADJ_AREA_X, .5, ceil(iN+dOffset*dD));
          if (dD)
            s_Fig[F_PLOT][ilArea][P_SET][S_BOXWIDTH]=
              {sprint("set boxwidth ", dD), "set boxwidth"};

          // Show zeroaxis
          s_Fig[F_PLOT][ilArea][P_TICSAXISBORDER][0][0]= TRUE;
          DrawLegend(ilArea, 1, 0, FALSE);

          dM= j= 0;
          ilArea+= 1;
        }

    }

// No return statement, as there was a bug connected to it
//   mACF= acf(mY', cLag)';
//   mPACF= pacf(mACF')';
//   return (fAcf ? mACF : <>) | (fPacf ? mPACF : <>);
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawAdjust(const iType, ...);
**
**      iType
**          in: int, type of adjustment
**      d1 ... d4
**          in: optional arguments
**
**  No return value.
**
**  The expected number of arguments depends on the type of adjustment
*/
DrawAdjust(const iType, ...)
{
  decl ava, dMinY, dMaxY, i, j, iArea, d1, d2, d3, d4, d5, sBase, sExtra,
       sXYZ, iN, vLocation, iIsXYZ, sRet, bGrow,
       iAxis, iTitle, vType, vView, mXY;

  s_InitFigure();

  sExtra= "";
  ava= va_arglist();

  // Make sure that at least one plot was initialized
  iArea= max(s_Fig[F_GLOB][G_LASTPLOT], 0);
  while ((sizeof(s_Fig) < F_LAST) || (s_Fig[F_NPLOTS] <= iArea))
    s_InitPlot();

  d1= (sizeof(ava) > 0) ? ava[0] : M_NAN;
  d2= (sizeof(ava) > 1) ? ava[1] : M_NAN;
  d3= (sizeof(ava) > 2) ? ava[2] : M_NAN;
  d4= (sizeof(ava) > 3) ? ava[3] : M_NAN;
  d5= (sizeof(ava) > 4) ? ava[4] : M_NAN;

  // Check if there is an area with lines in it
  iN= s_Fig[F_PLOT][iArea][P_NLINES];
  if (iType == ADJ_AREAMATRIX)
//     s_Fig[F_GLOB][G_ALIGN]= isnan(d1) ? round(d2) :
//         (isnan(d2) ? -round(d1) : round(d2)~round(d1));
    {
      s_Fig[F_GLOB][G_ALIGN]= round(d2)~round(d1);
      if (d3)   // Box all existing areas
        for (i= 0; i <= iArea; ++i)
          s_Fig[F_PLOT][i][P_TICSAXISBORDER][0][2]= TRUE;
      while (s_Fig[F_NPLOTS] < round(d2) * round(d1))
        s_InitPlot();
    }
  else if (iType == ADJ_ALIGN)
    // Deprecated option.
    //   d1 > 0: Number of columns,
    //   d1 < 0: Number of rows,
    //   d1 == 0: Balanced, default
    if (d1 > 0)
      DrawAdjust(ADJ_AREAMATRIX, M_NAN, round(d1));
    else if (d1 < 0)
      DrawAdjust(ADJ_AREAMATRIX, round(-d1), M_NAN);
    else
      s_Fig[F_GLOB][G_ALIGN]= 0;
  else if (((iType== ADJ_AREA_X) || (iType== ADJ_AREA_Y) || (iType== ADJ_AREA_Z)) && iN)
    {
      bGrow= FALSE;
      if (sizeof(ava) == 2)
        [d1, d2]= ava;
      else if (sizeof(ava) == 3)
        [iArea, d1, d2]= ava;
      else if (sizeof(ava) > 3)
        [iArea, d1, d2, bGrow]= ava;
      if (d1 > d2)
        [d1, d2]= {d2, d1};     // Swap

      d1= min(d1);      // Get rid of matrices, just in case
      d2= max(d2);

// print ("Adjusting ", iArea, " to range ", d1~d2);

      if (iArea >= s_Fig[F_NPLOTS])
        return;
      iN= s_Fig[F_PLOT][iArea][P_NLINES];
      if (iN == 0)
        return;
      iIsXYZ= (iType == ADJ_AREA_Y) + 2*(iType == ADJ_AREA_Z);

      if (s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_AXES][iIsXYZ])
        iIsXYZ+= 3;     // Use second set of axes

      sXYZ= {"x", "y", "z", "x2", "y2", "z2"}[iIsXYZ];

      if (!isstring(d1) && !isstring(d2))
        {
          j= iType == ADJ_AREA_X ? 1 :
             iType == ADJ_AREA_Y ? 2 :
             iType == ADJ_AREA_Z ? 4 : -1;

          if ((j > 0) && binand(s_Fig[F_PLOT][iArea][P_TYPE][T_DMY], j)
              && s_IsYear(d1~d2))
            {
              // oxwarning("Assuming range in ADJ_AREA_X/Y/Z in years, transforming to dayofcalendar()");
              d1= dayofcalendar(d1, 1, 1);
              d2= dayofcalendar(d2, 1, 1);
            }
          if (bGrow)
            {
              d1= min(s_Fig[F_PLOT][iArea][P_LIMITS][iIsXYZ][0], d1);
              d2= max(s_Fig[F_PLOT][iArea][P_LIMITS][iIsXYZ][1], d2);
            }
          s_Fig[F_PLOT][iArea][P_LIMITS][iIsXYZ][0]= double(d1);
          s_Fig[F_PLOT][iArea][P_LIMITS][iIsXYZ][1]= double(d2);
        }
      else
        {
          // Only for backwards compatibility, set range here for strings
          //   Numeric ranges are set in WriteFileUni
          sBase= sprint("set ", sXYZ, "range [");
          sExtra= ismissing(d1) ? sprint(sBase, "*:") :
            (isstring(d1) ? sprint(sBase, "'", d1, "':") :
             sprint(sBase, "%g", double(d1), ":"));
          sExtra~= ismissing(d2) ? sprint("*]") :
            (isstring(d2) ? sprint("'", d2, "']") :
             sprint("%g", double(d2), "]"));
          s_Fig[F_PLOT][iArea][P_SET][S_XRANGE+iIsXYZ]=
            {sExtra, sprint(sBase, "*:*]")};
          // println ("Initially set xrange to\n  ", sExtra);
        }
//       print ("Resulting limits: ", s_Fig[F_PLOT][iArea][P_LIMITS]);
    }
  else if ((iType== ADJ_AREA_P) && (sizeof(ava) >= 4))
    {
      if (sizeof(ava) == 5)
        [iArea, d1, d2, d3, d4]= ava[:4];
      vLocation= d1~d2~d3~d4;
      vLocation= isdotnan(vLocation) .? 1 .: vLocation;
      vLocation= vLocation .<= 1 .? vLocation .:
        vLocation ./ <15000, 10000, 15000, 10000>;
      s_Fig[F_PLOT][iArea][P_LOCATION]=
        vLocation;
    }
  else if (iType== ADJ_AXISHIDE)
    s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][0][:1]= d1;
  else if ((iType == ADJ_AXES2ND) && iN)
    {
      s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_AXES][:1]= d1~d2;

      // Turn the correct borders on
      s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][1][3*d1]=TRUE;
      s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][1][1+3*d2]=TRUE;

      // If this was the first and only line until now, turn
      //   old borders off
      if (iN == 1)
        {
          s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][1][3*!d1]=FALSE;
          s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][1][1+3*!d2]=FALSE;
        }
    }
  else if (iType == ADJ_AXISLABEL)
    s_Fig[F_PLOT][iArea][P_AXISLABEL]= d1~d2~d3~d4;
  else if (iType == ADJ_AXISTEXT)
    {
      if (ismissing(d3))  // Use either last axis, or by default x-axis
        iAxis= s_Fig[F_PLOT][iArea][P_AXIS];
      else
        iAxis= d3 == 1 ? 0 : d3 == 0 ? 1 : d3;
      if (isarray(d1) && ismatrix(d2) && (sizeof(d1) == sizerc(d2)))
        s_Fig[F_PLOT][iArea][P_AXISTEXT][iAxis]= {d1, d2};
    }
  else if ((iType== ADJ_COLOR) && iN)
    {
      // Copy over linewidth to fourth argument, if needed
      d4= isstring(d3) ? d4 : d3;

      if (!isstring(d1) && (d1 < -1))
        {
          if (s_Fig[F_PLOT][iArea][P_TYPE][T_XYZ])
            return; // 3-dim plot, no vector can be plotted?

          d1= double(fabs(d1));
          d2= TP_USER;
          d3= "vector";
        }

      if (isstring(d3) && (d3 == "vector") &&
          (sizerc(s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_XYZ][3])
            != sizerc(s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_XYZ][0])
             + sizerc(s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_XYZ][0])))
        {
          mXY= s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_XYZ][0]|
               s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_XYZ][1];
          mXY= diff0(mXY', 1)[1:][]'~0;
          s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_XYZ][3]= mXY;
        }
//    else
//      println ("Not overwriting existing vector data");

      if (isstring(d1) || (d1 > -1))
        s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_TYPE][1]= d1;
      if (d4 > 0)   // Linewidth
        s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_TYPE][2]= double(d4);
      s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_TYPE][0]=
        (d2 == TP_SOLID) ? "lines" :
        (d2 == TP_DOTTED) ? "dots" :
        ((d2 == TP_DASHED) || (d2 == TP_LDASHED)) ? "linespoints" :
        ((d2 == TP_USER) && isstring(d3)) ? d3 :
          s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_TYPE][0]; // default is no change
    }
  else if ((iType == ADJ_INDEX) && iN)
    {
      s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_TYPE][0]=
          (d1 == 0) ? "lines" :
          (d1 == 1) ? "impulses" :
          (d1 == 2) ? "boxes" :
          "lines";
      // d2, the base of the impulses, not implemented
    }
  else if (iType== ADJ_LABEL)
    {
      i= sizeof(s_Fig[F_PLOT][iArea][P_LABEL][0]);
      if (i == 0)
        {
          oxwarning ("No label found to adjust");
          return;
        }

      if (isint(d1))
        {
          iTitle= d1 == 0 ? TEXT_TEXT :
                  d1 == 1 ? TEXT_AREA :
                  d1 == 2 ? TEXT_PANEL :
                  s_Fig[F_PLOT][iArea][P_LABEL][1][i-1][5];
          s_Fig[F_PLOT][iArea][P_LABEL][1][i-1][5]= iTitle;
          if (isstring(d2))
            d2= strifind({"left", "center", "right"}, d2)-1;
          if (d2 > -2)
            s_Fig[F_PLOT][iArea][P_LABEL][1][i-1][6]= double(d2);
#ifdef GNUDRAW_DEBUG
  println ("Changing coordinates to ", iTitle, " with adjustment ", d2);
#endif
        }
      else
        { // Fully deprecated: DrawAdjust(ADJ_LABEL, "x-lab", "y-lab", "z-lab");
          if (isstring(d1))
            DrawText(iArea, d1, -1, -1, -1, -1, TEXT_XLABEL);
          if (isstring(d2))
            // Set ylabel
            DrawText(iArea, d2, -1, -1, -1, -1, TEXT_YLABEL);
          if (isstring(d3))
            // Set zlabel
            DrawText(iArea, d3, -1, -1, -1, -1, TEXT_ZLABEL);
        }
    }
  else if ((iType== ADJ_MINMAX) && iN)
    {
      dMinY= isnan(d1) ? M_INF_NEG : double(d1);
      dMaxY= isnan(d2) ? M_INF : double(d2);
      for (i= 0; i < s_Fig[F_PLOT][iArea][1]; ++i)
        {
          dMinY= min(dMinY,
                     s_Fig[F_PLOT][iArea][P_LINE][i][L_XYZ][1]);
          dMaxY= max(dMaxY,
                     s_Fig[F_PLOT][iArea][P_LINE][i][L_XYZ][1]);
        }
      sExtra= isdotinf(dMinY) ? "set yrange [*:" :
              sprint("set yrange [", "%g", dMinY, ":");
      sExtra~= isdotinf(dMaxY) ? "*]" :
               sprint("%g", dMaxY, "]");
      s_Fig[F_PLOT][iArea][P_SET][S_YRANGE]=
        {sExtra, "set yrange [*:*]"};
    }
  else if (iType == ADJ_SHOW)
    { // Use SetDraw instead
      SetDraw(SET_OUTPUT, d1);
    }
  else if ((iType == ADJ_SYMBOL) && iN)
    {
      d1= max(min(d1, PL_DOT), 0);
      if (sizeof(s_Fig[F_PLOT][iArea][P_LINE][iN-1]) < L_LAST)
        {
          oxwarning("Internal GnuDraw error: Line is missing...");
          return 0;
        }
      d2= isdotmissing(d2)
            ? s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_POINT][1]
            : d2/90;

      // In case symbol color is being changed, keep sign
      d3= s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_POINT][0] >= 0 ? d1 : -d1;
      s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_POINT]= d3~d2;
    }
  else if (iType == ADJ_AXISSCALE)
    {
      if (ismissing(d4))  // Use either last axis, or by default x-axis
        iAxis= s_Fig[F_PLOT][iArea][P_AXIS];
      else
        iAxis= d4 == 1 ? 0 : d4 == 0 ? 1 : d4;
      iAxis= min(max(iAxis, 0), 4);      // "x", "y", "z", "x2", "y2", "z2"

      d2= ismissing(d2) ? 1 : d2;
      d3= ismissing(d3) ? 0 : d3;

      vType= s_Fig[F_PLOT][iArea][P_TYPE];
      s_SetType(&vType, d1, iAxis);
      s_Fig[F_PLOT][iArea][P_TYPE]= vType;
      s_Fig[F_PLOT][iArea][P_AXISSCALE][iAxis][:2]= d1~d2~d3;

//       println ("Adjusting axis scale for axis ", iAxis, " with result",
//           s_Fig[F_PLOT][iArea][P_AXISSCALE]);
    }
//   else if ((iType == ADJ_AXISSCALE) && !iN)
//     {
//       // Indicate that date format is used
//       s_Fig[F_PLOT][iArea][P_TYPE][T_DMY]= TRUE;
//     }
  else if (iType == ADJ_AREA_3D)
    {
      vView= d2~d3~d4~d5;
      s_Fig[F_PLOT][iArea][P_3DVIEW]=
        isdotmissing(vView) .? s_Fig[F_PLOT][iArea][P_3DVIEW] .: vView;
    }
  else if (iType == ADJ_CNTRPARAM)
    {
      if ((sizeof(ava) == 2) || (sizeof(ava) == 3))
        // Start, inc, end
        s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_TYPE][3]= {d1, d2, d3};
      else if (sizeof(ava) == 1)
        s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_TYPE][3]= vecr(d1);
    }
  else if (iType == ADJ_WARN)
    if (d1)
      s_Fig[F_GLOB][G_WARN]+= !binand(s_Fig[F_GLOB][G_WARN], 1);
    else
      s_Fig[F_GLOB][G_WARN]-= binand(s_Fig[F_GLOB][G_WARN], 1);
  else if (iType == ADJ_AXISFORMAT)
    {
      sExtra= (d2 == 0) ? "y" :
              (d2 == 1) ? "x" : "z";
      // What contents could the format-setting have before?
      //   This seems to work out correctly...
      if (sizeof(s_Fig[F_PLOT][iArea][P_SET][S_FORMAT][0]) > 0)
        s_Fig[F_PLOT][iArea][P_SET][S_FORMAT]=
          {sprint(s_Fig[F_PLOT][iArea][P_SET][S_FORMAT][0], "\n",
                  "set format ", sExtra, " '", d1, "'"),
           sprint(s_Fig[F_PLOT][iArea][P_SET][S_FORMAT][1], "\n",
                  "set format ", sExtra, " '%g'")};
      else
        s_Fig[F_PLOT][iArea][P_SET][S_FORMAT]=
          {sprint("set format ", sExtra, " '", d1, "'"),
           sprint("set format ", sExtra, " '%g'")};
    }
  else if (iType == ADJ_AXISGRID)
    {
      iAxis= s_Fig[F_PLOT][iArea][P_AXIS];
      // Set grid colour type (not used yet, add function SetGrid)
      s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][7][iAxis]= d1 ?
        (ismissing(d2) ? -1 : d2 ) : M_NAN;
      // Set grid line type (not used yet, add function SetGrid)
      s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][8][iAxis]= d1 ?
        (ismissing(d3) ? -1 : d3 ) : M_NAN;

      // Once SetGrid is implemented, code below should be deleted
      if (!d1)
        s_Fig[F_PLOT][iArea][P_SET][S_GRID]= {"unset grid"};
      else
        {
          sExtra= "set grid ";
// 14/5/22: Don't depend on the axis, as one can't get a full xy grid
//   easily now.
//           if ((iAxis >= 0) && (iAxis <= 4))
//             sExtra= sprint(sExtra, {"xtics", "ytics", "ztics", "mxtics", "mytics"}[iAxis]);

          if (d2 > -1)
            sExtra= sprint(sExtra, " lt ", d2);
          // line type not implemented

          s_Fig[F_PLOT][iArea][P_SET][S_GRID]=
            {sExtra, "unset grid"};
        }
    }
  else if (iType == ADJ_FILLCURVE)
    {
      sExtra= "";
      // 1: x1, 0: y1, 2: z, 3:x2, 4:y2
      if ((d1 < 0) || (d1 > 4) || (d1 == 2))
        return;
      if (!ismissing(d1))
        sExtra= {"y1", "x1", "z", "x2", "y2"}[d1];
      if (!ismissing(d2))
        sExtra~= sprint("=", double(d2));
      d2= ismissing(d2) ? 0 : d2;
      DrawAdjust(ADJ_COLOR, -1, TP_USER, sprint("filledcurves ", sExtra,
        " fs solid ", "%3.1f", s_Fig[F_GLOB][G_DEFFILL]));
    }
  else if (iType == ADJ_LEGEND)
    { // ADJ_LEGEND: area, no columns, font size, resize, box
      if (sizeof(ava) < 5) // Area number not given
        {
          d1= iArea;
          d2= (sizeof(ava) > 0) ? ava[0] : M_NAN; // no columns
          d3= (sizeof(ava) > 1) ? ava[1] : M_NAN; // font size
          d4= (sizeof(ava) > 2) ? ava[2] : M_NAN; // resize
          d5= (sizeof(ava) > 3) ? ava[3] : M_NAN; // box
        }

      if (d1 < s_Fig[F_NPLOTS])
        {
          if (!ismissing(d5))
            s_Fig[F_PLOT][d1][P_LEGEND][5]= d5;
          if (!ismissing(d2) && (d2 != -1))
            s_Fig[F_PLOT][d1][P_LEGEND][7]= d2;
          if (!ismissing(d3) && (d3 != -1))
            s_Fig[F_PLOT][d1][P_LEGEND][8]= d3;
        }
    }
  else if (iType == ADJ_PAPERSCALE)
    {
      if (d1 > 1)
        d1/= 100;
      if (d2 > 1)
        d2/= 100;
      // scale either only y, or x and y
      s_Fig[F_SIZES][:1]= ismissing(d2) ? 1~d1 : d1~d2;
    }
  else if (iType == ADJ_PAPERCOLOR)
    {
      s_Fig[F_PLOT][0][P_PAPERCOLOR]= isstring(d1) ? d1 : d1~d2~d3;
    }
  else if (iType == ADJ_COLORMODEL)
    {
      // Set colormodel for FIRST area; only first counts
      d1= d1 != 0;
      d2= !ismissing(d2) ? max(min(d2, 3), 0) : d2;
      s_Fig[F_PLOT][0][P_COLORMODEL]= d1 ~ d2;
    }
  else if (iType == ADJ_ENCODING)
    // Set encoding for FIRST area; only first counts
    s_Fig[F_PLOT][0][P_SET][S_ENCODING]=
      {sprint("set encoding ", d1),
       ""};
  else if (iType == ADJ_DEBUG)
    s_Debug= d1;
  else
    if (iN && binand(s_Fig[F_GLOB][G_WARN], 1))
      println("DrawAdjust: This option is not implemented yet...");
}

/*
**  DrawAxis, DrawAxisAuto
**
**  #include <packages/gnudraw/gnudraw.h>
**  DrawAxis(const iArea, const iIsXaxis, const dAnchor,
**      const dAxmin, const dAxmax, const dFirstLarge,
**      const dLargeStep, const dSmallStep, const iFreq);
**  DrawAxisAuto(const iArea, const iIsXaxis, ...);
**  DrawAxisAuto(const iArea, const iIsXaxis, bool fShow,
**      int iAnchor, double dAnchor);
**
**      iArea
**          in: int, area index
**      iIsXaxis
**          in: int, TRUE: is X axis, else Y axis
**      dAnchor
**          in: int, anchor of the axis (e.g. Y location of X axis)
**      dAxmin
**          in: axis minimum
**      dAxmax
**          in: axis maximum
**      dFirstLarge
**          in: location of first large tick
**      dLargeStep
**          in: step size between large ticks
**      dSmallStep
**          in: step size between small ticks
**      iFreq
**          in: int, frequency (for time series axis)
**      fShow
**          in: int, TRUE: show the axis
**      iAnchor
**          in: int, axis anchor location, ANCHOR_MIN: at minimum, ANCHOR_MAX: at
**          maximum, ANCHOR_USER: at dAnchor
**
**  No return value.
*/
DrawAxis(const iArea, const iIsXaxis, const dAnchor,
         const dAxmin, const dAxmax, const dFirstLarge,
         const dLargeStep, const dSmallStep, ...)
{
  decl ava, iFreq, vOff, iOff, iAxis;

  ava= va_arglist();
  iFreq= sizeof(ava) ? int(ava[0]) : 1;

  // Order in vOff: x, y, z, x2, y2, from order in call y, x, z, x2, y2
  vOff= <1, 0, 2, 3, 4>;
  if ((iIsXaxis < 0) || (iIsXaxis > 4))
    {
      if (binand(s_Fig[F_GLOB][G_WARN], 1))
        println ("Warning: Incorrect value ", iIsXaxis, " of iIsXaxis in DrawAxis");
      return;
    }
  iOff= vOff[iIsXaxis];

  // Make sure there is a plot
  if (iArea < 0)
    return;
  while ((sizeof(s_Fig) < F_LAST) || (s_Fig[F_NPLOTS] <= iArea))
    s_InitPlot();

  // Remember last axis used
  iAxis= iIsXaxis == 1 ? 0 : iIsXaxis == 0 ? 1 : iIsXaxis;
  s_Fig[F_PLOT][iArea][P_AXIS]= iAxis;

  // Use zeroaxis
  s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][0][iOff]= !isnan(dAnchor);
  // Tics on zeroaxis
  s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][2][iOff]= (dAnchor == 0);

  // Start, increment and end
  s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][3][iOff]= dFirstLarge;
  s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][4][iOff]= dLargeStep/iFreq;
  s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][5][iOff]= dAxmax;

  // Number of intervals for minor tics
  s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][6][iOff]=
    dSmallStep > 0 ? floor(dLargeStep/dSmallStep) : floor(dLargeStep);

//   // Adapt time series data (moved to s_SetAxis)
//   if (s_Fig[F_PLOT][iArea][P_TYPE][T_DMY])
//     {
//       s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][4][iOff]*= G_GD_DAY;
//       if (dFirstLarge < 3000)
//         s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][3][iOff]=
//           dayofcalendar(floor(dFirstLarge), floor(fmod(dFirstLarge, 1)*12)+1, 1);
//     }
}

DrawAxisAuto(const iArea, const iIsXaxis, ...)
{
  decl ava, fShow, iAnchor, iBorder, dAnchor, dAnchor2, iOff, vOff,
    i, i1, i2, iN;

  s_InitFigure();
  if (iArea < 0)
    return;

  ava= va_arglist();
  fShow= (sizeof(ava) > 0) ? int(ava[0]) : TRUE;
  iAnchor= (sizeof(ava) > 1) && !ismissing(ava[1]) ? int(ava[1]) : M_NAN;
  dAnchor= (sizeof(ava) > 2) ? double(ava[2]) : M_NAN;
  dAnchor2= (sizeof(ava) > 3) ? double(ava[3]) : 0;
  if (binand(s_Fig[F_GLOB][G_WARN], 1) && (dAnchor != 0))
    println("Warning: dAnchor indicates if zeroaxis is shown, not the location");

  // Order in vOff: x, y, z, x2, y2, from order in call y, x, z, x2, y2
  vOff= <1, 0, 2, 3, 4>;
  if ((iIsXaxis < 0) || (iIsXaxis > 4))
    {
      if (binand(s_Fig[F_GLOB][G_WARN], 1))
        println ("Warning: Incorrect value ", iIsXaxis, " of iIsXaxis in DrawAxis");
      return;
    }
  iOff= vOff[iIsXaxis];


  // Make sure there is a plot
  while ((sizeof(s_Fig) < F_LAST) || (s_Fig[F_NPLOTS] <= iArea))
    s_InitPlot();

  // Remember last axis used
  s_Fig[F_PLOT][iArea][P_AXIS]= iOff;

  // Use zeroaxis?
  i1= !isnan(dAnchor) * ((dAnchor == 0) + 2*(dAnchor != 0));
  s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][0][iOff]= fShow * i1;
  // Tics on zeroaxis?
  s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][2][iOff]= (iAnchor == ANCHOR_USER) && (dAnchor == 0) && fShow;

  if (ismissing(iAnchor) || (iOff == 2))        // for z-axis
    // Show the respective border
    s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][1][iOff]= fShow;
  else
    {
// 12/5/2010: Intention of iAnchor= ANCHOR_MAX: for
//   all lines, the left/bottom axis is replaced by the right/top axis
      i1= imod(iOff, 3);                // Check x or y
      i2= (iAnchor == ANCHOR_MAX);      // Check second axis

      // Set either lower or upper border, turn off other border
      s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][1][i1]= (iAnchor == ANCHOR_MIN) && fShow;
      s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][1][i1+3]= (iAnchor == ANCHOR_MAX) && fShow;

      // Go over all lines, set them to correct axis...
      iN= s_Fig[F_PLOT][iArea][P_NLINES];
      for (i= 0; i < iN; ++i)
        s_Fig[F_PLOT][iArea][P_LINE][i][L_AXES][i1]= i2;
    }
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawBoxPlot(const iArea, const mY, const sY);
**  DrawBoxPlot(const iArea, const mY, const sY, const iIndex);
**
**      iArea
**          in: int, area index
**      mY
**          in: k x T matrix, each row is a new plot (needs T > 5)
**      asY
**          in: string, variable name, or array of strings (k > 1)
**      iIndex
**          in: int, line index, see default line attributes, (optional argument,
**              default is 2).
**
**  No return value.
*/
DrawBoxPlot(const iArea, const mY, const asY, ...)
{
  decl ir, sKey, vQ, i, iK, vY, nY, dIQR, vBX, vBY, vSY, iIndex,
  ava, iAreaL;

  s_InitFigure();
  if (iArea < 0)
    return;

  // These lines may be removed eventually
  while ((sizeof(s_Fig) < F_LAST) || (s_Fig[F_NPLOTS] <= iArea))
    s_InitPlot();

  iK= rows(mY);

  ava= va_arglist();
  iIndex= (sizeof(ava) > 0) ? vecr(ava[0]) : s_Fig[F_GLOB][G_DEFLINE][0];

  ir= s_CheckSize(mY, 0, 0, 0, "DrawBoxPlot", &nY);
  iAreaL= iArea;
  for (i= 0; i < iK; ++i)
    {
      vY= deletec(mY[i][]);
      nY= columns(vY);
      sKey= (i == 0) && isstring(asY) ? asY :
            (i < sizeof(asY)) && isarray(asY) ? asY[i] : "";
      if (nY > 5)
        {
          vQ= quantiler(vY, <.25, .5, .75>);
          dIQR= (vQ[2]-vQ[0]);

          // Draw box, Q1-Q3
          vBX= <-1, -1, 1, 1, -1>*dIQR;
          vBY= vQ[<0, 2, 2, 0, 0>];
          DrawXMatrix(iAreaL, vBY, "", vBX, "", 0, iIndex+1);
          if (sizeof(sKey))
            DrawTitle(iAreaL, sKey);

          // Draw median
          vBX= <-1, 1>*dIQR;
          vBY= vQ[<1, 1>];
          DrawXMatrix(iAreaL, vBY, "", vBX, "", 0, iIndex);

          // Draw whiskers
          vBY= (vQ[0]-1.5*dIQR)~vQ[0]~vQ[2]~(vQ[2]+1.5*dIQR);
          vBX= <0, 0>;
          DrawXMatrix(iAreaL, vBY[:1], "", vBX, "", 0, iIndex+1);
          DrawXMatrix(iAreaL, vBY[2:], "", vBX, "", 0, iIndex+1);

          vBY= (vQ[0]-1.5*dIQR)~(vQ[0]-1.5*dIQR)~
               (vQ[2]+1.5*dIQR)~(vQ[2]+1.5*dIQR);
          vBX= .2*<-1, 1>*dIQR;
          DrawXMatrix(iAreaL, vBY[:1], "", vBX, "", 0, iIndex+1);
          DrawXMatrix(iAreaL, vBY[2:], "", vBX, "", 0, iIndex+1);

          // Plot points outside whisker Q1-1.5*IQR, plus one point inside
          vSY= sortr(vY);
          vBX= limits(vSY' .> vQ[0]-1.5*dIQR)[2][];
          vBY= vSY[:vBX];
          DrawX(iAreaL, vBY, zeros(vBY));
          DrawAdjust(ADJ_COLOR, iIndex, -1);

          // Plot points outside whisker Q3+1.5*IQR, plus one point inside
          vBX= limits((vSY'|M_INF) .> vQ[2]+1.5*dIQR)[3][];
          vBY= vSY[vBX-1:];
          DrawX(iAreaL, vBY, zeros(vBY));
          DrawAdjust(ADJ_COLOR, iIndex, -1);

          // Get the graph to be wide enough
          DrawAdjust(ADJ_AREA_X, iAreaL, -1.3*dIQR, 1.3*dIQR);

          // Add x-zeroaxis
          s_Fig[F_PLOT][s_Fig[F_GLOB][G_LASTPLOT]][P_TICSAXISBORDER][0][0]=
            TRUE;

          // Show only y-border
          s_Fig[F_PLOT][s_Fig[F_GLOB][G_LASTPLOT]][P_TICSAXISBORDER][1][]=
            <FALSE, TRUE, FALSE, FALSE, FALSE>;

          ++iAreaL;
        }
    }
}

/*
**  DrawCdf(const iArea, const mX, const asX, ...)
**  DrawCdf(const iArea, const mX, const asX, const vP0, const vQ)
**
**  Purpose:
**    Draw extensive Cdf, indicating quantiles and true parameter
**    vector
**
**  Inputs:
**    iArea
**      in: int, first area to plot
**    mX
**      in: iP x iS matrix of parameter sample
**    asX
**      in: array of strings, names of parameters
**    vP0
**      in: iP x 1 vector of true parameter values (optional argument)
**    vQ
**      in: vector of length 3 with quantiles to
**      indicate (optional argument, default= <.05, .5, .95>)
**    iIndex
**      in: int, line index, see default line attributes, (optional
**      argument, default is 2).
**
**  Return value:
**    {mXX, mYY}    array of size two, with x and y values of CDFs
*/
DrawCdf(const iArea, const mX, const asX, ...)
{
  decl iAreaL, asXL, ava, vP0, vQ, iP, i, j, vX, vY, vXQ, iK, iIndex,
  vJ, mXX, mYY;

  if (iArea < 0)
    return;
  iAreaL= iArea;
  ava= va_arglist();

  iP= rows(mX);
  vP0= sizeof(ava) ? vecr(ava[0]) : constant(M_NAN, iP, 1);
  vQ= sizeof(ava) > 1 ? vecr(ava[1])' : <.05, .5, .95>;
  iIndex= sizeof(ava) > 2 ? int(ava[2]) : 2;

  asXL= array(asX);
  while (sizeof(asXL) < iP)
    asXL~= {""};
  while (sizerc(vP0) < iP)
    vP0|= M_NAN;
  if (sizerc(vQ) < 3)
    vQ= <.05, .5, .95>;

  mXX= mYY= <>;
  for (i= 0; i < iP; ++i)
    {
      [vX, vY]= DrawDensity(iAreaL, mX[i][], "", 0, 0, 0, 1, 0, 0, iIndex);
      mXX|= vX;
      mYY|= vY;

      // Find observations closest to quantiles on CDF
      vXQ= quantiler(mX[i][], vQ);
      vJ= mincindex(fabs(vX' - vXQ));

      // Get .05-.95 box
      DrawXMatrix(iAreaL, vY[vJ[0~0~2~2~0]], "",
                  vX[vJ[0~2~2~0~0]], "", 0, iIndex+1);
      // Get .5-.5 cross
      DrawXMatrix(iAreaL, vY[vJ[1~1]], "", vX[vJ[0~2]], "", 0, iIndex+1);
      DrawXMatrix(iAreaL, vY[vJ[0~2]], "", vX[vJ[1~1]], "", 0, iIndex+1);

      // Get cross at true parameters
      if (!isdotmissing(vP0[i]))
        {
          DrawXMatrix(iAreaL, 0~1, "", vecr(vP0[i~i])', "", 0, 0);
          j= mincindex(fabs(vX - vP0[i])');
          iK= sizerc(vX);
          DrawXMatrix(iAreaL, vY[j~j], "", vX[0~(iK-1)], "", 0, 0);
        }
      DrawText(iAreaL, asXL[i], .05, .95);
      DrawAdjust(ADJ_LABEL, 1, -1);
      DrawAdjust(ADJ_AREA_X, vX[0], vX[iK-1]);
      ++iAreaL;
    }
  return {mXX, mYY};
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawCorrelogram(const iArea, const vY, const asY,
**                  const cLag, ...);
**
**      iArea
**          in: int, area index
**      vY
**          in: k x T matrix, each row is a new plot
**      sY
**          in: string, variable name, or array of strings (k > 1)
**      cLag
**          in: int, highest lag to be used in correlogram
**      fJoin
**          in: boolean, optional, if TRUE the correlograms are
**              drawn jointly in one area
**
**  No return value.
**
**  Draws a correlogram (the ACF is computed using acf).
*/
DrawCorrelogram(const iArea, const mY, const asY, const cLag, ...)
{
  decl i, sKey, mACF, ir, vX, ilArea, iIndex, ava, fJoin, nDim, dD,
       dMY, iN, dOffset;

  s_InitFigure();
  if (iArea < 0)
    return;

  ava= va_arglist();
  fJoin= sizeof(ava) ? int(ava[0]) : s_Fig[F_GLOB][G_JOIN];

  ilArea= iArea;
  iIndex= s_Fig[F_GLOB][G_DEFLINE][0];
  nDim= rows(mY);
  dD= 1/(nDim+2);
  dOffset= (nDim-1)/2;
  dMY= 0;

  vX= range(1, cLag);
  mACF= acf(mY', cLag)';
  ir= s_CheckSize(mACF, 0, 0, 0, "DrawCorrelogram", &iN);

  sKey= "";
  if (isstring(asY))
    sKey= asY;
  for (i= 0; i < nDim; ++i)
    {
      if (isarray(asY) && (i < sizeof(asY)))
        sKey= asY[i];
      ir= s_AppendLine(ilArea, vX+fJoin*dOffset, mACF[i][1:], <>, <>,
        "impulses", iIndex, M_NAN, M_NAN, sKey);
      dOffset+= dD;

      if (ir)
        {
          sKey= "";

          dMY= min(fJoin ? dMY : 0, mACF[i][1:]);
          s_Fig[F_GLOB][G_LASTPLOT]= ilArea;
          DrawAdjust(ADJ_AREA_Y, dMY, 1);
          DrawAdjust(ADJ_AREA_X, dD, cLag+fJoin*dOffset);
          s_Fig[F_PLOT][ilArea][P_TICSAXISBORDER][0][0]= TRUE;
          DrawLegend(ilArea, 1, 0, FALSE);

          ilArea+= 1-fJoin;
          iIndex+= fJoin;
        }
    }
}

/*
**  DrawDensity
**
**  #include <packages/gnudraw/gnudraw.h>
**  DrawDensity(const iArea, const mY, const asY, ...)
**  DrawDensity(const iArea, const mY, const asY, const fDens,
**              const fHist, const fNormal, ...)
**  DrawDensity(const iArea, const mY, const sY, const fDens,
**              const fHist, const fNormal, const fCdf, const fStand,
**              const cBar, const iIndex, const vWeight);
**
**  iArea
**      in: int, area index
**  mY
**      in: k x T matrix, each row is a new plot
**  asY
**      in: string, variable name, or array of strings (k > 1)
**  fDens
**      in: int, TRUE: draw estimated density
**  fHist
**      in: int, TRUE: draw histogram
**  fNormal
**      in: int, TRUE: add normal density for reference
**  fCdf
**      in: int, TRUE: plot CDF in separate area (optional argument,
**      not drawn by default)
**  fStand
**      in: int, TRUE: use standardized data (optional argument,
**      default uses original data)
**  cBar
**      in: int, number of bars (0: use default; optional argument)
**  iIndex
**      in: int, line index, see default line attributes, (optional
**      argument, default is 2).
**  vWeight
**      in: 1 x T matrix with weights (optional argument, default
**      is 1).
**  fJoin
**      in: int, TRUE: draw densities together in one plot, else draw separately
**      (optional argument, drawing separate plots by default)
**
**  Return value
**      vX  out: 1 x K matrix with x-values of last kernel density estimate
**      mY  out: 1 x K matrix with last kernel density estimate
**
**  Draws the histogram and/or density of the data in the specified
**  area. When fNormal is TRUE, a normal density with the same mean and
**  variance as the data will be drawn.
*/
DrawDensity(const iArea, const mY, const asY, ...)
{
  decl ir, fDens, fHist, fNormal, fCdf, fStand, fJoin, cBar, cBarL, iIndex,
       i, iN, iN1, ilIndex, vX, vXK, vlY, vYK, vW, sKey, dM, dS, mRet,
       dMinX, dMaxX, dStep, dHStep, ilArea, ava, dD, dOffset, mXK, mYK;

  s_InitFigure();
  if (iArea < 0)
    return {<>, <>};

  ir= 1;
  ilArea= iArea;
  sKey= "";
  if (isstring(asY))
    sKey= asY;
  ava= va_arglist();

  fDens= (sizeof(ava) > 0) ? int(ava[0]) : TRUE;
  fHist= (sizeof(ava) > 1) ? int(ava[1]) : FALSE;
  fNormal= (sizeof(ava) > 2) ? int(ava[2]) : FALSE;
  fCdf= (sizeof(ava) > 3) ? int(ava[3]) : FALSE;
  fStand= (sizeof(ava) > 4) ? int(ava[4]) : FALSE;
  cBar= (sizeof(ava) > 5) ? int(ava[5]) : 0;
  iIndex= ilIndex= (sizeof(ava) > 6) ? vecr(ava[6]) : s_Fig[F_GLOB][G_DEFLINE][0];
  vW= (sizeof(ava) > 7) ? vecr(ava[7])' : 1;
  fJoin= (sizeof(ava) > 8) ? int(ava[8]) : s_Fig[F_GLOB][G_JOIN];

  vXK= vYK= M_NAN;
  mXK= mYK= <>;

  iN= rows(mY);
  dD= fJoin && (iN > 1) ? 1/(iN+0.5) : 0;
  dOffset= dD ? -0.5*(iN-1) : 0;

  // Adapt for accidental weight 0
  if (vW == 0)
    vW= 1;

  if ((fCdf < 0) || (fCdf > 1) || (sizerc(fStand) > 1))
    {
      if (binand(s_Fig[F_GLOB][G_WARN], 1))
        println ("Warning: Definition of DrawDensity has changed.\n",
                 "         Please adapt program");
      cBar= fCdf;
      vW= fStand;
      fCdf= fStand= FALSE;
      iIndex= s_Fig[F_GLOB][G_DEFLINE][0];
    }

  mXK= mYK= constant(M_NAN, iN*(fDens+fCdf+fHist), 128);
  for (i= 0; i < iN; ++i)
    {
      if (isarray(asY) && (i < sizeof(asY)))
        sKey= asY[i];
      if (!fJoin)
        ilIndex= iIndex;

      vlY= deletec(mY[i][]);
      if (fStand) // Standardize...
        vlY= (vlY-meanr(vlY))/sqrt(varr(vlY));
      if (isnan(cBar) || (cBar < 3))
        cBar= min(max(5, columns(vlY)/20), 20);

      dMinX= min(vlY);
      dMaxX= max(vlY);
      dStep= (dMaxX - dMinX)/(cBar-1);
      vX= vXK= range(0, cBar-1)*dStep + dMinX;

      if ((sizeof(vlY) > 0) && !isfeq(varr(vlY), 0))
        {
          if (fDens || fCdf)
            {
              if (sizerc(vW) == 1)
                { // Use original frequency-domain kernel without weights
                  mRet= DensEst(vlY', 0, 0, 0, 128);
                  vXK= mRet[0][];
                  vYK= mRet[2][];
                }
              else  // Use Gaussian kernel with weights
                {
                  vXK= vX;
                  ir= s_KernelUniIS(vXK, vlY, vW, &vYK);
                }
              if (fDens)
                {
                  ir= s_CheckSize(vXK, vYK, 0, 1, "DrawDensity-Dens", &iN1) &&
                      s_AppendLine(ilArea, vXK, vYK, <>, <>, "lines",
                        ilIndex, M_NAN, M_NAN, sKey);
                  mXK[i*(fDens+fCdf+fHist)][]= vXK;
                  mYK[i*(fDens+fCdf+fHist)][]= vYK;
                }
              if (fCdf)
                {
                  vYK= cumulate(vYK')';
                  vYK/= vYK[0][columns(vYK)-1];
                  ir= s_CheckSize(vXK, vYK, 0, 1, "DrawDensity-CDF", &iN1) &&
                      s_AppendLine(ilArea+(fDens||fHist||fNormal), vXK, vYK, <>, <>, "lines",
                        ilIndex, M_NAN, M_NAN, sKey);
                  mXK[i*(fDens+fCdf+fHist)+fDens][]= vXK;
                  mYK[i*(fDens+fCdf+fHist)+fDens][]= vYK;
                }
              ++ilIndex;
              sKey= "";
            }

          if (fHist)
            {
              if (vW == 1)
                {
                  vYK = discretize(vlY, dMinX, dMaxX, cBar, 0);
                  vYK /= (columns(vlY) * dStep);
                }
              else
                {
                  vYK= zeros(1, cBar);
                  dHStep= dStep/2;
                  for (i= 0; i < cBar; ++i)
                    vYK[i]= sumr(vW.*((vX[i]-dHStep .<= vlY) .&&
                                      (vlY .< vX[i]+dHStep)));
                  vYK /= (sumr(vW .* constant(1, vlY)) * dStep);
                }
              vXK= vX+dOffset*dStep*dD;
//              print (vXK|vYK, dMinX~dMaxX~cBar);
              ir= s_CheckSize(vX, vYK, 0, 1, "DrawDensity-Hist", &iN1) &&
                  s_AppendLine(ilArea, vXK, vYK, <>, <>,
                    fHist == 1 ? "boxes" : fHist == 2 ? "lines" : "index",
                    ilIndex, M_NAN, M_NAN, sKey);
              if ((sizerc(mXK) == 0) || (columns(mXK) == columns(vXK)))
                {
                  mXK[i*(fDens+fCdf+fHist)+fDens+fCdf][]= vXK;
                  mYK[i*(fDens+fCdf+fHist)+fDens+fCdf][]= vYK;
                }

              ++ilIndex;
              sKey= "";
            }

          if (fNormal)
            {
              cBarL= max(cBar, 128);
              dStep= (dMaxX - dMinX)/(cBarL-1);
              vX= range(0, cBarL-1)*dStep + dMinX;

              dM= meanisr(vlY, vW);
              dS= sqrt(varisr(vlY, vW));
              vYK= densn((vX - dM)/dS)/dS;
              if (sKey=="")
                sKey= sprint("N(s=", "%.3g", double(dS), ")");
              ir= s_CheckSize(vX, vYK, 0, 1, "DrawDensity-Norm", &iN1) &&
                  s_AppendLine(ilArea, vX, vYK, <>, <>, "lines",
                    ilIndex, M_NAN, M_NAN, sKey);

              ++ilIndex;
              sKey= "";
            }
        }
      if (!fJoin)
        ilArea+= (fDens||fHist||fNormal)+fCdf;
      else
        ++dOffset;
    }

  // Note last check: Robustness that should not be needed...
  if (fJoin && fHist && dD && ir && (sizeof(s_Fig[F_PLOT]) > ilArea))
    s_Fig[F_PLOT][ilArea][P_SET][S_BOXWIDTH]=
      {sprint("set boxwidth ", dD*dStep), "set boxwidth"};

  s_Fig[F_GLOB][G_LASTPLOT]= ilArea-!fJoin;

  return {mXK, mYK};
}

/*
**  DrawBivDensity
**
**  #include <packages/gnudraw/gnudraw.h>
**  DrawBivDensity(const iArea, const mY, const asY, const fDens,
**                 const fHist, const fNormal);
**  DrawBivDensity(const iArea, const mY, const asY, const fDens,
**                 const fHist, const fNormal, const iMode);
**  DrawBivDensity(const iArea, const mY, const asY, const fDens,
**                 const fHist, const fNormal, const iMode, const vW);
**
**      iArea
**          in: int, area index
**      vY
**          in: k x T matrix, each row is a new plot
**      asY
**          in: string, variable name, or array of strings (k > 1)
**      fDens
**          in: int, TRUE: draw estimated density
**      fHist
**          in: int, TRUE: draw histogram
**      fNormal
**          in: int, TRUE: add normal density for reference
**      iMode (optional)
**          in: int, type of plot:
**          0 - surface plot only
**          1 - surface plot with contours on ground level
**          2 - contour plot
**      vW (optional)
**          in: 1 x T vector, weights to apply in weighted density plot
**
**  Return value:
**      vX
**          out: 1 x nX vector with X-coordinates of last plot
**      vY
**          out: 1 x nY vector with Y-coordinates of last plot
**      mZ
**          out: nY x nX matrix with heights above plain
**
**  Draws the bivariate histogram and/or density of the data in the
**  specified area. When fNormal is TRUE, a normal density with the
**  same mean and variance as the data will be drawn.
**
**  For compatibility with the OxDraw version of  DrawXYZ, the first
**  coordinate is plotted on the Y axis, the second on the X axis.
**
**  For the last plot, the output [vX, vY, mZ] can be used as input for
**  DrawXYZ. This way, it is possible to quickly draw a surface and a contour
**  plot without reestimating the kernel approximation, i.e. using
**    {vX, vY, mZ}= DrawBivDensity(0, mY, asY, TRUE, FALSE, FALSE, 0);
**    DrawXYZ(1, vX, vY, mZ, 2, asY[0], asY[1], "height");
*/
DrawBivDensity(const iArea, const mY, const asY, const fDens,
               const fHist, const fNormal, ...)
{
  decl ir, iK, iX, iY, nDim, i, j, ii, jj, ilArea, mX, mXX, mlY, vlX,
       vlY, vInd, vW, sKeyX, sKeyY, mRet, mRetH, mRetN, vMMX, vStep, dH,
       vHStep, iMode, iContour, iIndex, ava, vM, mS, ddetS, miS, dLogDet, dSignDet;

  s_InitFigure();
  if (iArea < 0)
    return;

  ava= va_arglist();
  iMode= (sizeof(ava) > 0) ? int(ava[0]) : 0;
  vW= (sizeof(ava) > 1) ? vecr(ava[1])' : 1;

  ir= 1;
  ilArea= iArea;
  nDim= rows(mY);
  vlX= vlY= mRet= 0;
  iContour= iMode > 0 ? 10 : 0;
  dH= 0;     // 0= Gaussian, -1= Epanechnikov, -2= squared Epanechnikov

  for (i= 0; i < nDim-1; ++i)
    {
      sKeyX= sKeyY= "";
      if (isarray(asY) && (sizeof(asY) > i) && (isstring(asY[i])))
        sKeyX= asY[i];
      for (j= i+1; j < nDim; ++j)
        {
          if (isarray(asY) && (sizeof(asY) > j))
            sKeyY= asY[j];

          iIndex= s_Fig[F_GLOB][G_DEFLINE][0];

          // Only bivariate plots are catered for
          vInd= vecrindex(!(isdotnan(mY[i][]) .|| isdotnan(mY[j][])));
          iK= min(max(5, floor(sizerc(vInd)/20)), 20);

          vMMX= limits(mY[i|j][vInd]')[:1][]';
          vStep= (vMMX[][1] - vMMX[][0])/(iK-1);
          mX= range(0, iK-1) .* vStep + vMMX[][0];
          mXX= (mX[0][] ** ones(1, iK)) |
               (ones(1, iK) ** mX[1][]);

          if (ir && (sizerc(vInd) > 0) &&
              !isfeq(minc(varr(mY[i|j][vInd])), 0))
            {
              if (fDens)
                {
                  if (sizerc(vW) == 1)
                    {
                      // Use Boswijk's code
                      [mX, mRet]= BivDensEst(mY[i|j][vInd]', 0, 0, dH, iK); // Gaussian
                    }
                  else
                    { // Use special code, taking weights into account
                      ir= s_KernelMultIS(mXX, mY[i|j][vInd], vW, &mRet);
                      mRet= reshape(mRet, iK, iK);
                    }

//                   ir= s_CheckSize(mX[0][], mX[1][], mRet, 2, "DrawDensity-Dens", &iX, &iY) &&
//                       s_AppendLine(ilArea, mX[0][], mX[1][], mRet, <>, "lines", iIndex++, M_NAN, iContour, "");
//                       s_AppendLine(ilArea, mX[1][], mX[0][], mRet', <>,
//                         "lines", iIndex++, M_NAN, iContour, "");
                   // 18/12/13: Change direction of x-y axes? Contour plot seemed to come out incorrectly...
                     // DrawXYZ(ilArea, mX[0][], mX[1][], mRet, iContour, "",
                     //          "", "", 0, iIndex++);
                   DrawXYZ(ilArea, mX[1][], mX[0][], mRet, iContour, "",
                           "", "", 0, iIndex++);
                }
              if (fHist)
                {
                  mRet= zeros(iK, iK);
                  vHStep= vStep/2;
                  for (ii= 0; ii < iK; ++ii)
                    for (jj= 0; jj < iK; ++jj)
                      mRet[ii][jj]=
                        sumr(vW.*((mX[0][ii]-vHStep[0] .<= mY[i][vInd]) .&&
                                  (mY[i][vInd] .< mX[0][ii]+vHStep[0]) .&&
                                  (mX[1][jj]-vHStep[1] .<= mY[j][vInd]) .&&
                                  (mY[j][vInd] .< mX[1][jj]+vHStep[1])));
                  mRet /= (sumr(vW .* constant(1, mY[i][vInd])) * vStep[0] * vStep[1]);
//                   ir= s_CheckSize(mX[0][], mX[1][], mRetH, 2, "DrawDensity-Dens", &iX, &iY) &&
// //                       s_AppendLine(ilArea, mX[0][], mX[1][], mRetH, <>, "lines", 2, M_NAN, "");
//                       s_AppendLine(ilArea, mX[1][], mX[0][], mRetH', <>,
//                         "lines", iIndex++, M_NAN, iContour, "");
                  DrawXYZ(ilArea, mX[1][], mX[0][], mRet, iContour, "",
                          "", "", 0, iIndex++);
                }
              if (fNormal)
                {
                  vM= meanisr(mY[i|j][vInd], vW);
                  mS= choleski(varianceis(mY[i|j][vInd]', vW'));
                  miS=invert(mS, &dLogDet, &dSignDet);
                  ddetS=exp(dLogDet)*dSignDet;
                  mRet= prodc(densn(miS*(mXX - vM)))/ddetS;
                  mRet= shape(mRet, iK, iK)';

//                   ir= s_CheckSize(mX[0][], mX[1][], mRetN, 2, "DrawDensity-Dens", &iX, &iY) &&
// //                       s_AppendLine(ilArea, mX[0][], mX[1][], mRetN, <>, "lines", 2, M_NAN, "");
//                       s_AppendLine(ilArea, mX[1][], mX[0][], mRetN', <>,
//                         "lines", iIndex++, M_NAN, iContour, "");
                  DrawXYZ(ilArea, mX[1][], mX[0][], mRet, iContour, "",
                          "", "", 0, iIndex++);
                }

              // Set the labels
              if (sizeof(sKeyX))
                DrawText(ilArea, sKeyX, 0, 0, -1, -1, TEXT_XLABEL);
              if (sizeof(sKeyY))
                DrawText(ilArea, sKeyY, 0, 0, -1, -1, TEXT_YLABEL);
            }

          ++ilArea;
        }
    }
  s_Fig[F_GLOB][G_LASTPLOT]= ilArea-1;

  return {mX[0][], mX[1][], mRet};
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawHistogram(const iArea, const vBar, ...);
**  DrawHistogram(const iArea, const vBar, const dMin, const dStep,
**                const iIndex, const iColorIn);
**  DrawHistogram(const iArea, const vBar, const asNames, const dMin, const dStep,
**                const iIndex, const iColorIn);
**
**  iArea
**      in:   int, area index
**  vBar
**      in:   k x T matrix with bar heights, each row is a new plot
**  asY
**      in:   (optional) string, variable name, or array of strings (k > 1)
**  dMin
**      in:   double, first X-coordinate of histogram (optional argument, default is 0)
**  dStep
**      in:   double, bar step size (optional argument, default is 1)
**  iIndex
**      in:   int, line index, see <a href="#TabDraw1">default line attributes</a>,
**            (optional argument, default is 2).
**  iColorIn
**      in:   int, colour index for inside of bars, see default line attributes,
**            (optional argument, default is 0: white).
**
**  No return value.
**
**  Draws a histogram when the data is already in histogram
**  format, i.e. vBar contains the bar heights.
*/
DrawHistogram(const iArea, const vBar, ...)
{
  decl ir, iIndex, iColorIn, dMin, dStep, i, vX, asY, sY,
       cBar, ilArea, ava, iN, iLine;

  s_InitFigure();
  if (iArea < 0)
    return;

  ir= 1;
  ilArea= iArea;
  ava= va_arglist();

  i= 0;
  asY= (sizeof(ava) > i) && (isarray(ava[i]) || isstring(ava[i]))
    ? array(ava[i++]) : {};
  dMin= (sizeof(ava) > i) && !isarray(ava[i]) ? double(ava[i]) : 1;
  ++i;
  dStep= (sizeof(ava) > i) && !isarray(ava[i]) ? double(ava[i]) : 1;
  ++i;
  iIndex= (sizeof(ava) > i) ? (isstring(ava[i]) ? ava[i] : vecr(ava[i])[0]) : s_Fig[F_GLOB][G_DEFLINE][0];
  ++i;
  iColorIn= (sizeof(ava) > i) ? ava[i] : 0;

  cBar= columns(vBar);
  vX= range(0, cBar-1)*dStep + dMin + 0.5*dStep;
  for (i= 0; i < rows(vBar); ++i)
    {
      ir= s_CheckSize(vX, vBar[i][], 0, 1, "DrawHistogram", &iN);

      sY= (sizeof(asY) > i) ? asY[i] : "";

      if (iColorIn)             // Get filled boxes
        ir= ir && s_AppendLine(ilArea, vX, vBar[i][], <>, <>, sprint("boxes fs solid ", "%3.1f", s_Fig[F_GLOB][G_DEFFILL]),
                               iColorIn, M_NAN, M_NAN, "");
                                // Get outline, if this is of a different color
                                // Always get the outline
      //if (!isint(iColorIn) || !iColorIn || (iIndex != iColorIn))
        ir= ir && s_AppendLine(ilArea, vX, vBar[i][], <>, <>, "boxes", iIndex, M_NAN, M_NAN, "");

      // Set the key
      iLine= s_Fig[F_PLOT][ilArea][P_NLINES]-1;
      s_Fig[F_PLOT][ilArea][P_LINE][iLine][L_KEY]= sY;

      s_Fig[F_GLOB][G_LASTPLOT]= ilArea;
      //DrawAdjust(ADJ_AREA_X, ilArea, dMin-0.6*dStep, dMin+(cBar-1)*dStep + 0.6*dStep);
      DrawAdjust(ADJ_AREA_Y, ilArea, 0, M_NAN);
      // DrawAxis(ilArea, TRUE, M_NAN, M_NAN, M_NAN, dMin, dStep, dStep);
      ++ilArea;
    }
  s_Fig[F_GLOB][G_LASTPLOT]= ilArea-1;
}

/*
**  DrawLegend
**
**  #include <packages/gnudraw/gnudraw.h>
**  DrawLegend(const iArea, const iOffsX, const iOffsY,
**             const fHidden);
**
**      iArea
**          in: int, area index
**      iOffsX
**          in: int, X pixel offset from top left, 0-1000
**              double, X fraction offset from left, 0-1
**              int, 0 or 1, indicating left or right
**      iOffsY
**          in: int, Y pixel offset from top left, 0-1000
**              double, Y fraction offset from top, 0-1
**              int, 0 or 1, indicating top or bottom
**      fHidden
**          in: int, TRUE: hide the legend
**
**  No return value.
*/
DrawLegend(const iArea, const iOffsX, const iOffsY,
           const fHidden)
{
  decl sLoc, dX, dY, bCoord, bRight, bBox, bReverse;

  s_InitFigure();
  if (iArea < 0)
    return;

  while ((sizeof(s_Fig) < F_LAST) || (s_Fig[F_NPLOTS] <= iArea))
    s_InitPlot();

  dX= iOffsX;
  dY= iOffsY;
  bCoord= TRUE;
  bRight= FALSE;
  bBox= FALSE;
  bReverse= TRUE;

  if (max(iOffsX, iOffsY) > 1)
    { // Translate to fractions
      dX/= 1000;
      dY/= 1000;
    }
  if ((sizerc(exclusion(iOffsX~iOffsY, <-1, 0, 1>)) == 0) &&
      isint(iOffsX)  && (isint(iOffsY)))
    { // Both -1 or 0 go left/top, 1 goes right/bottom
      bCoord= FALSE;
      dX= iOffsX == 1;      // Check if legend should go right
      dY= iOffsY == 1;      // Check if legend should go at bottom
    }
  bReverse= (dX < .5);            // If legend goes right, do not reverse label and line
  bRight= (dX > .5);              // If legend goes right, right-justify
  s_Fig[F_PLOT][iArea][P_LEGEND]=
    dX~dY~bCoord~bRight~bReverse~bBox~fHidden~1~-1;
  //  [dX, dY, bCoord, bRight, bReverse,  bBox, bHide, iCol, iFontsize]

  // println ("Legend: coord= ", bCoord, ", right= ", bRight, ", rev= ", bReverse);

  s_Fig[F_GLOB][G_LASTPLOT]= iArea;
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  #include <packages/gnudraw/gnudraw.h>
**  DrawLine(const iArea, const dX1, const dY1, const dX2,
**           const dY2, const iIndex)
**  DrawLine(const iArea, const dX1, const dY1, const dZ1,
**           const dX2, const dY2, const dZ2, const iIndex)
**
**      iArea
**          in: int, area index
**      dX1, dY1
**          in: int, real-world coordinates of starting point
**      dX2, dY2
**          in: int, Y real-world coordinates of end point
**      iIndex
**          in: int, line index for first row, see default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index.
**
**  No return value.
*/
DrawLine(const iArea, const dX1, const dY1, const dX2, const dY2, const iIndex, ...)
{
  decl mlX, mlY, ir, ava, dlX2, dlY2, dZ1, dZ2, ilIndex, sLoc, bHead,
  iColor, iN;

  s_InitFigure();
  if (iArea < 0)
    return;

  dlX2= dX2;
  dlY2= dY2;
  dZ1= dZ2= M_NAN;
  ilIndex= iIndex;
  ava= va_arglist();
  if (sizeof(ava) >= 2)
    {
      dZ1= dX2;
      dlX2= dY2;
      dlY2= iIndex;
      dZ2= ava[0];
      ilIndex= ava[1];
      if (binand(s_Fig[F_GLOB][G_WARN], 1))
        println("Warning: DrawLine with Z-index not implemented yet");
    }

//  Try-out to use vector-style for drawing arrows/lines.
//  Gives trouble with time-data: As only the original point is given
//    an xy coordinate, the time specification does not work out
//    correctly
  if (ismissing(dZ1))
    {
      if (!isstring(ilIndex) && (ilIndex < 0))
        { // Plot a point
          DrawXMatrix(iArea, dY1, "", dX1, "", 1, fabs(ilIndex));

          // add xdelta, ydelta to the y-coordinate
          iN= s_Fig[F_PLOT][iArea][P_NLINES];
          s_Fig[F_PLOT][iArea][P_LINE][iN-1][L_XYZ][3]|=
               (dlX2-dX1)|(dlY2-dY1);

          // Make it a 'vector'
          DrawAdjust(ADJ_COLOR, fabs(ilIndex), TP_USER, "vector");
        }
      else
        DrawXMatrix(iArea, dY1~dlY2, "", dX1~dlX2, "", 0, ilIndex);
    }

  s_Fig[F_GLOB][G_LASTPLOT]= iArea;
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawMatrix(const iArea, const mYt, const asY,
**      const dXfirst, const dXstep, ...);
**  DrawMatrix(const iArea, const mYt, const asY,
**      const dXfirst, const dXstep, iSymbol, iIndex);
**
**      iArea
**          in: int, area index
**      mYt
**          in: m x T matrix with m rows of data
**      asY
**          in: array of strings (holds variable names), or 0 (no names)
**      dXfirst
**          in: double, X-value of first observation, x
**      dXstep
**          in: double, gap between X-values, dx
**      iSymbol
**          in: int, 0: draw line, 1: draw symbols, 2: draw both, 3:
**          (GnuDraw only) draw indices (optional argument, default
**          is 0).
**      iIndex
**          in: int, line index for first row, see default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index.
**
**  No return value.
**
**  This is a more flexible version of the Draw() function. DrawMatrix draws the m
**  variables in the rows of mYt. The X variable consists of evenly spaced observations x,
**  x+dx, x+2dx, x+3dx, .... The following table gives the default settings for each line index.
**  Note that index 0 is the background colour, and 1 the foreground colour.
*/
DrawMatrix(const iArea, const mYt, const asY,
           const dXfirst, const dXstep, ...)
{
  decl ava, mX, iM, iSymbol, iIndex;

  s_InitFigure();
  if (iArea < 0)
    return;

  ava= va_arglist();
  iSymbol= sizeof(ava) > 0 ? ava[0] : 0;
  iIndex= sizeof(ava) > 1 ? ava[1] : s_Fig[F_GLOB][G_DEFLINE][0];

  if (isstring(iSymbol) || isarray(iSymbol))
    {
      oxwarning("(Array of) strings not allowed for line symbol");
      iSymbol= 0;
    }

  if (sizerc(dXfirst) + sizerc(dXstep) > 2)
    return;
  mX= range(0, columns(mYt)-1)*dXstep + dXfirst;

  iM= rows(mYt);
  if ((iSymbol == ST_INDEX) || (iSymbol == ST_INDEXSYMBOLS))
    mX+= dXstep*(range(0, iM-1)'/iM + 0.5*(1/iM - 1));

  // Use DrawXMatrix for the final plotting
  DrawXMatrix(iArea, mYt, asY, mX, "", iSymbol, iIndex);
}

/*
**  DrawPLine, DrawPSymbol, DrawPText
**
**  #include <packages/gnudraw/gnudraw.h>
**  DrawPLine(const iArea, const iX1, const iY1, const iX2,
**      const iY2, const iIndex);
**  DrawPSymbol(const iArea, const iX1, const iY1, const iX2,
**      const iY2, const iSymType, const iIndex);
**  DrawPText(const iArea, const sText, const iPx1,
**      const iPy1, ...);
**  DrawPText(const iArea, const sText,const iPx1,const iPy1,
**      const iFontNo, const iFontSize, const iTitle);
**  DrawPText(const iArea, const sText,const iPx1,const iPy1,
**      const iFontNo, const iFontSize, const iTitle, const iJust);
**
**  No return value.
**    Pixel coordinate versions of DrawLine, DrawSymbol DrawText.
*/
DrawPLine(const iArea, const iX1, const iY1, const iX2,
          const iY2, const iIndex)
{
  decl ilX1, ilY1, ilX2, ilY2, sLoc, iMax;

  s_InitFigure();
  if (iArea < 0)
    return;

  iMax= max(iX1, iY1, iX2, iY2);
  ilX1= (iMax > 1) ? iX1/15000 : iX1;
  ilY1= (iMax > 1) ? iY1/10000 : iY1;
  ilX2= (iMax > 1) ? iX2/15000 : iX2;
  ilY2= (iMax > 1) ? iY2/10000 : iY2;

  while ((sizeof(s_Fig) < F_LAST) || (s_Fig[F_NPLOTS] <= iArea))
    s_InitPlot();

  sLoc= "set arrow";
  sLoc= sprint(sLoc, " from screen ", "%4.2f,", ilX1, "%4.2f", ilY1,
               " to screen ", "%4.2f,", ilX2, "%4.2f", ilY2, " nohead lt ",
               "%2.0i", iIndex);
  if (sizeof(s_Fig[F_PLOT][iArea][P_SET][S_ARROW][0]) > 0)
    sLoc= sprint(s_Fig[F_PLOT][iArea][P_SET][S_ARROW][0],
                   "\n", sLoc);

  s_Fig[F_PLOT][iArea][P_SET][S_ARROW]= {sLoc, "set noarrow"};

  s_Fig[F_GLOB][G_LASTPLOT]= iArea;
}

DrawPSymbol(const iArea, const iX1, const iY1, const iX2,
            const iY2, const iSymType, const iIndex)
{
  if (binand(s_Fig[F_GLOB][G_WARN], 1))
    println("DrawPSymbol: Not implemented yet");
}

DrawPText(const iArea, const sText, const iPx1,
          const iPy1, ...)
{
  decl ilArea, ava, iFontNo, iFontSize, iTitle,
    iJust, sLabel;

  ava= va_arglist();
  iFontNo= (sizeof(ava) > 0) ? ava[0] : 0;
  iFontSize= (sizeof(ava) > 1) ? ava[1] : 0;
  iTitle= (sizeof(ava) > 2) ? ava[2] : TEXT_PANEL;    // Default on panel
  iJust= (sizeof(ava) > 3) ? ava[3] : -1;     // Default is left-justification

  if (iTitle == TEXT_TEXT)
    iTitle= TEXT_PANEL;

  // Use DrawText instead
  DrawText(0, sText, iPx1, iPy1, iFontNo, iFontSize, iTitle, M_NAN, M_NAN, iJust);
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawQQ(const iArea, const mY, const asY, const iDens,
**         const df1, const df2);
**  DrawQQ(const iArea, const mY, const asY, const iDens,
**         const df1, const df2, const fJoin);
**
**      iArea
**          in: int, area index
**      mY
**          in: k x T matrix, each row is a new plot
**      asY
**          in: string, variable name, or array of strings (k > 1)
**      iDens
**          in: int, one of: QQ_CHI, QQ_F, QQ_N, QQ_T, QQ_U, QQ_N_SE
**      df1
**          in: int, first parameter for distribution
**      df2
**          in: int, second parameter for distribution
**      fJoin
**         in: (optional, GnuDraw only) boolean, if TRUE quantile plots
**          are plotted together in one window.
**
**  No return value.
**
**  Draws a QQ plot. The following distributions are supported:
**
**      QQ_CHI : chi^2(df1),
**      QQ_F : F(df1, df2),
**      QQ_N : N(0, 1),
**      QQ_T : t(df1),
**      QQ_U : Uniform(0,1), resulting in a Quantile plot.
**      QQ_N_SE : N(0, 1), with se according to Engler & Nielsen (2009)
*/
DrawQQ(const iArea, const mY, const asY, const iDens, ...)
{
  decl i, vqY, vlY, vqF, vX, v45, vSQ, sKey, ir, sKey2, ava, df1, df2, dQ0,
       iT, dS, fJoin, ilArea, iIndex, iN;

  s_InitFigure();
  if (iArea < 0)
    return;

  ava= va_arglist();
  df1= sizeof(ava) > 0 ? ava[0] : M_NAN;
  df2= sizeof(ava) > 1 ? ava[1] : M_NAN;
  fJoin= sizeof(ava) > 2 ? ava[2] : s_Fig[F_GLOB][G_JOIN];
  ilArea= iArea;
  iIndex= s_Fig[F_GLOB][G_DEFLINE][0];

  v45= <M_INF, M_INF_NEG>;
  ir= TRUE;
  for (i= 0; i < rows(mY); ++i)
    {
      vlY= deleteifc(mY[i][], isdotmissing(mY[i][]));
      iT= columns(vlY);
      dQ0= 1/(iT+1);

      vqF= range(dQ0, 1-dQ0, max(dQ0, 1/1000));
      vqY= quantiler(vlY, vqF);
      dS= sqrt(varr(vlY));
      switch_single (iDens)
        {
          case QQ_CHI:
            {
              vX= quanchi(vqF, df1);
              sKey2= sprint(" x Chi^2(", df1, ")");
            }
          case QQ_F:
            {
              vX= quanf(vqF, df1, df2);
              sKey2= sprint(" x F(", df1, ",", df2, ")");
            }
          case QQ_N:
            {
              vX= dS*quann(vqF)+meanr(vlY);
              sKey2= " x normal";
            }
          case QQ_T:
            {
              vX= quant(vqF, df1);
              sKey2= sprint(" x t(", df1, ")");
            }
          case QQ_U:
            {
              vX= vqF;
              if (iT < 1000)
                vqY= sortr(vlY);
              sKey2= " x uniform";
            }
          case QQ_N_SE:
            {
              vX= dS*quann(vqF)+meanr(vlY);
              vSQ= sqrt(vqF .* (1-vqF)/iT) ./ densn(quann(vqF));
              sKey2= " x normal";
            }
          default:
            oxwarning ("Unknown QQ-plot");
        }
      sKey= "";
      if (isstring(asY))
        sKey= asY;

      v45= min(v45[0], vX)~max(v45[1], vX);
      if (!fJoin || (i == rows(mY)-1))
        ir= ir && s_AppendLine(ilArea, v45, v45, <>, <>, "lines", 1, M_NAN, M_NAN, "");

      if (isarray(asY) && (sizeof(asY) > i))
        sKey= asY[i];
      if (sKey != "")
        sKey= sprint(sKey, sKey2);
      ir= ir && s_CheckSize(vX, vqY, 0, 1, "DrawQQ", &iN) &&
          s_AppendLine(ilArea, vX, vqY, <>, <>, "lines", iIndex, M_NAN, M_NAN, sKey);

      if (iDens == QQ_N_SE)
        ir= ir && s_CheckSize(vX, vqY, 0, 1, "DrawQQ", &iN) &&
            s_AppendLine(ilArea, vX, vX - quann(0.975) * vSQ,
              <>, <>, "lines", iIndex+1, M_NAN, M_NAN, "") &&
            s_AppendLine(ilArea, vX, vX + quann(0.975) * vSQ,
              <>, <>, "lines", iIndex+1, M_NAN, M_NAN, "");

      sKey= "";
      s_Fig[F_PLOT][ilArea][P_TICSAXISBORDER][0][:1]= FALSE;

      iIndex += fJoin;
      ilArea += !fJoin;
      if (!fJoin)
        v45= <M_INF, M_INF_NEG>;
    }

  s_Fig[F_GLOB][G_LASTPLOT]= ilArea - !fJoin;
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawSpectrum(const iArea, const mY, const asY,
**      const iOrder);
**  DrawSpectrum(const iArea, const mY, const asY,
**      const iOrder, const fJoin);
**
**      iArea
**          in: int, area index
**      mY
**          in: k x T matrix, each row is a new plot
**      asY
**          in: string, variable name, or array of strings (k > 1)
**      iOrder
**          in: int, lag truncation parameter m
**      fJoin
**         in: (optional, GnuDraw only) boolean, if TRUE spectra are plotted
**         together in one window.
**
**  No return value.
**
**  Draws the estimated spectral density.
*/
DrawSpectrum(const iArea, const mY, const asY, const iOrder, ...)
{
  decl ir, vX, vPer, sKey, i, iN, cp, vYL, iIndex, ava, ilArea, fJoin;

  s_InitFigure();
  if (iArea < 0)
    return;

  ava= va_arglist();
  fJoin= sizeof(ava) ? ava[0] : s_Fig[F_GLOB][G_JOIN];

  cp= 128;
  vX= range(0, 1, 1/(cp-1));
  sKey= "";
  if (isstring(asY))
    sKey= asY;

  iIndex= s_Fig[F_GLOB][G_DEFLINE][0];
  ilArea= iArea;
  for (i= 0; i < rows(mY); ++i)
    {
      if (isarray(asY) && (i < sizeof(asY)))
        sKey= asY[i];

      vYL= deleteifc(mY[i][], isdotmissing(mY[i][]));
      vPer= periodogram(vYL', iOrder, cp, 2)';
      ir= s_CheckSize(vX, vPer, 0, 1, "DrawSpectrum", &iN) &&
          s_AppendLine(ilArea, vX[:iN-1], vPer[:iN-1], <>, <>, "lines",
            iIndex, M_NAN, M_NAN, sKey);
      ilArea+= !fJoin;
      iIndex+= fJoin;
    }
  s_Fig[F_GLOB][G_LASTPLOT]= ilArea - !fJoin;
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawSymbol(const iArea, const dX1, const dY1, const dX2,
**             const dY2, const iSymType, const iIndex);
**
**      iArea
**          in: int, area index
**      dX1, dY1
**          in: int, real-world coordinates, lower-left corner of bounding box
**      dX2, dY2
**          in: int, real-world coordinates, upper-right corner of bounding box
**      iSymType
**          in: int, symbol type
**      iIndex
**          in: int, line index for first row, see default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index.
**
**  No return value.
*/
DrawSymbol(const iArea, const dX1, const dY1, const dX2,
           const dY2, const iSymType, const iIndex)
{
  decl vC, vAB, i, vR, mR;

  if (iArea < 0)
    return;

  vC= (dX1 + dX2)/2 | (dY1 + dY2)/2;        // Center point
  vAB= fabs(dX1 - dX2)/2 | fabs(dY1 - dY2)/2;   // Half distance
  switch_single (iSymType)
    {
      case PL_BOX:
        {
          DrawXMatrix(iArea, dY1~dY2~dY2~dY1~dY1, "", dX1~dX1~dX2~dX2~dX1, "", 0, iIndex);
        }
      case PL_FILLBOX:
        {
          DrawXMatrix(iArea, dY1~dY2~dY2~dY1~dY1, "", dX1~dX1~dX2~dX2~dX1, "", 0, iIndex);
          DrawAdjust(ADJ_FILLCURVE);
        }
      case PL_PLUS:
        {
          DrawXMatrix(iArea, (dY1~dY2)|(<1, 1> * vC[1]), "",
                         (<1, 1> * vC[0])|dX1~dX2, "", 0, iIndex~iIndex);
        }
      case PL_DASH:
        {
//           for (i= 0; i < 3; ++i)
//             DrawXMatrix(iArea, vC[1] + (sin(i*M_PI/3)~-sin(i*M_PI/3)),
//               "", vC[0] + (cos(i*M_PI/3)~-cos(i*M_PI/3)), "", 0, iIndex);
          DrawXMatrix(iArea, vC[1]~vC[1], "", dX1~dX2, "", 0, iIndex);
        }
      case PL_CIRCLE:
        {
          vR= range(0, 1, .01)*M_2PI;
          mR= vC + vAB .* (cos(vR)|sin(vR));
          DrawXMatrix(iArea, mR[1][],
              "", mR[0][], "", 0, iIndex);
        }
      case PL_FILLCIRCLE:
        {
          vR= range(0, 1, .01)*M_2PI;
          mR= vC + vAB .* (cos(vR)|sin(vR));
          DrawXMatrix(iArea, mR[1][],
              "", mR[0][], "", 0, iIndex);
          DrawAdjust(ADJ_FILLCURVE);
        }
      case PL_LINE:
        DrawXMatrix(iArea, dY1~dY2, "", dX1~dX2, "", 0, iIndex);
      case PL_DIAMOND:
        {
          DrawXMatrix(iArea, vC[1]~dY1~vC[1]~dY2~vC[1], "",
                      dX1~vC[0]~dX2~vC[0]~dX1, "", 0, iIndex);
        }
      case PL_FILLDIAMOND:
        {
          DrawXMatrix(iArea, vC[1]~dY1~vC[1]~dY2~vC[1], "",
                      dX1~vC[0]~dX2~vC[0]~dX1, "", 0, iIndex);
          DrawAdjust(ADJ_FILLCURVE);
        }
      case PL_TRIANGLE:
        {
          DrawXMatrix(iArea, dY1~dY2~dY1~dY1, "",
                      dX1~vC[0]~dX2~dX1, "", 0, iIndex);
        }
      case PL_FILLTRIANGLE:
        {
          DrawXMatrix(iArea, dY1~dY2~dY1~dY1, "",
                      dX1~vC[0]~dX2~dX1, "", 0, iIndex);
          DrawAdjust(ADJ_FILLCURVE);
        }
      case PL_CROSS:
        {
          DrawXMatrix(iArea, dY1~dY2|dY2~dY1, "",
                      dX1~dX2|dX1~dX2, "", 0, iIndex~iIndex);
        }
      case PL_CNTSYMBOL:     // Corresponds to a dot
        {
          DrawXMatrix(iArea, vC[1]~vC[1], "",
                      vC[0]~vC[0], "", 0, iIndex~iIndex);
        }

      default:
        if (binand(s_Fig[F_GLOB][G_WARN], 1))
          println("DrawSymbol: Symbol ", iSymType, " not implemented yet");
    }
//   DrawAdjust(ADJ_AREA_X, iArea, dX1, dX2);
//   DrawAdjust(ADJ_AREA_Y, iArea, dY1, dY2);
}

/*
**  DrawT
**
**  #include <packages/gnudraw/gnudraw.h>
**  DrawT(const iArea, const mYt, const mnYear,
**        const mnPeriod, const iFreq);
**  DrawT(const iArea, const mYt, const mMY, const dDum, const dDum);
**
**      iArea
**          in: int, area index
**      mYt
**          in: m x T matrix with m y variables
**      mnYear
**          in: int, year of first observation
**      mnPeriod
**          in: int, period of first observation
**      iFreq
**          in: int, frequency of observations
**      mDMY
**          in: 2 x T matrix with month and year or
**              3 x T matrix with day, month and year
**      dDum
**          in: not used, placeholder
**
**  No return value.
**
**  Draws m variables in the specified area against time. Each variable is drawn by linking up the
**  points. The first line index is 2. Time can be specified as a
**  combination of the year and period of the first observation together
**  with the frequency, as a 2 x T matrix with months and years or as a 3 x
**  T matrix with day, month and years
*/
DrawT(const iArea, const mYt, const mnYear,
      const mnPeriod, const iFreq)
{
  DrawTMatrix(iArea, mYt, "", mnYear, mnPeriod, iFreq);
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawText(const iArea, const sText, const dX1, const dY1, ...);
**  DrawText(const iArea, const sText, const dX1, const dY1, ...);
**  DrawText(const iArea, const sText, const dX1, const dY1, const iFontNo, const iFontSize, const iTitle, const iRotation, const dZ1)
**  DrawText(const iArea, const sText, const dX1, const dY1, const iFontNo, const iFontSize, const iTitle, const iRotation, const dZ1, const iJust);
**  Deprecated: DrawText(const iArea, const sText, const dX1, const dY1, const iFontNo, const iFontSize, const iTitle, const iRotation, const dZ1, const iGraph, const iJust);
**  DrawTitle(const iArea, const sText);
**
**    iArea
**        in: int, area index
**    dX1, dY1, dZ1
**        in: real-world coordinates, anchor of text. Z-coordinate is
**        required for 3-dimensional plots
**    iFontNo (not implemented in GnuDraw)
**        in: int, font number (0 for first font; use -1 for the default font)
**    iFontSize (not implemented in GnuDraw)
**        in: int, font size (e.g. 300; use -1 for the default size)
**    iTitle
**        in: int, 0: is normal text, else is graph title (coordinates are ignored):
**        TEXT_TITLE - graph title
**        TEXT_XLABEL - label along X-axis
**        TEXT_YLABEL - label along Y-axis
**        TEXT_ZLABEL - label along Z-axis
**    iRotation
**        in: int, rotation (in degrees, default is 0)
**    dZ1
**        in: int, real-world Z coordinate of text anchor (for text in 3D graphs; default is 0)
**    iGraph (deprecated)
**        in: boolean (optional, GnuDraw specific, default is FALSE), if
**        TRUE coordinates are takes to run from (0,0) to (1,1) within this
**        graphing window.
**    sJust/iJust
**        in: string or integer (optional, GnuDraw specific), indicates
**        justification of label. Possible (GnuPlot) options are left/-1
**        (default), center/0 or right/1
**
**  No return value.
*/
DrawText(const iArea, const sText, const dX1, const dY1, ...)
{
  decl ava, dX1L, dY1L, dZ1, iAreaL, i, sLabel, iMax,
       iTitle, iJust, b3Dim, iGraph, iFontNo, iFontSize,
       iRotation, sTextL;

  s_InitFigure();

  ava= va_arglist();
  dX1L= dX1;
  dY1L= dY1;
  dZ1= 0;
  iAreaL= max(iArea, 0);
  iFontNo= (sizeof(ava) > 0) ? ava[0] : -1;
  iFontSize= (sizeof(ava) > 1) ? ava[1] : -1;
  iTitle= (sizeof(ava) > 2) ? ava[2] : FALSE;
  iRotation= (sizeof(ava) > 3) ? ava[3] : M_NAN;
  dZ1= (sizeof(ava) > 4) ? ava[4] : 0;
  iJust= (sizeof(ava) > 5) ? ava[5] : -1;
  if (sizeof(ava) > 6)
    { // Deprecated use of iGraph, but still allowed.
      iGraph= ava[5];
      iTitle= iGraph == 1 ? TEXT_AREA :
              iGraph == 2 ? TEXT_PANEL : iTitle;
      iJust= ava[6];
    }
  if (isstring(iJust))
    iJust= iJust == "left" ? -1 :
           iJust == "center" ? 0 :
           iJust == "right" ? 1 : M_NAN;
  sTextL= string(array(sText)[0]);

  // Take care with limits of iTitle
  iTitle= (iTitle < TEXT_TEXT) || (iTitle > TEXT_PANEL) ? TEXT_TEXT : iTitle;

  while ((sizeof(s_Fig) < F_LAST) || (s_Fig[F_NPLOTS] <= iAreaL))
    s_InitPlot();

  if (ismissing(dX1L~dY1L~dZ1) && (iTitle == TEXT_TEXT))
    return;

  if ((iTitle == TEXT_TITLE) && (iArea == -1))
    { // For a global title, set a title at .5,1, centered
      dX1L= .5;
      dY1L= .97;
      iJust= 0;
      iTitle= TEXT_PANEL;
      // Scale y down by .97, to allow for title
      s_Fig[F_SIZES][3]= min(s_Fig[F_SIZES][3], .97);
    }
  if (iTitle == TEXT_PANEL) // Place screen-coordinate label preferably on area 0
    iAreaL= 0;

  iMax= max(dX1L, dY1L);
  dX1L= (iMax > 1) && !(iTitle != <TEXT_PANEL, TEXT_AREA>) ? dX1L/15000 : dX1L;
  dY1L= (iMax > 1) && !(iTitle != <TEXT_PANEL, TEXT_AREA>) ? dY1L/10000 : dY1L;

  s_Fig[F_PLOT][iAreaL][P_LABEL][0]~= {sTextL};
  s_Fig[F_PLOT][iAreaL][P_LABEL][1]|=
    dX1L~dY1L~dZ1~iFontNo~iFontSize~iTitle~iJust~iRotation;

  #ifdef GNUDRAW_DEBUG
    println ("Put text ", sTextL, " at title = ", iTitle, " with coord (",
             dX1L, ",", dY1L, ") at area ", iAreaL, " with rot ",
             iRotation, ", with fontsize ",
             iFontSize);
  #endif
  s_Fig[F_GLOB][G_LASTPLOT]= iAreaL;
}

DrawTitle(const iArea, const sText)
{
  DrawText(iArea, sText, 0, 0, -1, -1, TEXT_TITLE);
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawTMatrix(const iArea, const mYt, const asY, ...);
**  DrawTMatrix(const iArea, const mYt, const asY,
**      const mnYear, const mnPeriod, const iFreq, ...);
**  DrawTMatrix(const iArea, const mYt, const asY,
**      const mYMD, const dDum, const dDum, ...);
**  DrawTMatrix(const iArea, const mYt, const asY,
**      const mnYear, const mnPeriod, const iFreq, iSymbol,
**      iIndex);
**  DrawTMatrix(const iArea, const mYt, const asY,
**      const mYMD, const dDum, const dDum, iSymbol, iIndex);
**
**      iArea
**          in: int, area index
**      mYt
**          in: m x T matrix with m y variables
**      asY
**          in: array of strings (holds variable names), or 0 (no names)
**      mnYear
**          in: int, year of first observation
**      mnPeriod
**          in: int, period of first observation, or (GnuDraw only)
**          in: string with format for date labels on axis
**      iFreq
**          in: int, frequency of observations, or (GnuDraw only)
**          in: int, number of small tics between larger ticmarks
**      mYMD
**          in: 1 x T matrix with Julian day, and possibly intra-day
**              timing as a fraction of the day
**              4 x T matrix with Julian day, hour, min, seconds or
**              6 x T matrix with year, month, day, hour, min, and seconds, or
**              2 x T matrix with Julian day and seconds since midnight, or
**              4 x T matrix with year, month, day, and seconds since midnight.
**              Higher frequencies can be skipped from the matrix if not
**              used. To leave lower frequencies unspecified, use  rows of
**              missings for year, month, day, etc.
**      sFormat
**          in: string, with user-specified datings the format on the
**              time axis can be specified, using a subset of %Y, %y,
**              %m, %d, %H, %M, %S for the year (4-digit or 2-digit),
**              month, day, hour, minute and second. The string should
**              contain opening and closing quotes, e.g.
**              sFormat= "'%H:%M:%s'". If not-specified (e.g., zero,
**              missing, or empty string), a default format is chosen.
**      imxTics
**          in:   int, number of intervals between larger tic-marks. If zero,
**                default value is chosen.
**      iSymbol DrawTMa
**          in: int, 0: draw line, 1: draw symbols, 2: draw both, 3:
**          (GnuDraw only) draw indices (optional argument, default
**          is 0).
**      iIndex
**          in: int, line index for first row, see Default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index.
**
**  No return value.
**
**  This is a more flexible version of the DrawT() function. Draws m variables in the specified
**  area against time. See under DrawMatrix for the default settings for each line index.
*/
DrawTMatrix(const iArea, const mYt, const asY, ...)
{
  decl vDates, vSec, iT, iM, iEM, ir, mlY, vm, i, j, ava, mnYear,
       mnPeriod, iFreq, vSymbol, iIndex, dL, sFormat, imxTics, asTics,
       vMonthMax, iMonth, iDay, mYMD;

  s_InitFigure();
  if (iArea < 0)
    return;

  ava= va_arglist();
  if (sizeof(ava) == 0)
    return DrawMatrix(iArea, mYt, asY, 1, 1);

  mnYear= (sizeof(ava) > 0) ? ava[0] : 0;
  mnPeriod= (sizeof(ava) > 1) ? ava[1] : 0;
  iFreq= (sizeof(ava) > 2) ? int(ava[2]) : 0;
  vSymbol= (sizeof(ava) > 3) ? ava[3] : 0;
  iIndex= (sizeof(ava) > 4) ? vecr(ava[4]) : s_Fig[F_GLOB][G_DEFLINE][0];

  if (isstring(vSymbol) || isarray(vSymbol))
    {
      oxwarning("(Array of) strings not allowed for line symbol");
      vSymbol= 0;
    }
  vSymbol= vecr(matrix(vSymbol));

  iT= columns(mYt);
  sFormat= "";
  imxTics= 0;

  if ((s_Fig[F_NPLOTS] > iArea) &&
      s_Fig[F_PLOT][iArea][P_NLINES] &&
      !s_Fig[F_PLOT][iArea][P_TYPE][T_DMY])
    {
//       print (iArea~s_Fig[F_PLOT][iArea][P_NLINES], s_Fig[F_PLOT][iArea]);
      oxwarning("Combining xy and time series plots not implemented");
      return 0;
    }

  if ((sizerc(mnYear)*sizerc(mnPeriod)*sizerc(iFreq) == 1)
      && !isstring(mnPeriod) && (iFreq > 0))
    {
      if ((mnYear >= 0) && (iFreq <= 12))
        { // Monthly or quarterly data
          vm= range(0, iT-1)'*12/iFreq + (mnPeriod-1)*12/iFreq + 1;
          // Set yy/mm
          vDates= dayofcalendar(idiv(vm-1, 12)+mnYear, imod(vm-1, 12)+1, ones(vm))';
        }
      else if (mnYear >= 0)
        { // Frequency higher than monthly, difficult...
          vMonthMax= <31, 28, 31, 30, 31, 30,
                      31, 31, 30, 31, 30, 31>;
          iMonth= ceil(12*mnPeriod/iFreq);
          iDay= ceil(mnPeriod/iFreq*365) - (iMonth-1)*30;
          iDay= min(iDay, vMonthMax[iMonth-1]);
          vDates= dayofcalendar(mnYear, iMonth, iDay)+round(365*range(0, iT-1)/iFreq);
        }
      else if (mnYear < 0)
        { // Year -> Hours, Period -> Minutes, Freq -> per hour
          vSec= -mnYear*3600 + range(0, iT-1)*3600/iFreq + 60*mnPeriod;
          vDates= vSec/(24*3600);
        }
      else
        {
          oxwarning ("Incorrect specification of time in DrawT");
          return;
        }
    }
  else if (sizer(mnYear) == 1)
    {
      vDates= mnYear;
    }
  else if (sizec(mnYear) == 1)
    { // In case a column of dates is given, GnuDraw will not complain
      vDates= mnYear';
    }
  else if (sizer(mnYear) == 3)
    {
      if ((max(mnYear[2][]) > 31) && (max(mnYear[0][]) < 31))
        // Plot DMY
        vDates= dayofcalendar(mnYear[2][]', mnYear[1][]', mnYear[0][]')';
      else
        // Plot YMD
        vDates= dayofcalendar(mnYear[0][]', mnYear[1][]', mnYear[2][]')';
    }
  else if (sizer(mnYear) == 6)
    {
      vDates= dayofcalendar(mnYear[0][]', mnYear[1][]', mnYear[2][]')'
        + timeofday(mnYear[3][]', mnYear[4][]', mnYear[5][]')';
    }
  else
    {
      oxwarning("Specify mnYear as mYMDHMS or vDates, DrawTMatrix not backwards compatible");
      return;
    }

  if (sizerc(mnYear) == 0)
    {
      //  oxwarning ("Incorrect specification of time in DrawT");
      return;
    }

  // Get out string with format
  if (isstring(mnPeriod) && (sizeof(mnPeriod) > 1))
    sFormat= mnPeriod;
  if (isint(iFreq) && (iFreq > 0))
    imxTics= iFreq;

  iT= min(columns(mYt), columns(vDates));

  // Build special ticmarks for quarters, 1984:1--1987:4 etc.
  asTics= "";
  if ((sizerc(mnYear) == 1) && (iFreq == 4) && (mnYear > 0))
    { // If there is quarterly data
      iEM= iT < 13 ? 1 : // Number of observations to skip
           iT < 25 ? 2 :
           iT < 49 ? 4 :
           4*idiv(iT, 3*12);
      mYMD= dayofcalendar(vDates)'|timeofday(fmod(vDates, 1))';
      j= 0;
      if (iEM > 1)
        {
          j= min(vecrindex(mYMD[1][] .== 1));
          j= isdotnan(j) ? 0 : j;
        }
      while (j < iT)
        {
          if (!sizeof(asTics))
            asTics= "set xtics (\\\n  ";
          else
            asTics~=", \\\n  ";
          asTics~=sprint("'", "%02d", imod(mYMD[0][j], 100), ":",
                         idiv(mYMD[1][j]-1, 3)+1,
                         "' '");
          asTics~= sprint("%i", mYMD[0][j], " ");
          for (i= 1; i < 6; ++i)
            asTics~= sprint("%i", mYMD[i][j], " ");
          asTics~= "'";
          j+= iEM;
        }
      asTics ~= ")";
      imxTics= M_NAN;
    }

  // Compute time length in days
  dL= iT ? vDates[iT-1]-vDates[0] : 0;
  if (!imxTics)
    imxTics= dL < 1/24 ? 5 :            // Less than an hour
             dL < 6/24 ? 4 :            // 6 hours
             dL < 1   ? 4 :             // day
             dL < 365 ? 3 :             // year
             dL < 50*365 ? 4 :          // 50 years
             dL < 100*365 ? 2 : 1;      // 100 years

  // Plot using DrawXMatrix
  DrawXMatrix(iArea, mYt, asY, vDates, "", vSymbol, iIndex);
  DrawAdjust(ADJ_AXISSCALE, AXIS_DATE);

  s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][6][0]=
    isnan(imxTics) ? -1 : imxTics;

  if (sizeof(asTics))
    s_Fig[F_PLOT][iArea][P_SET][S_XTICS]=
      {sprint(asTics), "set xtics autofreq"};

  if (sizeof(sFormat))
    {
      s_Fig[F_PLOT][iArea][P_SET][S_FORMAT][0]~=
        sprint("set format x ", sFormat);
      s_Fig[F_PLOT][iArea][P_SET][S_FORMAT][1]~=
        sprint("set format x '%g'");
      s_Fig[F_PLOT][iArea][P_AXISSCALE][0][3]= TRUE;
    }

  s_Fig[F_GLOB][G_LASTPLOT]= iArea;
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawX(const iArea, const mYt, const vX);
**
**      iArea
**          in: int, area index
**      mYt
**          in: m x T matrix with m y variables
**      vX
**          in: 1 x T matrix with x variable
**
**  No return value.
**
**  Draws m y variables in the specified area against an x variable. Each point is marked, but the
**  points are not linked, resulting in a cross plot. The first line index is 2.
*/
DrawX(const iArea, const mYt, const vX)
{
  DrawXMatrix(iArea, mYt, "", vX, "", 1);
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawXMatrix(const iArea, const mYt, const asY, const vX,
**              const sX, ...);
**  DrawXMatrix(const iArea, const mYt, const asY, const vX,
**              const sX, iSymbol, iIndex);
**
**      iArea
**          in: int, area index
**      mYt
**          in: m x T matrix with m y variables
**      asY
**          in:   array of strings (holds variable names), or 0 (no names)
**      vX
**          in: 1 x T matrix or m x T matrix with x variable(s)
**      sX
**          in:   string or array of strings (holds variable names), or 0 (no names)
**      iSymbol
**          in: int, 0: draw line, 1: draw symbols, 2: draw both, 3:
**          (GnuDraw only) draw indices (optional argument, default
**          is 0).
**      iIndex
**          in: int, line index for first row, see Default line attributes, (optional argument,
**          default is 2). Each subsequent row will have the next index.
**
**  No return value.
**
**  This is a more flexible version of the DrawX() function. Draws m variables in the specified
**  area against an x variable See under DrawMatrix for the default settings for each line
**  index.
*/
DrawXMatrix(const iArea, const mYt, const asY, const vX,
            const sX, ...)
{
  decl ir, i, iM, ava, iIndex, vIndex, vSymbol, iSymbol, sStyle, sKey, asX, alsY,
    bX, bY, iT, vI;

  s_InitFigure();
  if (iArea < 0)
    return;
  while ((sizeof(s_Fig) < F_LAST) || (s_Fig[F_NPLOTS] <= iArea))
    s_InitPlot();

  ava= va_arglist();
  vSymbol= sizeof(ava) > 0 ? ava[0] : 0;
  if (isstring(vSymbol) || isarray(vSymbol))
    {
      oxwarning("(Array of) strings not allowed for line symbol");
      vSymbol= 0;
    }
  vSymbol= vecr(matrix(vSymbol));
  vSymbol= (vSymbol .> ST_SHADING) .|| (vSymbol .< ST_LINE)
        .? ST_LINE
        .: vSymbol;
  iIndex= sizeof(ava) > 1 ? ava[1] : s_Fig[F_GLOB][G_DEFLINE][0];

  alsY= isarray(asY) ? asY : isstring(asY) ? {asY} : {""};
  asX= isarray(sX) ? sX : isstring(sX) ? {sX} : {""};

  ir= s_CheckSize(vX, mYt, 0, 1, "DrawX", &iT);

  bX= rows(vX) > 1;
  bY= rows(mYt) > 1;
  iM= max(sizer(vX), sizer(mYt));

  if (sizerc(iIndex) == 0)
    iIndex= s_Fig[F_GLOB][G_DEFLINE][0];

  i= sizerc(iIndex);
  vIndex= isstring(iIndex) ? {iIndex}
          : i < iM
          ? vecr(iIndex)|range(iIndex[i-1]+1, iIndex[i-1]+iM-i)'
          : iIndex;
  i= sizerc(vSymbol);
  vSymbol= i < iM
           ? vSymbol|vSymbol[i-1].*ones(iM-i, 1)
           : vSymbol;
  for (i= 0; i < iM; ++i)
    {
      if (i < sizerc(vIndex))
        iIndex= vIndex[i];
      sKey= "";

      iSymbol= vSymbol[i];
      sStyle= {"lines", "points", "linespoints", "impulses", "impulses",
           "boxes", "filledcurves"}[iSymbol];
      if ((sizeof(alsY) > i) || (sizeof(asX) > i))
        {
          sKey= alsY[min(i, sizeof(alsY)-1)];
          if ((sizeof(asX[0]) > 0) && (sizeof(sKey) > 0))
            sKey~= sprint(" x ", asX[min(sizeof(asX)-1, i)]);
        }
      if (iSymbol == ST_SHADING)
        { // Find shading indices
          vI= s_Shading(mYt[i*bY][:iT-1]);
          // Shade from x1 to x2 axes
          ir= ir && s_AppendLine(iArea, vX[i*bX][:iT-1], vI,
                      <>, <>, sStyle~" x1", iIndex, M_NAN, M_NAN, sKey, TRUE)
                 && s_AppendLine(iArea, vX[i*bX][:iT-1], vI,
                      <>, <>, sStyle~" x2", iIndex, M_NAN, M_NAN, "", TRUE);
        }
      else
        ir= ir && s_AppendLine(iArea, vX[i*bX][:iT-1], mYt[i*bY][:iT-1],
                               <>, <>, sStyle, iIndex, M_NAN, M_NAN, sKey);
    }

  s_Fig[F_GLOB][G_LASTPLOT]= iArea;
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawZ(const vZ, ...);
**  DrawZ(const vZ, const sZ, const iMode, const dFac,
**      const iIndex);
**  DrawZ(const vZ, const sZ, const iMode, const dFac,
**      const iSymbol, const iIndex);
**
**      vZ
**          in: 1 x T matrix with Z variable (symmetric bounds) or
**              2 x T matrix with Z variable (asymmetric bounds, with
**              BAR, BAND and HILO plots) or
**              q x T matrix, with width of quantiles to be plotted (for
**              FAN plot)
**      sZ
**          in: string, 1 name of Z variable
**      iMode
**          in: int, type of Z variable (optional argument):
**          ZMODE_SYMBOL - use values as symbol size,
**          ZMODE_VALUE - draw value as text,
**          ZMODE_BAR - draw error bars (the default),
**          ZMODE_BAND - draw error bands.
**          ZMODE_FAN - Draw fan plot
**          ZMODE_HILO - Draw hi-lo plot
**      dFac
**          in: double, bar/band factor (optional argument, 2.0 is
**              default) or
**              q vector with quantiles which are used in FAN plot
**              with vZ of size q x T
**      iSymbol
**          in: int, 0: draw line, 1: draw symbols, 2: draw both, 3:
**          (GnuDraw only) draw indices (optional argument, default
**          is 0). Specifies the line/symbol type of the error bands,
**          when used in combination with ZMODE_BAND; skipped otherwise.
**      iIndex
**          in: int, line index for first row, see Default line attributes,
**          (optional argument, default is 2).
**
**  No return value.
**
**  DrawZ adds a Z component to the most recent graphics object. DrawZ should be a
**  should be used immediately after a call to one of the draw functions Draw,
**  DrawMatrix, DrawX, etc.).
*/
DrawZ(const vZ, ...)
{
  decl ava, sZ, iMode, dFac, iSymbol, iIndex, iArea, iLine, i, ir, mX,
       mY, mYZ, mZ, mLU, mYMD, vSpec, vZl, vQ, vQY, iZ, iT, vOff,
       sStyle, aLine, sKey1, sKey2, vCol, dS;

  s_InitFigure();

  iArea= s_Fig[F_GLOB][G_LASTPLOT];
  if (iArea < 0)
    return; // No plot available
  iLine= s_Fig[F_PLOT][iArea][P_NLINES]-1;
  if (iLine < 0)
    return; // No line available
//   if (s_Fig[F_PLOT][iArea][P_TYPE][T_XYZ])
//     {
//       println ("Warning: Already 3-dim plot found...");
// //       return; // 3-dim plot, no Z can be added
//     }

  // Read out the line to change
  aLine= s_Fig[F_PLOT][iArea][P_LINE][iLine];

  ava= va_arglist();
  sZ=     (sizeof(ava) > 0) && isstring(ava[0]) ? ava[0] : "";
  iMode=  sizeof(ava) > 1 ? ava[1] : ZMODE_BAR;
  dFac=   sizeof(ava) > 2 ? ava[2] : 2;
  iIndex= sizeof(ava) > 3 ? ava[3] : aLine[L_TYPE][1]+1;

  iSymbol= -1;
  if (sizeof(ava) > 4)
    {
      iIndex= ava[4];
      iSymbol= max(min(ava[3], 3), -1);
//       println ("Deprecated usage: Do not use DrawZ(..., iSymbol, iIndex)");
    }

  // Adapt the key to this line
  sKey1= sizerc(aLine[L_KEY]) && sizerc(sZ) ?
            sprint(aLine[L_KEY], " x ", sZ) : aLine[L_KEY];
  sKey2= "";
  if (s_Fig[F_GLOB][G_LABDRAWZ] == 0)  // Old style, separate zmode labels
    {
      sKey1= aLine[L_KEY];
      sKey2= sZ;
    }

  iZ= rows(vZ);
  vZl= vZ;
  [mX, mY]= aLine[L_XYZ][0:1];
  if (sizerc(aLine[L_XYZ][2]))
    oxwarning("Resetting the Z-values in existing 3d plot not implemented");
  sStyle= (iSymbol >= ST_LINE) && (iSymbol <= ST_SHADING)
    ? {"lines", "points", "linespoints", "impulses", "impulses",
       "boxes", "boxes"}[iSymbol]
    : aLine[L_TYPE][0];
  iT= columns(mX);
  if (iT > columns(vZl))
    vZl~= constant(M_NAN, iZ, iT-columns(vZl));
  ir= s_CheckSize(mX, mY, vZl, 3, "DrawZ", &iT);
  if (!ir || (iZ == 0))
    {
      // Do nothing
    }
  else if (iMode == ZMODE_BAR)
    {
      // For errorbars, gnudraw should receive either
      //   fac, z= sigma
      //   fac, z= sigma+, sigma-
      //   fac=0, z= L, U
      if (iZ == 1)                   // Gnuplot needs
        mZ= dFac*vZl[0][:iT-1];                     // x y ydelta
      else if (dFac)
        mZ= mY + dFac*<-1; 1> .* vZl[:1][:iT-1];    // x y ylow yhigh
      else
        mZ= vZl[:1][:iT-1];                         // x y ylow yhigh

      // Add an extra line; the original one draws the line, the
      //   new one adds the error bars
      aLine[L_KEY]= sKey1;
      s_Fig[F_PLOT][iArea][P_LINE][iLine]= aLine;

      // Note that extra line later on gets changed again...
      aLine[L_KEY]= sKey2;     // New line will not contain a key
      s_Fig[F_PLOT][iArea][P_LINE]|= {aLine};
      ++iLine;
      ++s_Fig[F_PLOT][iArea][P_NLINES];

      // Place the error bars in the 'extra' component
      aLine[L_TYPE][0]= "yerrorbars";
      aLine[L_TYPE][1]= iIndex;     // Change the color of the bars
      aLine[L_POINT][0]= PL_DOT;    // Choose pointtype DOT, no symbol
      aLine[L_XYZ][3]= mZ;
    }
  else if (iMode == ZMODE_BAND)
//     {
//       // Reset the key
//       aLine[L_KEY]= sKey1;
//
//       // For errorbars, gnudraw should receive either
//       //   fac, z= sigma
//       //   fac, z= sigma+, sigma-
//       //   fac=0, z= L, U
//       if (iZ == 1)                  // Gnuplot needs x L, x U
//         mZ= mY + dFac*<-1; 1> .* vZl[0][:iT-1];
//       else if (dFac)
//         mZ= mY + dFac*<-1; 1> .* vZl[:1][:iT-1];
//       else
//         mZ= vZl[:1][:iT-1];
//
//       // Add extra standard lines, with bands
//       ir= s_AppendLine(iArea, mX, mZ[0][:iT-1], <>, <>, sStyle,
//                        iIndex, M_NAN, M_NAN, sKey2) &&
//           s_AppendLine(iArea, mX, mZ[1][:iT-1], <>, <>, sStyle,
//                        iIndex, M_NAN, M_NAN, sKey2);
//     }
    {
      // Reset the key
      s_Fig[F_PLOT][iArea][P_LINE][iLine][L_KEY]= sKey1;

      // For errorbars, gnudraw should receive either
      //   fac, z= sigma
      //   fac, z= sigma+, sigma-
      //   fac=0, z= L, U
      if (iZ == 1)                  // Gnuplot needs x L, x U
        mZ= mY + dFac*<-1; 1> .* vZl[0][:iT-1];
      else if (dFac)
        mZ= mY + dFac*<-1; 1> .* vZl[:1][:iT-1];
      else
        mZ= vZl[:1][:iT-1];

      aLine[L_KEY]= sKey2;
      aLine[L_TYPE][0]= sStyle;
      aLine[L_TYPE][1]= iIndex;

      aLine[L_XYZ][1]= mZ[0][:iT-1];
      s_Fig[F_PLOT][iArea][P_LINE]|= {aLine};
      aLine[L_XYZ][1]= mZ[1][:iT-1];
      s_Fig[F_PLOT][iArea][P_LINE]|= {aLine};
      iLine+= 2;
      s_Fig[F_PLOT][iArea][P_NLINES]+= 2;
    }
  else if (iMode == ZMODE_FAN)
    {
      vQ= vecr(dFac)';
      if ((iZ == 1) && (sizerc(vQ) == 1))
        { // One set of z-values, no quantiles specified
          vQ= range(.05, .95, .025);
          vZl= quann(vQ') .* vZl;
        }
      else if ((iZ == 2) && (sizerc(vQ) == 2))
        {
          // Too few quantiles for plotting; add center line
          vZl= vZl[0][]|0|vZl[1][];
          vQ= vQ[0]~.5~vQ[1];
        }
      else if ((iZ > 1) && (iZ != sizerc(vQ)))
        {
          vQ= range(.05, .95, .9/(iZ-1));
        }

      // Combine Q/Z/Y into a new Y-value
      mYZ= vZl + mY;

      // Get scale running from 255 (at center) to zero (edge)
      vCol= fabs(vQ'-quantiler(vQ, .5));
      vCol-= min(vCol);
      vCol/= max(vCol);
      vCol= 255*(1-0.9*vCol);

      if (sizeof(ava) <= 3)
        iIndex= aLine[L_TYPE][1];       // Get back original color

      // Place into mZ
      mZ= vCol .* ones(1, iT);

      if ((sizerc(vQ) > 2) && s_CheckSize(mX, mYZ, mZ, 4, "DrawZ", &iT))
        {
          // Add third dimension to this line, just zero-level
          aLine[L_XYZ][2]= zeros(mY);

          // Add extra 3-dimensional line, with fan plot
          s_AppendLine(iArea, mX, mYZ', mZ, <>, "pm3d", iIndex, M_NAN,
                       FALSE, "");

          // Plot the fan in the present line
          s_Fig[F_PLOT][iArea][P_LINE][iLine]= s_Fig[F_PLOT][iArea][P_LINE][iLine+1];
          // Plot the simple line in the next one (is placed at end of routine)
          ++iLine;

// Use a legend, at top right, roughly within the graph
          DrawLegend(iArea, .9, 0, FALSE);
          if (sizerc(sKey1))
            aLine[L_KEY]= sKey1;

          // Plot xyz-plot, no contour
          s_Fig[F_PLOT][iArea][P_TYPE][T_XYZ]= TRUE;
          s_Fig[F_PLOT][iArea][P_TYPE][T_CONTOUR]= FALSE;
          s_Fig[F_PLOT][iArea][P_TYPE][T_FAN]= TRUE;

          // Set and reset (if necessary) 3-d settings
//          s_Fig[F_PLOT][iArea][P_SET][S_YRANGE]=
//            {"set yrange [*:*] reverse","set yrange [*:*] noreverse"};
          s_Fig[F_PLOT][iArea][P_LIMITS][1][2]= TRUE; // Reverse y-axis

          // Default bottom left border, and y2-border
          s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][1][]=
//            <TRUE, TRUE, FALSE, FALSE, FALSE>;
            <-1, TRUE, FALSE, FALSE, TRUE>;
          // Note that getting the plot of the bottom border is hard...
          //   The settings indicate that the y and y2-border are used (TRUE at
          //   location [1] and [4]), but that xtics is not-false (-1 at [0])
        }
      else
        oxwarning ("Incorrect number of quantiles specified for ZMODE_FAN...");
    }
  else if (iMode == ZMODE_HILO)
    { // Adapt line, with bars without endpoints or midpoints
      aLine[L_KEY]= sKey1;

      if (iZ == 1)
        {
          mLU= limits(mY|vZl[][:iT-1])[:1][];
          aLine[L_XYZ][1]= meanc(mLU);
          mZ= (mLU[1][]-mLU[0][])/2|aLine[L_XYZ][3];
        }
      else if (dFac)
        mZ= (mY+dFac*<-1; 1>.*vZl[:1][:iT-1])|aLine[L_XYZ][3];
      else
        mZ= vZl[:1][:iT-1]|aLine[L_XYZ][3];
      aLine[L_XYZ][3]= mZ;
      aLine[L_TYPE][:1]= {"yerrorbars", iIndex};
      s_Fig[F_PLOT][iArea][P_SET][S_BAR]=
        {"unset bar", "set bar"};
    }
  else if (iMode == ZMODE_3D)
    { // Add z-component
      aLine[L_KEY]= sKey1;
      aLine[L_XYZ][2]= vZl[][:iT-1];
      aLine[L_TYPE][0]= sStyle;

//       if (sizerc(sZ))
//         DrawText(iArea, sZ, 0, 0, -1, -1, TEXT_ZLABEL);

      // Plot xyz-plot, no contour
      s_Fig[F_PLOT][iArea][P_TYPE][T_XYZ]= TRUE;
      s_Fig[F_PLOT][iArea][P_TYPE][T_CONTOUR]= FALSE;
      s_Fig[F_PLOT][iArea][P_TYPE][T_FAN]= FALSE;
      s_Fig[F_PLOT][iArea][P_TYPE][T_GRID]= FALSE;

    }
  else if (iMode == ZMODE_VALUE)
    {
      if (s_Fig[F_PLOT][iArea][P_TYPE][T_XYZ])
        {
          oxwarning ("ZMODE_VALUE not implemented for 3D graphs");
          return;
        }
      aLine[L_KEY]= sKey1;
      if ((isarray(vZl) && isstring(vZl[0])) || isstring(vZl))
        {
          vZl= array(vZl);
          mYZ= mX[0][:iT-1]|mY[0][:iT-1];
          vOff= dFac * (maxc(mYZ')-minc(mYZ'))/100;
          for (i= 0; i < sizeof(vZl); ++i)
            DrawText(iArea, vZl[i],
                     mYZ[0][i]+vOff[0], mYZ[1][i]+vOff[1]);
        }
      else
        {
          mYZ= mX[0][:iT-1]|mY[0][:iT-1]|vZl[0][:iT-1];
//           if (s_Fig[F_PLOT][iArea][P_TYPE][T_DMY])
//             {
//               mYMD= (dayofcalendar(mX[:iT-1])~timeofday(fmod(mX[:iT-1], 1))');
//               if (vSpec[0]) // Years used
//                 mYZ[0][]= dayofcalendar(mYMD[0][]', mYMD[1][]', mYMD[2][]')';
//               // Add the fraction of seconds into the day
//               mYZ[0][]+= <3600, 60, 1> * mYMD[3:][] / (24*3600);
//             }
          mYZ= deleteifc(mYZ, isdotmissing(mYZ));
          vOff= dFac * (maxc(mYZ')-minc(mYZ'))/100;
          for (i= 0; i < columns(mYZ); ++i)
            DrawText(iArea, sprint(mYZ[2][i]),
                     mYZ[0][i]+vOff[0], mYZ[1][i]+vOff[1]);
        }
    }
  else if (iMode == ZMODE_COLOR)
    {
      // Adapt iIndex, take the original symbol type, not line type
      if (sizeof(ava) < 4)
        iIndex= aLine[L_POINT][0];

      // Set vZ in mExtra part of line, and indicate negative pointtype
      aLine[L_KEY]= sKey1;
      aLine[L_XYZ][3]= aLine[L_XYZ][3]|vZ;
      aLine[L_POINT][0]= iIndex;
      aLine[L_TYPE][1]= M_NAN;   // Missing linetype indicates changing color
    }
  else if (iMode == ZMODE_SYMBOL)
    {
      // Set vZ in mExtra part of line, and indicate negative pointsize
      // Default Ox symbol size is 90
      aLine[L_KEY]= sKey1;
      aLine[L_XYZ][3]= vZ/90|aLine[L_XYZ][3];
      aLine[L_POINT][1]= -1;
    }
  else if (binand(s_Fig[F_GLOB][G_WARN], 1))
    println ("Warning: This ZMODE not implemented yet in DrawZ");

  // Replace the line back in place
  s_Fig[F_PLOT][iArea][P_LINE][iLine]= aLine;

//  print ("Two lines: ", s_Fig[F_PLOT][iArea][P_LINE][iLine:]);
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  DrawXYZ(const iArea, const vX, const vY, const mZ, ...);
**
**      iArea
**          in: int, area index
**      vX
**          in: 1 x k matrix with X variable
**      vY
**          in: 1 x n matrix with Y variable
**      mZ
**          in: n x k matrix with Z variable, heights above XY plane
**                or
**          in: 1 x n = k matrix with Z coordinates for points
**              (X, Y, Z), rough approximation of surface is constructed
**      iMode
**          in: int, type of plot:
**          0 - surface plot only
**          1 - surface plot with contours on ground level
**          2 - contour plot
**      sX
**          in: string, name of X variable (optional argument)
**      sY
**          in: string, name of Y variable (optional argument)
**      sZ
**          in: string, name of Z variable (optional argument)
**      iPalette
**          in: int, palette index, not implemented in GnuDraw (optional
**              argument).
**      iIndex
**          in: int, line index for mesh (optional argument, default is 2).
**
**  No return value.
*/
DrawXYZ(const iArea, const vX, const vY, const mZ, ...)
{
  decl ava, sKeyX, sKeyY, sKeyZ, nSize, iMode, iIndex, ir,
       bGrid, iContour, iX, iY, mZL;

  s_InitFigure();
  if (iArea < 0)
    return;

  ava= va_arglist();
  iMode= (sizeof(ava) > 0) ? ava[0] : 0;
  sKeyX= (sizeof(ava) > 1) ? ava[1] : "";
  sKeyY= (sizeof(ava) > 2) ? ava[2] : "";
  sKeyZ= (sizeof(ava) > 3) ? ava[3] : "";
  // iPalette not used
  iIndex= (sizeof(ava) > 5) ? ava[5] : s_Fig[F_GLOB][G_DEFLINE][0];
  if (isstring(iMode))
    {
      if (binand(s_Fig[F_GLOB][G_WARN], 1))
        println("Warning: Implementation of DrawXYZ changed, fifth argument iMode should\n",
                "be integer");
      ir= iMode; iMode= sKeyY; sKeyY= sKeyX; sKeyX= ir;
    }
  bGrid= (rows(mZ) == 1);
  iContour= iMode > 0 ? 10 : M_NAN;

  mZL= mZ;
  ir= s_CheckSize(vX, vY, mZL, 2, "DrawXYZ", &iX, &iY);
  if (!ir && s_CheckSize(vX, vY, mZL', 2, "DrawXYZ", &iX, &iY))
    {
      // Turn Z around
      mZL= mZL';
      ir= 1;
    }
  if (ir)
    {
      // For compatibility with OxDraw, switch x-y axes
//       mZL= bGrid ? mZ[][:iX-1] : mZ[:iY-1][:iX-1]';
//       s_AppendLine(iArea, vY[0][:iY-1], vX[0][:iX-1], mZL, <>,
//         "lines", iIndex, M_NAN, iContour, "");
      s_AppendLine(iArea, vX[0][:iX-1], vY[0][:iY-1], mZL, <>,
        "lines", iIndex, M_NAN, iContour, "");

      // Plot xyz-plot for modes 0, 1
      s_Fig[F_PLOT][iArea][P_TYPE][T_XYZ]= iMode < 2;
      // Plot contour for modes 1, 2
      s_Fig[F_PLOT][iArea][P_TYPE][T_CONTOUR]= iMode > 0;
      s_Fig[F_PLOT][iArea][P_TYPE][T_FAN]= 0;
      s_Fig[F_PLOT][iArea][P_TYPE][T_GRID]= bGrid;

      // Get z-border for surface plot
      s_Fig[F_PLOT][iArea][P_TICSAXISBORDER][1][2]= iMode < 2;

      // Set the labels; for compatibility with OxDraw: Switch x-y axes
      if (sizeof(sKeyX))
        DrawText(iArea, sKeyX, 0, 0, -1, -1, TEXT_XLABEL);
      if (sizeof(sKeyY))
        DrawText(iArea, sKeyY, 0, 0, -1, -1, TEXT_YLABEL);
      if (sizeof(sKeyZ))
        DrawText(iArea, sKeyZ, 0, 0, -1, -1, TEXT_ZLABEL);
    }

  s_Fig[F_GLOB][G_LASTPLOT]= iArea;
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  SetDraw(const iOption, ...);
**
**      iType
**          in: int, type of adjustment
**      i1 ... i5
**          in: optional arguments
**
**  No return value.
**
**  The expected number of arguments depends on the type of adjustment:
**
**  option          i1           i2             i3            i4           i5
**
**  SET_AXIS        fontsize     step           tick
**  SET_AXISLINE    no X-line    no Y-line      center dates  no small Y
**  SET_BOX         box:0--1     X-grid:0--1    Y-grid:0--1
**  SET_COLOR       lineno:0--15 red:0--255     green:0--255  blue:0--255
**  SET_COLORMODEL  model:0--3
**  SET_FILL        density:0.0--1.0
**  SET_FONT        fontno:0--3  fontsize       enhanced:0--1
**  SET_GRID        color:0--15  type:0--15
**  SET_HISTOGRAM   inside:0--15 outside:0--15
**  SET_JOIN        join:0--1
**  SET_LABDRAWZ    style:0--1
**  SET_LEGEND      boxed:0--1   columns
**  SET_LEGENDHIDE  hide:0--1
**  SET_LINE        lineno:0--15 linetype:0--4  width         on           off
**  SET_MARGIN      left t top
**  SET_PAPERCOLOR  red:0-255    green:0-255    blue:0-255
**  SET_PRINTPAGE   orient:0--1  papertype:0--2 X-size        Y-size
**  SET_SYMBOL      lineno:0--15 symtype:0--4   size
**
**  SET_OUTPUT      0--2 (0: none, 1: x11/win, 2: wxt, 3: qt)
*/
SetDraw(const iOption, ...)
{
  decl ava, d1, d2, d3, d4;

  s_InitFigure();

  ava= va_arglist();
  d1= (sizeof(ava) > 0) ? ava[0] : M_NAN;
  d2= (sizeof(ava) > 1) ? ava[1] : M_NAN;
  d3= (sizeof(ava) > 2) ? ava[2] : M_NAN;
  d4= (sizeof(ava) > 3) ? ava[3] : M_NAN;

  switch_single (iOption)
    {
      case SET_COLORMODEL:
        {
          // println ("Setting colormodel from ", s_Fig[F_GLOB][G_COLORMODEL], " to ", d1);
          s_Fig[F_GLOB][G_COLORMODEL]= d1;
        }
      case SET_FILL:
        s_Fig[F_GLOB][G_DEFFILL]= max(min(d1, 1), 0);
      case SET_FONT:
        {
          s_Fig[F_GLOB][G_DEFFONT]= d1~d2~d3 .> -1
            .? d1~d2~d3
            .: s_Fig[F_GLOB][G_DEFFONT];
        }
      case SET_LABDRAWZ:    // TRUE: oxdraw-style 'a x s' label
                            // FALSE: old-style, separate 'a' and 's' labels
        s_Fig[F_GLOB][G_LABDRAWZ]= d1 != 0;
      case SET_LEGEND:
        {
          s_Fig[F_GLOB][G_LEGEND][5]= d1 == TRUE;
          s_Fig[F_GLOB][G_LEGEND][7]= d2;
        }
      case SET_LEGENDHIDE:
        s_Fig[F_GLOB][G_LEGEND][6]= d1 == TRUE;
      case SET_LINE:
        {
          s_Fig[F_GLOB][G_DEFLINE][0]= (d1 >= PL_FILLBOX) && (d1 < PL_LAST) ? d1 : 2;
          s_Fig[F_GLOB][G_DEFLINE][1]= (d2 >= TP_SOLID) && (d2 < TP_USER) ? d2 : 0;
          s_Fig[F_GLOB][G_DEFLINE][2]= d3 > 0 ? d3 : M_NAN;
        }
      case SET_SYMBOL:
        {
          s_Fig[F_GLOB][G_DEFPOINT]= d2~(d3/90);
    //      s_Fig[F_GLOB][G_DEFLINE][0]= (d1 >= 0) && (d1 <=15) ? d1 : 2;
        }
      case SET_OUTPUT:
        {
          d1= min(max(d1, 0), 3);
          s_Fig[F_GLOB][G_SHOW]= d1 > 0;
          s_Fig[F_GLOB][G_TERM]= {"x11", "x11", "wxt", "qt"}[d1];
          if ((s_Fig[F_GLOB][G_OXWINDOWS] > 0) && (d1 == 1))
            s_Fig[F_GLOB][G_TERM]= "win";
        }
      case SET_JOIN:
        {
          s_Fig[F_GLOB][G_JOIN]= d1 > 0;
        }
      case SET_BOX:
        {
          s_Fig[F_GLOB][G_BOX]= (d1 != 0)~(d2 != 0)~(d3 != 0);
          if (d2 || d3)
            println ("Option of setting a grid through SetDraw() is not implemented yet");
        }
      case SET_PAPERCOLOR:
        {
          s_Fig[F_GLOB][G_PAPERCOLOR]= isstring(d1) ? d1 : d1~d2~d3;
        }
      default: // if (binand(s_Fig[F_GLOB][G_WARN], 1))
        println ("This option ", iOption, " of SetDraw() is not implemented yet");
    }
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  SetDrawWindow(const sTitle);
**
**      sTitle
**          in: string, name of window
**
**  No return value.
**
**  This function is only relevant when interacting with GiveWin otherwise it does nothing. It sets
**  the name of the GiveWin window in which the graphs of the Ox program appear to
**  sTitle.
*/
SetDrawWindow(const sTitle)
{
  decl i, fh, sUser, iInd;

  s_InitFigure();

  if (sizeof(sTitle) && (s_Fig[F_GLOB][G_OXWINDOWS] > 0))
    {
      // Only put title if user did not do so him/herself
      DrawTitle(-1, sTitle);

      if (binand(s_Fig[F_GLOB][G_WARN], 1) &&
          !binand(s_Fig[F_GLOB][G_WARN], 4))
        {
          println ("Warning: SetDrawWindow has no effect in GnuDraw on non-unix systems");
          s_Fig[F_GLOB][G_WARN]+= 4;
        }
    }
  else
    s_Fig[F_GLOB][G_TITLE][0]= sTitle;
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  SetTextWindow(const sTitle);
**
**      sTitle
**          in: string, name of window
**
**  No return value.
**
**  This function is only relevant when interacting with GiveWin otherwise it does nothing. It sets
**  the name of the GiveWin window in which the output (from the print() function) of the
**  Ox program appears to sTitle.
*/
SetTextWindow(const sTitle)
{
  // No effect
}

/*
**  #include <packages/gnudraw/gnudraw.h>
**  ShowDrawWindow();
**
**  No return value.
**
**  Shows the drawing. Note that in some implementations the graphs cannot be displayed.
**  Then a message is printed (SaveDrawWindow() will still work in that case!). A call to
**  ShowDrawWindow also clears the drawing buffer, so that subsequent graphing starts
**  from an empty sheet.
*/
ShowDrawWindow()
{
  decl sTempName;

  s_InitFigure();
  if ((sizeof(s_Fig) > 0) && (s_Fig[F_NPLOTS] > 0)
      && s_Fig[F_GLOB][G_SHOW])
    SaveDrawWindow(s_TempName());     // Save drawwindow to temporary name
  if (sizeof(s_Fig))
    s_Fig[F_GLOB][G_INITFIGURE]= TRUE;
}

/*
**  DrawDebug()
**
**  Purpose:
**    Only debugging
*/
DrawDebug()
{
  decl iArea, inLines;
  iArea= s_Fig[F_NPLOTS]-1;
  if (iArea < 0)
    return 0;
  inLines= s_Fig[F_PLOT][iArea][P_NLINES];
  println ("Area ", iArea, " had ", inLines, " or ",
    sizeof(s_Fig[F_PLOT][iArea][P_LINE]), " lines");
}
