/*
**  Library file GnuDraw_lib.ox
**
**  Purpose:
**    Contain external routines for drawing plots, with high level of
**    compatibility with Ox drawing routines
**
**  Date:
**    7/5/07
**
**  Author:
**    email: c.s.bos at vu dot nl
**    http://personal.vu.nl/c.s.bos/
**
**  Based on:
**    oxdraw plotting package by Jurgen Doornik.
*/

static decl s_Debug= 0;

/*
**  s_DirUnix2Dos(const asString)
**
**  Purpose:
**    Change a unix filename with forward slashes to windows format
*/
s_DirUnix2Dos(const asString)
{
  decl i;
  for (i= 0; i < sizeof(asString[0]); ++i)
    if (asString[0][i:i]=="/")
      asString[0][i:i]= "\\";
}

/*
**  s_EscapeSpaces(const sIn)
**
**  Purpose:
**    Change a unix dir/filename such that spaces are escaped
**
**  Inputs:
**    sIn      string, possibly with spaces
**
**  Return value:
**    sOut         string, with spaces escaped
*/
s_EscapeSpaces(const sIn)
{
  decl i, sOut;
  i= find(sIn, " ");
  if (i < 0)
    return sIn;

  sOut= "";
  for (i= 0; i < sizeof(sIn); ++i)
    {
      if (sIn[i:i] == " ")
        sOut~= "\\";
      sOut~= sIn[i:i];
    } 
  return sOut;
}

/*
**  s_IsYear(const vY)
**
**  Purpose:
**    Check if vY is a year, not a day-of-calendar. Central location
**    for such a check
**
**  Inputs:
**    vY        double, possibly a year value
**
**  Return value:
**    ir        TRUE if this seems to be a year
*/
s_IsYear(const vY)
{
  return (min(vY) > 1900) && (max(vY) < 2500);
}

/*
**  s_RunGnuPlotLin(const sDirectory, const sBasename, const sExt, const bRemove,
**					const bPersist, const iColor)
**
**  Inputs:
**    sDirectory
**        in: String, directory where file is located
**    sBasename 
**        in: valid file name excluding extension
**    bRemove
**        in: Boolean, TRUE if file should be removed afterwards
**    bPersist
**        in: Boolean, TRUE if file should be displayed until a key is
**            pressed
**    iColor
**        in: Boolean, 0 if color graph should be shown, 1 for gray; links to
**          first option of ADJ_COLORMODEL
**
**  Return value:
**    sCommand      string, command line to run
**
**  Prepares to run GnuPlot on the file, optionally removes the file afterwards,
**    on Linux/Unix/Mac
*/
s_RunGnuPlotLin(const sDirectory, const sBasename, const sExt, const bRemove, 
                const bPersist, const iColor)
{
  decl sCommand;
  
  sCommand= sprint("(cd ", s_EscapeSpaces(sDirectory), " && gnuplot ");
  if (iColor == 1)
    sCommand~= "-gray ";
  if (bPersist)
    {
      sCommand= sprint(sCommand, "-persist ");
// Old: Use canvas size change...      
//       // Change final size
//       if ((aSizes[5] != 1) || (aSizes[6] != 1))
//         sCommand= sprint(sCommand, "-geometry ", double(640*aSizes[5]),
//                          "x", double(480*aSizes[6]), "+0+0 ");
    }                             
  sCommand= sprint(sCommand, s_EscapeSpaces(sBasename), sExt, ".plt > /dev/null");
  #ifndef NODEL
  if (bRemove)
    sCommand= sprint(sCommand, " && rm ", s_EscapeSpaces(sBasename), sExt, ".plt ");
  #endif  
  sCommand= sprint(sCommand, "&& rm ", 
                   s_EscapeSpaces(sBasename), "*.log > /dev/null 2>&1)&");

  return sCommand;
}

/*
**  s_RunGnuPlotWin(const sDirectory, const sBasename, const sExt, 
**	                const bPersist)
**
**  Inputs:
**    sDirectory
**        in: String, directory where file is located
**    sBasename 
**        in: valid file name excluding extension
**    bRemove
**        in: Boolean, TRUE if file should be removed afterwards (not
**            used on Windows though)
**    bPersist
**        in: Boolean, TRUE if file should be displayed until a key is
**            pressed
**
**  Return value:
**    sCommand      string, command line to run
**
**  Prepares to run GnuPlot on the file, optionally removes the file afterwards,
**    on Windows
*/
s_RunGnuPlotWin(const sDirectory, const sBasename, const sExt, const bRemove, 
                const bPersist)
{
  decl ir, sCommand, fh, sOS;

  ir= 1;

  // Prepare a command like
  //   cd dir && ...\wgnuplot.exe --persist gd0.plt && del gd0.plt && del gd0*.log > nul;
  // Note: Original used to be 
  //   cd dir && ...\wgnuplot.exe gd0.plt - && del gd0.plt && del gd0*.log > nul;
  sCommand= sDirectory == "./" ? "" : sprint("cd ", sDirectory, " && ");
  sCommand= sprint(sCommand, s_Fig[F_GLOB][G_GNUPLOTEXEC]);
  if (bPersist)
    sCommand= sprint(sCommand, " --persist");
  sCommand= sprint(sCommand, " ", sBasename, sExt, ".plt");
  if (bRemove)
    sCommand= sprint(sCommand, " && del ", sBasename, sExt, ".plt ");
  sCommand= sprint(sCommand, "&& del ", sBasename, "*.log > nul");

  // Change slashes to backslashes
  s_DirUnix2Dos(&sCommand);
  if (s_Fig[F_GLOB][G_OXWINDOWS] == 2)	  
    sCommand= sprint("start \"GnuDraw\" /min \"cmd /c ", sCommand, "\"");
  else		
    sCommand= sprint("start /m \"command.com /c ", sCommand, "\"");

  return sCommand;
}

/*
**  s_RunGnuPlot(const sDirectory, const sBasename, const sExt, const bRemove,
**				 const bPersist)
**
**  Inputs:
**    sDirectory
**        in: String, directory where file is located
**    sBasename 
**        in: valid file name excluding extension
**    sExt
**        in: extension
**    bRemove
**        in: Boolean, TRUE if file should be removed afterwards
**    bPersist
**        in: Boolean, TRUE if file should be displayed until a key is
**            pressed
**
**  No return value.
**
**  Runs GnuPlot on the file, optionally removes the file afterwards,
**    on Linux/Unix/Mac
*/
s_RunGnuPlot(const sDirectory, const sBasename, const sExt, const bRemove, 
             const bPersist, const iColor)
{
  decl sDirectoryL, sCommand;
  
  sDirectoryL= (sDirectory == "") ? "./" : sDirectory;
  sCommand= (s_Fig[F_GLOB][G_OXWINDOWS] > 0) 
    ? s_RunGnuPlotWin(sDirectoryL, sBasename, sExt, bRemove, bPersist) 
    : s_RunGnuPlotLin(sDirectoryL, sBasename, sExt, bRemove, bPersist, iColor);

  #ifdef GNUDRAW_DEBUG
    println ("\nRunning ", sCommand);
  #endif  
  if (s_Fig[F_GLOB][G_SHOW])
    systemcall(sCommand);
    
  if (bPersist)
    s_Fig[F_GLOB][G_INITFIGURE]= TRUE;
}

/*
**  s_TempName()
**
**  Purpose:
**    Find a temporary file name
**
**  Inputs:
**    s_Fig[F_GLOB]
**
**  Return value:
**    string, with temporary file name, ending either in ".x11" or
**    ".win"
*/
s_TempName()
{
  decl sUser, i, sFilename, sTerm;
  
  // Use a temporary file
  sUser= getenv("USERNAME");
  i= strifind(sUser, " ");
  while (i >= 0)        // Get rid of spaces
	{
	  sUser[i:i]= "_";
      i= strifind(sUser, " ");
	} 
  sTerm= s_Fig[F_GLOB][G_TERM];
  sFilename= sprint("gd_", sUser, "_", s_Fig[F_GLOB][G_TEMPCOUNT], "_",
    imod(today(), 10000), ".", sTerm);
  ++s_Fig[F_GLOB][G_TEMPCOUNT];
 
  return sFilename;
}

/*
**  s_Shading(const vY)
**
**  Purpose:
**    Get shading variable. Shading starts at the first 1, and stops
**    in Ox at the next one, or 0
**
**  Inputs:
**    vY        1 x iT vector with indices of shading regions
**
**  Return value:
**    vI        1 x iT vector with missings for non-shades and 1 for
**              shade
*/
s_Shading(const vY)
{
  decl iT, vI, vJ, i;
  
  iT= sizerc(vY);
  vI= constant(M_NAN, 1, iT);
  vJ= vecrindex((vY .== 1) .&& 
                ((vY .!= lag0(vY', 1, 0)') .|| 
                 (vY .!= lag0(vY', -1, 0)')))|iT;
  for (i= 0; i < sizerc(vJ)-1; i+=2)
    vI[vJ[i]:vJ[i+1]]= 1;    
  return vI;  
}

/*
**  s_ParseFilename(const sFilename, const asBasename, const asOutfile, 
**                  const aiOuttype)
**
**  Purpose:
**    Parse the filename, splitting it up in a base filename, an output
**    filename, and a string indicating the output type
**
**  Output:
**    asDirectory Directory for output file
**    asBasename  Filename without extension, without directory
**    aiOuttype   integer indicating output type
**
**  Return value:
**    1     if filename could be parsed, zero otherwise
**
*/
s_ParseFilename(const sFilename, const asDirectory, const asBasename, 
                const aiOuttype)
{
  decl iBar, ir, il, i, sExt, asOut;

  if (sizeof(sFilename) == 0)
    return 0;

  il= strifindr(sFilename, ".");
  if (il == -1)
    il= sizeof(sFilename);
  // Output type is defined by extension
  sExt= (il < sizeof(sFilename))
    ? strlwr(sFilename[il+1:]) : "plb";
  asOut= {{O_X11, {"x11", "scr"}},      // scr is deprecated...
          {O_WXT, {"wxt"}},
          {O_QT, {"qt"}},
          {O_WIN, {"win", "scr"}},
          {O_PLB, {"plb"}}, 
          {O_EPS, {"eps", "epc", "tkf"}},
          {O_PNG, {"png"}},
          {O_GIF, {"gif"}},
          {O_TEX, {"tex", "aux"}},
          {O_ETEX, {"etex"}},           // epslatex
          {O_SVG, {"svg"}},
          {O_PDF, {"pdf"}},
          {O_FIG, {"fig"}}};

  aiOuttype[0]= -1;     // Search output type indication
  for (i= 0; i < sizeof(asOut); ++i)
    if (strifind(asOut[i][1], sExt) > -1)
      aiOuttype[0]= asOut[i][0];

  if (aiOuttype[0] < 0)
    { // If undefined, use PLB extension
      if (binand(s_Fig[F_GLOB][G_WARN], 1) && !binand(s_Fig[F_GLOB][G_WARN], 2))
        {
          println("Warning in GnuDraw: Extension .plb added to filename ", sFilename);
          s_Fig[F_GLOB][G_WARN]+= 2;
        }  
      aiOuttype[0]= O_PLB;  
    }

  if (sExt == "epc")        // Deprecated
    // Change first plot to color, others follow
    {
      oxwarning ("Deprecated use of .epc");
      s_Fig[F_PLOT][0][P_COLORMODEL][1]= 3;
    }  

  iBar= strifindr(sFilename, "/"); 
  if (iBar == -1)
    iBar= strifindr(sFilename, "\\"); 
  asBasename[0]= sFilename[iBar+1:il-1];
  asDirectory[0]= (iBar > 0) ? sFilename[:iBar-1]~"/" : "";
  
  return 1;
}

/*
**  s_InitOutfile(const sBasename, const sOuttype, const afh)
**
**  Purpose:
**    Initialize the file
**
**  Inputs:
**    sBasename   Base for output file, excluding extension
**    iOuttype    integer indicating output type
**
**  Outputs:
**    afh   Pointer to opened file
**
**  Return value:
**    1 if succesfully opened the file, 0 otherwise
*/
s_InitOutfile(const sBasename, const iOuttype, const afh)
{
  decl sOutfile, iInd, asOuttype, ava, iA, i;

  asOuttype= {"tmp", "tmp", "tmp", ".tmp", "plb", "eps", "png", "gif", "tex",
              "etex", "svg", "pdf", "fig", "missing"};
  sOutfile= sBasename~"."~asOuttype[iOuttype]~".plt";
  if (iOuttype == O_PLB)
    sOutfile= sBasename~".plb";

  if (((iOuttype == O_X11) || (iOuttype == O_WXT) || (iOuttype == O_QT)) && sizeof(s_Fig[F_GLOB][G_TITLE][0]))
    {
      iInd= strifind(s_Fig[F_GLOB][G_TITLE][1], s_Fig[F_GLOB][G_TITLE][0]);
      if (iInd < 0)
        {
          afh[0]= popen_C("gnuplot -persist", "w");
          s_Fig[F_GLOB][G_TITLE][1]~= {s_Fig[F_GLOB][G_TITLE][0]};
          s_Fig[F_GLOB][G_TITLE][2]~= {afh[0]};
        }
      else
        afh[0]= s_Fig[F_GLOB][G_TITLE][2][iInd];
    }
  else
    {
      afh[0]= fopen(sOutfile, "w");
      #ifdef GNUDRAW_DEBUG
        println ("Opening ", sOutfile, " with type ", asOuttype[iOuttype], 
                 " results in ", isfile(afh[0]));
      #endif  
    }  
    
  if (!isfile(afh[0]))
    {
//    if (s_Fig[F_GLOB][G_WARN])
      println ("Warning in GnuDraw: Writing file ", sOutfile, " failed");
      return 0;
    }
      
  if (iOuttype <= O_WIN)
    fprintln(afh[0], "# Plotting file on screen");
  else
    fprintln(afh[0], "# Writing file ", sBasename~"."~asOuttype[iOuttype]);
  fprintln(afh[0], "#   ", date(), ", ", time());
  fprint(afh[0], "# Ox command used:\n#   oxl");
  ava= arglist();
  iA= sizeof(ava);
  for (i= 0; i < iA; ++i)
    fprint(afh[0], " ", ava[i]);
  fprint(afh[0], "\n");    
  fprintln(afh[0], "reset");
  fprintln (afh[0], "set encoding iso_8859_1");   // Not done from s_InitPlot, do it early
  
  return 1;
}

/*
**  s_Capabilities(const avCap, const aasFonts, const iOuttype)
**
**  Purpose:
**    Collect matrix of capabilities of diverse terminals
**
**  Inputs:
**    iOuttype      integer, output type indicator
**
**  Outputs:
**    aasFonts      array of 4 strings, basic font names
**    avCap         1 x 10 vector, with output type and capabilities
**      on enhanced, color/mono, background rgb, title, font, fontsize, x-size, y-size, file
**
**  Return value:
**    none
*/
s_Capabilities(const avCap, const aasFonts, const iOuttype)
{
  decl aasFontsL, mCap, j;
  
  aasFontsL= {{"arial", "times", "courier", "helvetica"}, 
              {"Arial", "Times New Roman", "Courier New", "Helvetica"},
              {"Arial", "Times", "Courier", "Helvetica"},
              {"arial", "times", "cour", "verdana"}, 
              {"arial", "times", "courier", "helvetica"}, 
              {"Helvetica", "Times-Roman", "Courier", "Helvetica"}};
// Old settings, might accidentally have changed...              
//   aasFonts= {{{"arial", "times", "courier", "helvetica"}, 10},     // x11
//     {{"Arial", "Times New Roman", "Courier New", "Helvetica"}, 10}, // win
//     {{"Arial", "Times New Roman", "Courier New", "Helvetica"}, 10}, // wxt
//     {{"Arial", "Times", "Courier", "Helvetica"}, 12},         // plb
//     {{"Arial", "Times", "Courier", "Helvetica"}, 14},         // eps
//     {{"arial", "times", "cour", "verdana"}, 10},              // png
//     {{"arial", "times", "cour", "verdana"}, 10},              // gif
//     {{}, 10},                                                 // tex
//     {{}, 10},                                                 // etex
//     {{"arial", "times", "courier", "helvetica"}, 12},         // svg
//     {{"Helvetica", "Times-Roman", "Courier", "Helvetica"}, 10},     // pdf
//     {{}, 10}};                                                // fig

  // Capabilities: enhanced, color/mono, background rgb, title, font, fontsize, x-size, y-size, file
  mCap= <O_X11, TRUE, FALSE, TRUE, TRUE, 0, 10, 640, 450, FALSE;
         O_WXT, TRUE, FALSE, TRUE, TRUE, 1, 10, 640, 384, FALSE;
         O_QT, TRUE, FALSE, FALSE, TRUE, 1, 10, 640, 480, FALSE;
         O_WIN, TRUE, TRUE, TRUE, TRUE, 1, 10, 640, 480, FALSE;
         O_PLB, TRUE, ., ., ., 2, 12, ., ., FALSE;
         O_EPS, TRUE, TRUE, TRUE, FALSE, 1, 10, 5, 3, TRUE;     // cairo
         O_PNG, TRUE, TRUE, TRUE, FALSE, 1, 10, 640, 480, TRUE; // cairo
         O_GIF, TRUE, TRUE, TRUE, FALSE, 3, 10, 640, 480, TRUE;
         O_TEX, FALSE, TRUE, TRUE, FALSE, ., 10, 5, 3, TRUE;      // pslatex
         O_ETEX, FALSE, TRUE, TRUE, FALSE, ., 10, 5, 3, TRUE;     // epslatex
         O_SVG, TRUE, FALSE, TRUE, FALSE, 4, 12, 640, 480, TRUE; 
         O_PDF, TRUE, TRUE, TRUE, FALSE, 1, 10, 5, 3, TRUE;     // cairo
         O_FIG, FALSE, TRUE, FALSE, FALSE, ., ., 5, -3, TRUE;    // note: No comma in size...
         O_EMF, TRUE, TRUE, TRUE, FALSE, 1, 10, 1024, 768, TRUE>;
          
  if (iOuttype >= rows(mCap))
    oxrunerror ("Outtype too large, incorrect capabilities matrix");
  if (mCap[iOuttype][0] != iOuttype)
    oxwarning ("Incorrect capabilities matrix");

  j= mCap[iOuttype][5];
  if (avCap)
    avCap[0]= mCap[iOuttype][];
  if (aasFonts)
    aasFonts[0]= ismissing(j) ? {} : aasFontsL[j];
}


/*
**  s_GetFontTypeSize(vFont, const iNPlots, const iOutType, const bResize)
**
**  Purpose:
**    Extract font name and size
**
**  Inputs:
**    vFont     vector of size 2 with specified font number and size
**    iNPlots   integer, number of plots
**    iOuttype  integer, type of output
**    bResize   boolean, if TRUE (when this is the global font) use
**              the resizing trick for the font
**
**  Return value:
**    array of size 2 with
**      sFont   string, with font name for terminal
**      iSize   integer, font size for terminal
*/
s_GetFontTypeSize(vFont, const iNPlots, const iOuttype, const bResize)
{
  decl iFac, vCap, asFonts, sFont, iSize;
  
  // In case of PNG, force a fonttype if size is specified
  if ((iOuttype == O_PNG) && (vFont[1] > -1))
    vFont[0]= max(vFont[0], 0);
  if (iOuttype <= O_WIN)        // Show on screen
    vFont[0]= max(vFont[0], 0);

  // In general, force a fonttype if size is specified
  //   Turned off dd 4/1/10
  //   Turned on dd 22/2/10, as else 'greek' fonts are used
  //   On 27/7/10, turned on also if font-type is not specified...
  if (vFont[1] > -1)
    vFont[0]= max(vFont[0], 0);

  // In case of bResize, no fixed font, and a larger number of 
  //   plots, resize the font, except when plb file is written
//   if (bResize && (vFont[1] == -1) && (iNPlots >= 2) && (iOuttype != O_PLB))  
//     {
//       iFac= (iNPlots < 2) ? 1 :
//             iNPlots < 5 ? 13/14:  // 10/14
//             iNPlots < 9 ? 12/14:  //  8/14
//                           10/14;  //  6/14   // Don't go beyond 8 (6/14 was old setting)
//       vFont[1]= 300*iFac;
//       println ("Setting font using fac...");
//     }
                     
  s_Capabilities(&vCap, &asFonts, iOuttype);                     

  sFont= (vFont[0] > -1) && (vFont[0] < sizeof(asFonts))
    ? asFonts[vFont[0]] : "";

  iSize= -1;
  if (vFont[1] > -1)
    { // Adapt the font for the standard fontsize of the terminal
      iSize= vCap[6] * vFont[1] / 300;
      iSize= round(min(max(iSize, 6), 36));
    }  

  return {sFont, iSize};
}

/*
**  s_SetTerm(const fh, const sBasename, 
**            const vSize, const vColorModel, const sTitle, 
**            const vRGB, const iOuttype)
**
**  Purpose:
**    Open the terminal in the output file
**
**  Input:
**    fh   Opened file handle
**    sBasename   Base (without extension) of output file name
**    vSize       vector, size of plot in XSize, YSize, XScale, YScale
**    vColorModel vector of size 2, with colormodel of display (0= bw, 
**                1= colour) and print (0-2: bw and/or gray, 3: color)
**    sTitle      string, title of plot window
**    vRGB        vector of size 3, with RGB colours, or string, or
**                missing
**    iOuttype    integer, output type
**
**  Return value:
**    ir        TRUE if all went well
*/
s_SetTerm(const fh, const sBasename, 
          const vSize, const vColorModel, const sTitle, 
          const vRGB, const iOuttype)
{
  decl sFont, vSizeCanvas, sTerm, vCap, iColorModel, bDash, bCol;
  
  if ((iOuttype < 0) || (iOuttype >= O_LAST))
    {
      oxwarning ("Output type not known");
      return FALSE;
    }  
             
  s_Capabilities(&vCap, 0, iOuttype);
  
  sTerm= "set term ";
  // add terminal
  sTerm~= {"x11", "wxt", "qt", "windows", "plb", "epscairo", "pngcairo", "gif",
           "pslatex", "epslatex", "svg dynamic", "pdfcairo", "fig",
           "emf"}[iOuttype];          
  // add color/mono
  if (vCap[2])
    {
      if (iOuttype <= O_PLB)
        {
          bDash= vColorModel[0] != 0;
          bCol= vColorModel[0] == 0;
        }  
      else
        {
          bDash= vColorModel[1] != <0, 3>;
          bCol= vColorModel[1] != <0, 1>;
        }
      sTerm~= sprint({" monochrome", " color"}[bCol], 
                     {"", " dashed"}[bDash]);
    }      
  // add title
  if (vCap[4] && sizeof(sTitle))
    sTerm~= sprint(" title \"", sTitle, "\"");
  // add size (requested size * canvas size); don't adapt for 
  //   the scale in vSize[2:], to allow for the text title.
  vSizeCanvas= vSize[:1] .* vCap[7:8];
//  if (vCap[])
  sTerm~= sprint(" size ", vSizeCanvas[0], 
              vSizeCanvas[1] > 0 ? "," : " ", fabs(vSizeCanvas[1]));
  // Background RGB
  if (vCap[3] && !ismissing(vRGB))
    sTerm ~= isstring(vRGB) ? sprint(" background \"", vRGB, "\"") 
      : sprint(" background \"#", 
      "%x", vRGB[0], "%x", vRGB[1], "%x", vRGB[2], "\"");

//  fprintln(fh, "reset");
  fprintln(fh, "# Setting terminal and output");
  if (iOuttype != O_PLB)
    fprintln (fh, sTerm);
  else
    // For plb, leave a size command in the file...
    // But it won't work, in combination with the novel
    //   set multiplot layout x,y 
    fprintln (fh, "#set size ", vSize[0], ",", vSize[1]);
  // Prepare output file    
  if (vCap[9])  
    fprintln (fh, "set output '", sBasename, 
      {".x11", ".wxt", ".qt", ".windows", ".plb", ".eps", ".png", ".gif",
       ".tex", ".tex", ".svg", ".pdf", ".fig", ".emf"}[iOuttype], "'");
       
  if (iOuttype == O_WIN)
    fprintln(fh, "unset mouse");
          
  // Prepare for possible missings
  fprintln(fh, "set datafile missing 'NaN'");
  
  return TRUE;
}

/*
**  s_SetColors(const fh, const iColors)
**
**  Purpose:
**    Set the color scheme, where 'classic' is the original gnuplot
**    scheme
**
**  Inputs:
**    fh        file handle, open output file
**    iColors   integer, 0= default, 1= classic, 2= podo
**
**  Return value:
**    None
*/
s_SetColors(const fh, const iColors)
{
  fprintln (fh, "set colors ", {"default", "classic", "podo"}[iColors]);
}

/*
**  s_SetFont(const fh, const vFont, const iNPlots, const iOuttype)
**
**  Purpose:
**    Set the font, if possible/needed
**
**  Inputs:
**    fh        file handle, open output file
**    vFont     vector of 3 integers, with default font type (0) and
**                  size (300), and a boolean indicating enhanced fonts
**    iNPlots   integer, number of plots to make
**    iOuttype    integer, output type
**
**  Return value:
**    None
*/
s_SetFont(const fh, const vFont, const iNPlots, const iOuttype)
{
  decl sFont, iSize, vCap, asFonts;
  
  s_Capabilities(&vCap, &asFonts, iOuttype);
  if (vFont[2] && vCap[1])  
    fprintln(fh, "set termoption enhanced");  
    
  sFont= (vFont[0] > -1) && (vFont[0] < sizeof(asFonts))
    ? asFonts[vFont[0]] : "";

  iSize= -1;
  if (vFont[1] > -1)
    { // Adapt the font for the standard fontsize of the terminal
      iSize= vCap[6] * vFont[1] / 300;
      iSize= round(min(max(iSize, 6), 36));
    }  

  if (sizeof(sFont) || (iSize > -1))
    fprintln(fh, "set termoption font \"",
             sFont, 
             iSize > -1 ? sprint (",", "%.0f", iSize) : "",
             "\"");
}

/*
**  s_CloseOutfile
**
**  Purpose:
**    In case we are working with a true disk file, close it, possibly close
**      the terminal as well
**    If this is a pipe, leave it open.
*/
s_CloseOutfile(const afh, const iNPlots, const sDirectory, const sBasename, 
               const iOuttype)
{
  decl i, iInd, sDirectoryL, asOuttype;
  
  if (!isfile(afh[0]))
    return;
  
  asOuttype= {".tmp", ".tmp", ".tmp", ".tmp", ".plb", ".eps", ".png", ".gif", ".tex",
              ".etex", ".svg", ".pdf", ".fig", ".missing"};
  if (iNPlots > 1)
    fprintln(afh[0], "unset multiplot");

  iInd= strifind(s_Fig[F_GLOB][G_TITLE][1], s_Fig[F_GLOB][G_TITLE][0]);
  if (((iOuttype == O_X11) || (iOuttype == O_WXT) || (iOuttype == O_QT)) && (iInd >= 0))
    { // Pipe, just flush
      #ifdef GNUDRAW_DEBUG
        println ("Flushing file");
      #endif  
        fflush_C(afh[0]);
      return; 
    }
    
  // Dirty trick: On windows, reset the terminal to monochrome at 
  //   the end as well, else it won't work  
  if ((iOuttype == O_WIN) && (s_Fig[F_GLOB][G_COLORMODEL] < 3))
    fprintln(afh[0], "set term windows monochrome");
  if (iOuttype != O_PLB)
    {
      fprintln(afh[0], "set term unknown");
      fprintln(afh[0], "set output");
    }

// Possibly add in code to delete files here
//           fprintln (afh[0], "! rm *.log > /dev/null");  
//           fprintln (afh[0], "! rm ", sBasename, ".plt");  
  afh[0]=fclose(afh[0]);

  if (iOuttype != O_PLB)
    s_RunGnuPlot(sDirectory, sBasename, asOuttype[iOuttype], 
                 iOuttype <= O_WIN,
                 iOuttype <= O_WIN,
                 s_Fig[F_PLOT][0][P_COLORMODEL][0]);
}

/*
**  s_GnuRunOnce()
**
**  Purpose:
**    Decide at runtime, not at compile-time, on the location of files 
**    and choice of operating system.
**
**  Output:
**    s_Fig[F_GLOB][G_GNUPLOTEXEC]     (win) location of gnuplot executable
**    s_Fig[F_GLOB][G_GNUWINDIR]       (win) windows main directory
**    s_Fig[F_GLOB][G_OXWINDOWS]       type of operating system, 
**                            -1=OSX, 0=lin, 1=win95/98, 2= w2k/xp
**    s_Fig[F_GLOB][G_TERM]        terminal, x11, wxt, qt, or win
*/
s_GnuRunOnce()
{
  decl i, j, ir, iOxWindows, sOS, asOptions, sGnuPlot, sGnuWinDir,
  sGnuPlotExec, sCommand, sGnuDrawDir, fh, sUpdir;
  
  [iOxWindows, sOS, sGnuDrawDir, sGnuWinDir, sGnuPlotExec]=
    s_RunTimeEnv();

  s_Fig[F_GLOB]= new array [G_LAST];
  s_Fig[F_GLOB][G_OXWINDOWS]= iOxWindows;
  s_Fig[F_GLOB][G_GNUWINDIR]= sGnuWinDir;
  s_Fig[F_GLOB][G_GNUPLOTEXEC]= sGnuPlotExec;

  s_Fig[F_GLOB][G_COLORMODEL]= 3;  
  s_Fig[F_GLOB][G_INITFIGURE]= 1;
  s_Fig[F_GLOB][G_WARN]= 0;
  s_Fig[F_GLOB][G_LOCATION]= "top left Left reverse";
  s_Fig[F_GLOB][G_TEMPCOUNT]= 0;
  s_Fig[F_GLOB][G_ALIGN]= 0;
  s_Fig[F_GLOB][G_SHOW]= TRUE;
  s_Fig[F_GLOB][G_TITLE]= { "", {}, {}};
  s_Fig[F_GLOB][G_DEFPOINT]= <M_NAN, M_NAN>;
  s_Fig[F_GLOB][G_DEFLINE]= <2, 0, M_NAN>;  // Line index, type (=lines) and width
  s_Fig[F_GLOB][G_DEFFONT]= <-1, -1, 1>;    // Font type and size, enhanced boolean
  s_Fig[F_GLOB][G_DEFFILL]= .5;             // fill density
  s_Fig[F_GLOB][G_LEGEND]= <0, 0, 0, 0, 1, 0, 0, 1, -1>; 
  s_Fig[F_GLOB][G_BOX]= <0, 0, 0>;
  s_Fig[F_GLOB][G_PAPERCOLOR]= M_NAN;
  s_Fig[F_GLOB][G_COLORS]= 1;           // 0=new default, 1=Old style red/green colors, 2= podo
  #ifdef GUI
    s_Fig[F_GLOB][G_LABDRAWZ]= 0;       // Old-style zmode labels
  #else
    s_Fig[F_GLOB][G_LABDRAWZ]= 1;       // Oxdraw style 'a x s' zmode labels
  #endif  
    
  
  if (s_Fig[F_GLOB][G_OXWINDOWS] > 0)
    {
      // Find the directory of the gnudraw
      i= strifind(sGnuDrawDir, "program files");
      j= i + sizeof("program files");
      if (i > -1)
        sGnuDrawDir= sGnuDrawDir[:i-1]~"progra~1"~sGnuDrawDir[j:];

      // Find the windows directory
      if (sizeof(s_Fig[F_GLOB][G_GNUWINDIR]) == 0)
        s_Fig[F_GLOB][G_GNUWINDIR]= "c:\windows";

      // Find the GnuPlot executable        
	  ir= 0;
      sUpdir= oxversion() < 400 ? "..\\..\\..\\" : "..\\..\\..\\..\\";
      asOptions= {s_Fig[F_GLOB][G_GNUPLOTEXEC],
                  "c:\\progra~1\\gnuplot\\bin\\wgnuplot.exe", 
                  "c:\\progra~2\\gnuplot\\bin\\wgnuplot.exe", 
                  sprint(sGnuDrawDir, sUpdir~"gnuplot\\bin\\wgnuplot.exe"),
                  "c:\\progra~1\\gnuplot\\binary\\wgnuplot.exe", 
                  "c:\\progra~2\\gnuplot\\binary\\wgnuplot.exe", 
                  sprint(sGnuDrawDir, sUpdir~"gnuplot\\binary\\wgnuplot.exe"),
                  sprint(sGnuDrawDir, "..\\tsmod4\\wgnuplot.exe"),
                  sprint(sGnuDrawDir, "..\\tsmod40\\wgnuplot.exe"),
                  sprint(sGnuDrawDir, "..\\tsmod32\\wgnuplot.exe")};
    
      sGnuPlot= "wgnuplot.exe";
      for (i= 0; i < sizeof(asOptions); ++i)
        {
          fh= sizeof(asOptions[i]) ? fopen(asOptions[i], "r") : 0;
          if (isfile(fh))
            {
              sGnuPlot= asOptions[i];
              fh= fclose(fh);
              i= sizeof(asOptions);
			  ir= 1;
            }
        }

      sGnuPlot= strlwr(sGnuPlot);
      i= strfind(sGnuPlot, "program files");
      j= i + sizeof("program files");
      if (i > -1)
        sGnuPlot= sGnuPlot[:i-1]~"progra~1"~sGnuPlot[j:];
    
	  s_Fig[F_GLOB][G_GNUPLOTEXEC]= sGnuPlot;
      s_Fig[F_GLOB][G_OXWINDOWS]= (sOS == "Windows_NT") ? 2 : 1;
      s_Fig[F_GLOB][G_TERM]= "windows";
    }
  else
    {
      // s_Fig[F_GLOB][G_OXWINDOWS]= 0;
      s_Fig[F_GLOB][G_TERM]= "x11";
      if (s_Fig[F_GLOB][G_OXWINDOWS] < 0)
        // Turn Enhanced plotting off on OSX
        s_Fig[F_GLOB][G_DEFFONT][2]= 0;
    }    
    
  #ifdef GNUDRAW_DEBUG
    println ("OxWindows: ", s_Fig[F_GLOB][G_OXWINDOWS], ", term: ", s_Fig[F_GLOB][G_TERM],
             ", GnuWinDir: ", s_Fig[F_GLOB][G_GNUWINDIR], ", GnuPlotExec: ", s_Fig[F_GLOB][G_GNUPLOTEXEC],
             ", sOs: ", sOS);  
  #endif  
}  

/*
**  s_InitFigure()
**
**  Purpose:
**    Initialize the figure
*/
s_InitFigure()
{
  if (sizeof(s_Fig) == 0)
    {
      s_Fig= new array[F_LAST];
      s_GnuRunOnce();
    }
        
  if (s_Fig[F_GLOB][G_INITFIGURE])
    {
      // Indicate that no figure has been initialized yet
      s_Fig[F_PLOT]= {};
      s_Fig[F_NPLOTS]= 0;
      s_Fig[F_SIZES]= ones(1, 4);
      s_Fig[F_GLOB][G_ALIGN]= 0;
      s_Fig[F_GLOB][G_LASTPLOT]= -1;
      s_Fig[F_GLOB][G_INITFIGURE]= FALSE;
      s_Fig[F_GLOB][G_JOIN]= 0;
    }  
}

/*
**  s_InitPlot()
**
**  Purpose:
**    Initialize a new plot
*/
s_InitPlot()
{
  decl nP, aPlot, i, mOld;

  if (sizeof(s_Fig) == 0)
    s_InitFigure();

  // Add a plot
  ++s_Fig[F_NPLOTS];

  // Indicate this plot as the last
  s_Fig[F_GLOB][G_LASTPLOT]= s_Fig[F_NPLOTS]-1;

  aPlot= new array [P_LAST];
  aPlot[P_LINE]= {};
  aPlot[P_NLINES]= 0;
  aPlot[P_LEGEND]= s_Fig[F_GLOB][G_LEGEND];
    //  [iX, iY, bCoord, bRight, bReverse,  bBox, bHide, iCol, iFontsize]

  aPlot[P_SET]= new array [S_LAST];
  for (i= 0; i < S_LAST; ++i)
    aPlot[P_SET][i]= {""};
  aPlot[P_SET][S_FORMAT]= {{}, {}, TRUE};

  // Indicate that this probably will be a normal xy plot
  aPlot[P_TYPE]= zeros(1, T_LAST);

  // Indicate that the location is not prespecified
  aPlot[P_LOCATION]= M_NAN;

  // Indicate that the tics should appear on the border, and no 
  //   specification of min, incr, end and incr for minor tics.
  // Contents of rows of P_TICSAXISBORDER is
  //   0: UseZeroAxis
  //   1: UseBorder
  //   2: Tics on Axis?
  //   3-5: min, incr, end of tics
  //   6: incr for minor tics
  //   7: colour grid on this axis
  //   8: line type grid on this axis
  aPlot[P_TICSAXISBORDER]= constant(M_NAN, 9, 5);
  // Default bottom left border (x, y, z, x2, y2)
  aPlot[P_TICSAXISBORDER][1][]= <TRUE, TRUE, FALSE, FALSE, FALSE>;
  // Do not put tics on zeroaxis
  aPlot[P_TICSAXISBORDER][2][]= 0;
  // Indicate no rotation, default font of default size, default tick size
  aPlot[P_AXISLABEL]= <0, -1, -1, -1>;
  // Limits on the x, y, z, x2, y2, z2-axis of this plot; initially at 
  //   infinity, indicating that they don't have to be set. Last 
  //   column indicates whether the axis should be reversed
  aPlot[P_LIMITS]= constant(M_INF, 6, 2)~M_NAN;
  // No time series plot
  aPlot[P_DATESPEC]= zeros(1, 6);
  aPlot[P_AXIS]= 0;     // By default, plot on x-axis
  aPlot[P_LABEL]= {{}, <>};    // No labels defined
  // Scaling indicators: Originally a standard axis, scale of one, shift
  //   0, and no format specified in DrawTMatrix
  aPlot[P_AXISSCALE]= AXIS_LINEAR~ones(6, 1)~zeros(6,2);
  aPlot[P_3DVIEW]= <60, 120, M_NAN, M_NAN>;
  aPlot[P_PAPERCOLOR]= M_NAN;
  aPlot[P_COLORMODEL]= 0~s_Fig[F_GLOB][G_COLORMODEL];
  aPlot[P_AXISTEXT]= {M_NAN, M_NAN, M_NAN, M_NAN, M_NAN};

  if (s_Fig[F_NPLOTS] > 1)
    {
      mOld= s_Fig[F_PLOT][s_Fig[F_NPLOTS]-2][P_TICSAXISBORDER];
      // Clear out zero axes
      aPlot[P_TICSAXISBORDER][0][]= (mOld[0][].== 1) .? 0 .: M_NAN;
      // If all min, incr and end are missing, don't reset range
      aPlot[P_TICSAXISBORDER][4][]= minc(isdotnan(mOld[3:5][])) 
        .? M_NAN .: -1;
      // If minor increment missing, don't reset range
      aPlot[P_TICSAXISBORDER][6][]= isdotnan(mOld[6][]) 
        .? M_NAN .: -1;
    }    

  s_Fig[F_PLOT] |= {aPlot};
  if (binand(s_Fig[F_GLOB][G_WARN], 1) && 
      (sizeof(s_Fig[F_PLOT]) != s_Fig[F_NPLOTS]))
    println ("Warning in InitPlot: nFigs <> size(Plots)", 
             sizeof(s_Fig[F_PLOT])~s_Fig[F_NPLOTS]);

  // Do not force a common encoding, to escape utf-8 errors, 
  // but do it earlier...
  // DrawAdjust(ADJ_ENCODING, "iso_8859_1");
}

/*
**  s_SetMultPlot(const aaPlot, const iPlot, const iNPlots, 
**                const vSize, const bEmpty)
**
**  Purpose:
**    Prepare the file for the next element of the multiplot
**
**  Inputs:
**    aaPlot    address, a plot to alter
**    iPlot     Integer, number of plot
**    iNPlots   Integer, total number of plots
**    vSize     1 x 4 vector with XSize, YSize, XScale, YScale
**    bEmpty    boolean, TRUE if one of the plots is empty
**              (in which case plot locations are indicated fully,
**              instead of using the newer multiplot layout)
**  
**  Outputs:
**    aaPlot    address, plot with altered settings
**
**  Return value:
**    1 if this plot had its location specified, 0 otherwise
*/
s_SetMultPlot(const aaPlot, const iPlot, const iNPlots, 
              const vSize, const bEmpty)
{
  decl nx, ny, dSx, dSy, vLocation, ir, sLayout;
  
  // Default setting
  ny= max(ceil(sqrt(iNPlots)), 1);
  nx= ceil(iNPlots/ny);
  
  if (sizerc(s_Fig[F_GLOB][G_ALIGN]) == 2)
    {
      [nx, ny]= {s_Fig[F_GLOB][G_ALIGN][0], s_Fig[F_GLOB][G_ALIGN][1]};

      nx= nx >= 1 ? nx : ceil(iNPlots/ny);
      ny= ny >= 1 ? ny : ceil(iNPlots/nx);
    }    

  ir= 0;
  if ((iPlot == 0) && (iNPlots > 1))
    { // Set multiplot layout, in general for multiple panels
      sLayout= sprint("set multiplot layout ", ny, ",", nx);
      aaPlot[0][P_SET][S_MULTIPLOT]= 
        {sLayout, ""};
    }            
  
  vLocation= aaPlot[0][P_LOCATION];
  if (isnan(vLocation) && !(vSize[2:3] == 1) || bEmpty)
    { // If scaling is requested, fill the location variable,
      //   effectively overruling the default multiplot layout
      vLocation= zeros(1, 4);
      vLocation[2:3]= 1.0 ./ (nx~ny);    // size= scale/n
      vLocation[0]= imod(iPlot, nx)/nx;         // origin
      vLocation[1]= (ny-idiv(iPlot, nx)-1)/ny;
    }
  
  if (!isnan(vLocation))
    {
      ir= 1;
      dSx= vSize[2];
      dSy= vSize[3];
      aaPlot[0][P_SET][S_SIZE]= 
          {sprint("set size ", "%4.3f", dSx*vLocation[2], ",", 
                               "%4.3f", dSy*vLocation[3]), "", 1};
      aaPlot[0][P_SET][S_ORIGIN]= 
          {sprint("set origin ", "%4.3f", dSx*vLocation[0], ",", 
                                  "%4.3f", dSy*vLocation[1]), ""};
    }
  
  return ir;  
}

/*
**  s_AppendLine(const iArea, const mX, const mY, const mZ, 
**               const vExtra, const sLineStyle, 
**               const iLineType, const iLineWidth, const iContour, const sKey)
**
**  Purpose:
**    Create info on a line in appropriate array
**
**  Inputs:
**    iArea       integer, area number
**    mX          1 x n vector with x coordinates, or 3 x n matrix when
**                linetype is yerrorbars
**    mY          1 x n vector with y coordinates, or 3 x n matrix when
**                linetype is xerrorbars
**    mZ          matrix or column of z coordinates
**    mExtra      iX x n vector with yerrorbars, (RGB) color numbers
**                between 0 and 2^24 or symbol sizes
**    sLineStyle  string with style of line (lines, points etc)
**    iLineType   integer with type (=color) of line
**    iLineWidth  integer with width of line; if missing, standard width
**                is used
**    iContour    integer, number of contour lines (<=0 or missing is default)
**    sKey        string with key
**    bFront      (optional) boolean, line is placed upfront
**
**  Output:
**    aaLine      Pointer to array of line type
**
**  Return value
**    1 if succeeded
*/
s_AppendLine(const iArea, mX, mY, mZ, const mExtra,
             const sLineStyle, const iLineType, const iLineWidth, 
             const iContour, const sKey, ...)
{
  decl ir, iAreaL, aLine, vSel, iLineWidthL, ava, bFront; 

  ava= va_arglist();
  bFront= sizeof(ava) && ava[0];
  
  while ((sizeof(s_Fig) < F_LAST) || (s_Fig[F_NPLOTS] <= iArea))
    s_InitPlot();
    
  ir= 1; 
  iAreaL= max(iArea, 0);

  // Use global linewidth
  iLineWidthL= ismissing(iLineWidth) ? s_Fig[F_GLOB][G_DEFLINE][2] : iLineWidth;
//  println ("Using lw ", iLineWidthL);

  aLine= new array[L_LAST];
  aLine[L_XYZ]= {mX, mY, mZ, mExtra};
  aLine[L_TYPE]= {sLineStyle, iLineType, iLineWidthL, matrix(iContour)};
  aLine[L_KEY]= sKey;
  aLine[L_AXES]= <0, 0, 0>;        // x, y, z axes, if true x2, y2 axes (note that z2 axis is not really possible)
  aLine[L_POINT]= double(iLineType)~s_Fig[F_GLOB][G_DEFPOINT][1];
//   aLine[L_POINT]= iLineType~M_NAN;
//   aLine[L_POINT]= s_Fig[F_GLOB][G_DEFPOINT];
  // Get rid of center points at errorbars
  if ((sLineStyle == "yerrorbars") || (sLineStyle == "xerrorbars"))
    aLine[L_POINT][0]= 0;
  // If every point is missing, skip this line  
  aLine[L_SKIP]= sizerc(mZ) == 0 
    ? (sizeof(sKey) == 0) && (sumc(isdotmissing(mX|mY)) > 0)
    : 0;    
  
  if (bFront)
    s_Fig[F_PLOT][iAreaL][P_LINE]= {aLine}~s_Fig[F_PLOT][iAreaL][P_LINE];
  else
    s_Fig[F_PLOT][iAreaL][P_LINE] |= {aLine};
  ++s_Fig[F_PLOT][iAreaL][P_NLINES];

  if (sizeof(sKey) > 0)
    {
      s_Fig[F_PLOT][iAreaL][P_TYPE][T_USEKEYS]= TRUE;
    }
  if (s_Fig[F_PLOT][iAreaL][P_NLINES] != sizeof(s_Fig[F_PLOT][iAreaL][P_LINE]))
    {
      if (binand(s_Fig[F_GLOB][G_WARN], 1))
        println("Warning in s_AppendLine: Size of plot ", iAreaL, 
                " not equal to number of lines.",
                s_Fig[F_PLOT][iAreaL][P_NLINES]~sizeof(s_Fig[F_PLOT][iAreaL][P_LINE]));
      return 0;
    }
  
  return ir;
}


/*
**  s_CheckSize(const mX, const mY, const mZ,
**              const iType, const sFunc, const aiN)
**  s_CheckSize(const mX, const mY, const mZ,
**              const iType, const sFunc, const aiN, const aiY)
**
**  Purpose:
**    Check the size of the matrices
**
**  Inputs:
**    mX, mY, mZ    matrices to be plotted
**    iType         int, type of plot
**                    0: x plot
**                    1: x-y plot
**                    2: x-y-z plot
**                    3: x-y-err plot
**                    4: z-fan plot
**
**    sFunc         string, function which is calling
**
**  Output:
**    aiN           int, number of observations to use
**    aiY           int, number of y-variables
**
**  Return value
**    TRUE if there is something to plot
*/
s_CheckSize(const mX, const mY, const mZ,
            const iType, const sFunc, const aiN, ...)
{
  decl va, aiY, iR;
  
  aiN[0]= 0;
  if (iType == 0)
    { // x data, row format
      iR= sizer(mX);
      aiN[0]= min(sizec(mX));
    }
  else if (iType == 1)
    { // x-y
      iR= min(sizer(mX), sizer(mY));
      aiN[0]= min(sizec(mX), sizec(mY));
    } 
  else if (iType == 2)
    { // x-y-z, 
      va= va_arglist();
      if ((sizeof(va) == 0) || !isarray(va[0]))
        return 0;
      aiY= va[0];  
      if (sizer(mZ) == 1)  // row-row-row
        {
          aiN[0]= aiY[0]= min(sizec(mX), sizec(mY), sizec(mZ));
          iR= min(sizer(mX), sizer(mY), aiN[0]-1);
        }  
      else 
        { // row-row-matrix
          aiN[0]= min(sizec(mX), sizec(mZ));
          aiY[0]= min(sizec(mY), sizer(mZ));
          
          iR= min(sizer(mX), sizer(mY), aiN[0]-1, sizec(mY) == sizer(mZ));
        } 
    }
  else if (iType == 3)
    { // x-y-err
      iR= min(sizer(mX), sizer(mY), sizer(mZ));
      aiN[0]= min(sizec(mX), sizec(mY), sizec(mZ));
      // Check that x and y have already been plotted, the
      //   size of Z should not be smaller
      if (aiN[0] != min(sizec(mX), sizec(mY)))
        aiN[0]= 0;
    } 
  else if (iType == 4)
    {
      iR= min(sizer(mX), sizer(mY), sizer(mZ));
      aiN[0]= min(sizec(mX), sizec(mY), sizec(mZ));
      if (aiN[0] != max(sizec(mX), sizec(mY), sizec(mZ)))
        oxwarning (sprint("Incorrect number of columns in ", sFunc));
      if (sizer(mY) != sizer(mZ))
        aiN[0]= 0;
    }  
  else
    oxwarning("Type not recognized in CheckSize");
  
  if (iR == 0)
    aiN[0]= 0;  
    
//   if (aiN[0] == 0)
//     println ("CheckSize fails on ", sFunc);
    
  return (aiN[0] > 0);               
}
/*
**  s_CheckDimUni(const mX, const mY, const mErr, 
**                const amlX, const amlY, const sFilename)
**
**  Purpose:
**    Check dimensions for a univariate plot
**
**    Returns two matrices mlX and mlY of the same size
*/
s_CheckDimUni(const mX, const mY, const mErr, 
              const amlX, const amlY, const sFilename)
{
  decl ErrStr, rx, ry, re, cx, cy, ce, ir, mlX, mlY;

  ir= 1;

  ErrStr= "";
  mlX= mX;
  if ((mlX == 0) && (rows(mlX) == 0))
    mlX= range(1, rows(mY))';
  mlY= mY;
  if ((mlY == 0) && (rows(mlY) == 0))
    mlY= range(1, rows(mX))';

  rx= rows(mlX);
  ry= rows(mlY);
  re= rows(mErr);
  cx= columns(mlX);
  cy= columns(mlY);
  ce= columns(mErr);
  
  if (rx != ry)
    ErrStr |= "Rows x should equal rows y\n";
  if (cx != cy)
    if (min(cx, cy) > 1)
      ErrStr |= "x or y should have one column, or the number of columns should be equal\n";
    else if (cx == 1)
      mlX= mlX .* ones(1, cy);
    else if (cy == 1)
      mlY= mlY .* ones(1, cx);

  if (!(mErr == 0))
    {
      if (cy > 1)
        ErrStr |= "With error bars, only one column y allowed\n";
      if (rx != re)
        ErrStr |= "Rows x not equal to number of error bars\n";
      if (ce > 2)
        ErrStr |= "Maximum of two columns allowed for error bars\n";
    }

  if (ErrStr != "")
    {
      if (binand(s_Fig[F_GLOB][G_WARN], 1))
        {
          println ("Error: Measures of matrices incorrect while drawing ", 
                   sFilename);
          println (ErrStr);
          println ("x: ", "%5.0f", rows(mlX), "%5.0f", columns(mlX));
          println ("y: ", "%5.0f", rows(mlY), "%5.0f", columns(mlY));
        }  
      ir= 0;
    }
  if (isarray(amlX))
    amlX[0]= mlX;
  if (isarray(amlY))
    amlY[0]= mlY;

  return ir;
}

/*
**  s_CheckDimBiv(const vX, const vY, const mZ, 
**                const avlX, const avlY, const sFilename)
**
**  Purpose:
**    Check dimensions for a bivariate plot
**
**  Inputs (if correct):
**    vX    k x 1 vector
**    vY    n x 1 vector
**    mZ    n x k matrix
**  or
**    vX    k x 1 vector
**    vY    n x k vector
**    mZ    n x k matrix
**  or
**    vX    k x k vector
**    vY    n x k vector
**    mZ    n x k matrix
**  or
**    vX    n x 1 vector
**    vY    n x 1 vector
**    mZ    n x 1 vector
**
**
*/
s_CheckDimBiv(const vX, const vY, const mZ, 
              const avlX, const avlY, const sFilename)
{
  decl ErrStr, rx, ry, rz, cx, cy, cz, ir;

  ir= 1;
  avlX[0]= vX;
  if ((avlX[0] == 0) && (rows(avlX[0]) == 0))
    avlX[0]= range(1, columns(mZ))';
  avlY[0]= vY;
  if ((avlY[0] == 0) && (rows(avlY[0]) == 0))
    avlY[0]= range(1, rows(mZ))';

  rx= rows(avlX[0]);
  ry= rows(avlY[0]);
  rz= rows(mZ);
  cx= columns(avlX[0]);
  cy= columns(avlY[0]);
  cz= columns(mZ);
  
  ErrStr= "";
  if (ry != rz)
    ErrStr |= "Rows y not equal to rows z\n";
  if ((cx != 1) && (cx != cz))
    ErrStr |= "Columns x should either be 1 or equal the rows of x and the columns of z";  
  if ((cx == 1) && (cy == 1) && (cz == 1))
    {
      if (min(rx, ry, rz) != max(rx, ry, rz))
        ErrStr |= "X, y and z should the same number of rows for non-grid data\n";
    }
  else if (cy == cz)
    {
      if (rx != cz)
        ErrStr |= "Rows x not equal to columns z\n";
    }

  if (ErrStr != "")
    {
      if (binand(s_Fig[F_GLOB][G_WARN], 1))
        {
          println ("Error: Measures of matrices incorrect while drawing ", 
                   sFilename);
          println (ErrStr);
          print ("%r", {"x:", "y:", "z:"}, "%c", {"Rows", "Columns"},
                 "%cf", {"%5.0f", "%5.0f"}, 
                 rx~cx|ry~cy|rz~cz);
        }                 
      ir= 0;
    }

  return ir;
}

/*
**  s_WriteTime(const dTime, const bSec)
**
**  Purpose:
**    Write the time into either 'yy mm dd hh mm ss' format,
**    or in number of seconds
**
**  Inputs:
**    dTime     double, time value
**    bSec      boolean, if TRUE, the number of seconds is printed
**
**  Return value:
**    sTime     String, either 'yy mm dd hh mm ss' or just the number 
**              of seconds
*/
s_WriteTime(const dTime, const bSec)
{
  decl j, sTime, vTime;
  
  if (bSec)
    {
      if (fabs(fmod(dTime, 1) < 1e-6))
        sTime= sprint("%i", dTime, "*24*60*60");
      else
        sTime= sprint("%i", dTime*24*60*60);
    }  
  else    
    {
      vTime= dayofcalendar(dTime)~timeofday(fmod(dTime, 1));
      sTime= "'";
      for (j= 0; j < sizerc(vTime); ++j)
        sTime~= sprint("%i ", vTime[j]);
      sTime~= "'";
    }
  return sTime;      
}

/*
**  s_DateSpec(const avL, const avSpec, const aaPlot)
**
**  Purpose:
**    Read off the date specification for a plot, for all possible 
**    x1, x2, y1, y2 axes
**
**  Inputs:
**    aaPlot
**
**  Outputs:
**    avL       6 x 1 vector, with length (in days) of x1, y1, z1, x2, y2,
**              z2 axes. Note that z/z2-axis by definition is not in days.
**    avSpec    1 x 6 matrix of booleans, indicating usage of years,
**              months, days, hours, minutes and seconds, for all axes.
**
**  Return value:
**    ir        integer, number of date axes
*/
s_DateSpec(const avL, const avSpec, const aaPlot)
{
  decl mScale, vD1, vDates, iDMY, iN, i, j, iAxis;
  
  avL[0]= zeros(6, 1);
  avSpec[0]= zeros(1, 6);
  aaPlot[0][P_TYPE][T_DMY]= 0;

  mScale= aaPlot[0][P_AXISSCALE];
  
  iN= aaPlot[0][P_NLINES];
  iDMY= 0;
  vDates= <>;
  for (iAxis= 0; iAxis < 6; ++iAxis)
    if (mScale[iAxis][0] == AXIS_DATE)
      {
        iDMY= binor(iDMY, 2^iAxis);

        vD1= <>;
        for (i= 0; i < iN; ++i)       // check all lines
          {  
            j= imod(iAxis, 3);        // Either x, y, z
            // Check whether this line is plotted on this specific axis
            if (idiv(iAxis, 3) != aaPlot[0][P_LINE][i][L_AXES][j])
              continue;

            vD1~= aaPlot[0][P_LINE][i][L_XYZ][j];
          }
        avL[0][iAxis]= sizerc(vD1) ? max(vD1)-vD1[0] : M_NAN;    
        vDates~= vD1;
      }      

  if (iDMY)  
    {
      // DateSpec is slightly too large; too often minutes are used.
      // If true day-of-calendars are used, including year, then use
      //   them all  
      if (vDates > dayofcalendar(1, 1, 1))
        avSpec[0][:2]= 1;
      else // else only take the varying ones
        avSpec[0][:2]= varc(dayofcalendar(vDates)) .> 0;  

      // If there is some variability in timing, use them all
      if (!(vDates == floor(vDates)))
        avSpec[0][3:5]= 1;

      // Get rid of excessive tics
      // 21/10/10: No clue why I did this...
      // aaPlot[0][P_TICSAXISBORDER][6][0]= -1;
      //            isnan(imxTics) ? -1 : imxTics;
    }

  aaPlot[0][P_TYPE][T_DMY]= iDMY;
  
  return sumc(mScale[][0] .== AXIS_DATE);  
}

/*
**  s_SetType(const avType, const iType, const iAxis)
**
**  Purpose:
**    Set the type of the axis
**
**  Inputs:
**    avType    T_LAST-vector with type information
**    iType     either AXIS_LINEAR standard axis, AXIS_LOG log-scale (data
**              is in natural logarithms), AXIS_LOG10 log10-scale (data is
**              in base-10 logarithms), AXIS_SCALED scaled: set scale and
**              shift, plotting C'= aC+b, AXIS_DATE dated: interpret as
**              Julian date/time values.
**
**    iAxis     0-4, for x, y, z, x2, y2 axis
**
**  Output:
**    avType    specific axis was set to the corresponding type
**
**  Return value:
**    None
*/
s_SetType(const avType, const iType, const iAxis)
{
  decl j, iBit;
  
  iBit= 2^iAxis;
  for (j= T_DMY; j <= T_LOG10; ++j)
    { // Turn on this axis
      avType[0][j]= binand(avType[0][j], bincomp(iBit));
    }
  switch_single (iType)
    { 
      case AXIS_LOG:
        j= T_LOG;
      case AXIS_LOG10:
        j= T_LOG10;
      case AXIS_DATE:
        j= T_DMY;
      default:
        j= -1;
    }

  if (j > -1)    
    avType[0][j]= binor(avType[0][j], iBit);
}


/*
**  s_SetKey(const aaPlot)
**
**  Purpose:
**    Set the key in its place
**
**  Inputs:
**    aaPlot        Pointer to plot
**
**  Outputs:
**    aaPlot        Pointer to plot, with settings for key altered
**
**  Return value:
**    None
**
**  Note:
**    Coordinates [iX, iY] run between 0-1, with [0,0] being top left.
**    In GnuPlot, [0,0] is bottom left.
*/
s_SetKey(const aaPlot)
{
  decl vLegend, sLoc, iX, iY, bCoord, bRight, bReverse, bBox, bHide;
  
  vLegend= aaPlot[0][P_LEGEND];
  [iX, iY, bCoord, bRight, bReverse, bBox, bHide]=
    {vLegend[0], vLegend[1], vLegend[2], vLegend[3], vLegend[4],
     vLegend[5], vLegend[6]};
  switch_single (bHide)
    {
      case 1: 
        sLoc= "set key off";
      case -1: 
        sLoc= "set key outside";
      default:  
        {
          sLoc= "set key ";
          if (bCoord)
            sLoc= sprint(sLoc, "at graph ", "%4.2f,", iX, "%4.2f ", iY);
          else
            sLoc= sprint(sLoc, "inside ", (iY == 1) .? "bottom " .: "top ", 
                                 (iX == 1) .? "right " .: "left ");
          sLoc= sprint(sLoc, bRight ? "Right " : "Left ",
                       bReverse ? "reverse " : "noreverse ",
                       bBox ? "box " : "nobox ");
        }
    }                               
  aaPlot[0][P_SET][S_KEY]= {sLoc, ""};   
}

/*
**  s_SetRange(const aaPlot, const vSpec)
**
**  Purpose:
**    Prepare the range over which the plot runs. Also resets
**    ranges that were used before, and returns a vector with 
**    ranges to reset next round.
**
**  Inputs:
**    aaPlot    Pointer to plot, including range information
**    vSpec     vector of size 6, indicating use of YY, MM, DD, hh, mm, ss
**
**  Output:
**    aaPlot    Pointer to plot, with P_SET including range information
*/
s_SetRange(const aaPlot, const vSpec)
{
  decl iAxis, j, k, dX, vTime, pl, pl2, vSel, bTime, vR;
  
//   println ("Getting into setrange", rows(aaPlot[0][P_LIMITS]));
  // Set the range of the plot
  for (iAxis= 0; iAxis < 5; ++iAxis)
    // Range is initialised at infinity; if unchanged, no reason 
    //   to print it.
    if ((sumr(isdotinf(aaPlot[0][P_LIMITS][iAxis][:1])) != 2) || 
        aaPlot[0][P_LIMITS][iAxis][2])      // Reverse
      { // Set range
        bTime= aaPlot[0][P_AXISSCALE][iAxis][0] == AXIS_DATE;
        pl= sprint("set ", {"x", "y", "z", "x2", "y2"}[iAxis], "range [");
        pl2= pl~"*:*]";
        for (j= 0; j < 2; ++j)
          {            
            dX= aaPlot[0][P_LIMITS][iAxis][j];
            if (ismissing(dX))
              pl= sprint(pl, "*");
            else if (!bTime)
              pl= sprint(pl, dX);
            else
              {
                vSel= vecrindex(vSpec);
                pl= pl~"'";
                vTime= dayofcalendar(dX)~timeofday(fmod(dX, 1));
                vTime[5]+= vTime[6]/100;
                vTime= vTime[vSel];       // Get rid of elements not used
                for (k= 0; k < sizerc(vTime); ++k)
                  pl= sprint(pl, vTime[k], " ");
                pl= pl~"'";
              }
            pl= sprint(pl, j == 0 ? ":" : "]");
            
          }  
        if (aaPlot[0][P_LIMITS][iAxis][2])
          {
            pl~= " reverse";
            pl2~= " noreverse";
          }  
        aaPlot[0][P_SET][S_XRANGE+iAxis]= {pl, pl2};
      }
//     else
//       println ("Not setting range ", {"x", "y", "z", "x2", "y2",
//       "z2"}[iAxis], ", limits: ", aaPlot[0][P_LIMITS][iAxis][]);  
}

/*
**  s_SetAxis(const aaPlot, const sFont, const iSize)
**
**  Purpose:
**    Set the axis parameters
**
**  Inputs:
**    aaPlot    Plot, all settings
**    sFont     string, possible choice of font
**    iFont     integer, font size for this output type
**
**  Outputs:
**    aaPlot    Plot, adapted settings
**
**  Return value:
**    None
*/
s_SetAxis(const aaPlot, const vAxisLabel, const sFont, const iSize)
{
  decl i, j, iL, bSurf, mAxis, vBorder, asAxis, sAxis,
       sTics, vDMY, bDMY, mScale, aAxisText, vC;
  
  bSurf= aaPlot[0][P_TYPE][T_XYZ];
  mAxis= aaPlot[0][P_TICSAXISBORDER];
  mScale= aaPlot[0][P_AXISSCALE];
  
  // Set the border
  vBorder= <1; 2; 16; 4; 8>;    // x, y, z, x2, y2 borders
// Note: If ticks are on zero-axis, skip border. But 
//  this can be wrong, if zero-axis is put at the border...  
  vC= (!mAxis[2][] .&& (mAxis[1][] .== 1));
  if (s_Fig[F_GLOB][G_BOX][0] || mAxis[0][2])   // Set full box
    vC[<0, 1, 3, 4>]= TRUE;

  aaPlot[0][P_SET][S_BORDER]= {sprint("set border ", 
    double(vC * vBorder)), ""};

  // Set the standard axes (possibly reset below)
  asAxis= {"x", "y", "z", "x2", "y2"};
  for (i= 0; i < sizeof(asAxis); ++i)
    {
      sAxis= asAxis[i];
      aAxisText= aaPlot[0][P_AXISTEXT][i];
      bDMY= mScale[i][0] == AXIS_DATE;
      
      if (bDMY)
        { // Replace time axes; code moved here from DrawAxis()
          mAxis[4][i]*= G_GD_DAY;
          j= mAxis[3][i];       // First large tick, replace by date if needed
          if (s_IsYear(j))
            mAxis[3][i]= dayofcalendar(floor(j), floor(fmod(j, 1)*12)+1, 1);
        }
      
      if (sAxis != "z")
        {
          if (mAxis[0][i] >= 1)
            aaPlot[0][P_SET][S_XZEROAXIS+i]=
              {sprint("set ", sAxis, "zeroaxis lt ", mAxis[0][i]-2), 
               sprint("set no", sAxis, "zeroaxis")};
          else if (mAxis[0][i] == 0)
            aaPlot[0][P_SET][S_XZEROAXIS+i]=
              {sprint("set no", sAxis, "zeroaxis"), ""};
        }    
        
      if ((sAxis != "z") || bSurf)
        { 
          if ((mAxis[0][i] && mAxis[2][i]) || mAxis[1][i])
            {        
              sTics= sprint("set ", sAxis, "tics ",
                            mAxis[2][i] ? "axis " : "border ",
                            "nomirror ");
              if (vAxisLabel[3] > -1)
                sTics~= sprint("scale ", vAxisLabel[3], " ");                            
              if (!ismissing(vAxisLabel[0]) && (vAxisLabel[0] != 0))
                sTics~= sprint("rotate by ", vAxisLabel[0], " ");
              
              if ((sizeof(aAxisText) == 2) 
                    && (sizeof(aAxisText[0]) > 0))
                { // Place text labels on the axis
                  iL= sizeof(aAxisText[0]);
                  sTics~= "(";
                  for (j= 0; j < iL; ++j)
                    sTics~= sprint(j > 0 ? ", " : "", "\"", aAxisText[0][j], "\" ",
                      bDMY ? s_WriteTime(aAxisText[1][j], 0) : aAxisText[1][j]);
                  sTics~= ")";
                }              
              else if ((!ismissing(mAxis[5][i]) && ismissing(mAxis[3:4][i]))
                // . , ., End   (end without start + incr not allowed)
                // Start, .     (start without incr not allowed)
                || (!ismissing(mAxis[3][i]) && ismissing(mAxis[4][i])))
                sTics= sprint(sTics, "autofreq  # incorrect tic specification ",
                  mAxis[3][i], ",", mAxis[4][i], ",", mAxis[5][i]);
              else if ((mAxis[4][i] <= 0) || 
                       (isdotnan(mAxis[3:5][i]) == <1; 1; 1>))
                  sTics~= "autofreq";
              else if (!bDMY)
                for (j= 0; (j < 3) && !isdotmissing(mAxis[3+j][i]); ++j)
                  sTics~= sprint(j > 0 ? "," : "", mAxis[3+j][i]);
              else 
                { // Date specification for increment, either
                  //   autofreq 
                  //   | <incr>
                  //   | <start>, <incr> {,<end>}
                  if (!ismissing(mAxis[3:4][i]))
                    {
                      sTics~= sprint(s_WriteTime(mAxis[3][i], FALSE), ",");
                    }    
                  if (!ismissing(mAxis[4][i]))
                    {
                      sTics~= sprint(s_WriteTime(mAxis[4][i], TRUE));
                      if (!ismissing(mAxis[3:5][i]))
                        sTics~= sprint(",", s_WriteTime(mAxis[5][i], FALSE));
                    }
                  else
                    sTics~= "autofreq";      
                }              
              if (sizeof(sFont) || (iSize > 0))
                sTics~= sprint(" font \"",
                  sizeof(sFont) ? sFont : "",
                  iSize > -1 ? sprint (",", "%.0f", iSize) : "",
                  "\"");
            }
          else 
            sTics= sprint("set no", sAxis, "tics");
          aaPlot[0][P_SET][S_XTICS+i]= {sTics, ""};
        }  

      if (!isnan(mAxis[6][i]) && (mAxis[6][i] != -1))
        aaPlot[0][P_SET][S_MXTICS+i]= {sprint("set m", sAxis, "tics ", 
          mAxis[6][i] <= 1 ? "default" : mAxis[6][i]), 
          sprint("unset m", sAxis, "tics")};
          
    }  
}

/*
**  s_SetScale(const aaPlot, const vL, const vSpec)
**
**  Purpose:
**    Prepare the scaling of the graph, including the time axes
**
**  Inputs:
**    aaPlot        array, with a full plot
**    vL            vector of size 6, with length in days of axes
**    vSpec         vector of size 6, indicating use of YY, MM, DD, hh, mm, ss
**
**  Outputs:
**    aaPlot        array, with a full plot, with P_SET adapted
**
**  Return value:
**    none
*/
s_SetScale(const aaPlot, const vL, const vSpec)
{
  decl vScale, asAxis, iAxis, iAxisM, d1, sXYZ, sFormat, i, iN;
  
  iN= aaPlot[0][P_NLINES];
  vScale= aaPlot[0][P_AXISSCALE][][0];
  asAxis= {"x", "y", "z", "x2", "y2"};
  
  aaPlot[0][P_SET][S_AXISSCALE]= {{}, "", TRUE};        // Force reset
  aaPlot[0][P_SET][S_XDATA]= {{}, {}, TRUE};
  for (iAxis= 0; iAxis < 5; ++iAxis)
    if (vScale[iAxis] != AXIS_LINEAR)
      {
        d1= vScale[iAxis];
        sXYZ= asAxis[iAxis];
        iAxisM= imod(iAxis, 3);

        if (d1 == AXIS_LOG)
          {
            aaPlot[0][P_SET][S_AXISSCALE][0]~= 
              sprint("set logscale ", sXYZ, "% 7.4f", exp(1));
            aaPlot[0][P_SET][S_AXISSCALE][1]= 
              "set nologscale";
          }    
        else if (d1 == AXIS_LOG10)
          {
            aaPlot[0][P_SET][S_AXISSCALE][0]~= 
              sprint("set logscale ", sXYZ);
            aaPlot[0][P_SET][S_AXISSCALE][1]= 
              "set nologscale";
          }
        else if (d1 == AXIS_SCALED)
          { 
            oxwarning("Should be treated in the plot statement...");
//             if (sizeof(aaPlot[0][P_LINE][iN-1][L_XYZ]) > iAxisM)
//               {
//                 mXY= aaPlot[0][P_LINE][iN-1][L_XYZ][iAxisM];
//                 dL= columns(mXY);
//                 mXY[][dL-1]= d2*mXY[][dL-1] + d3;
//                 aaPlot[0][P_LINE][iN-1][L_XYZ][iAxisM]= mXY;
//               }  
          }  
        else if (d1 == AXIS_DATE)
          { // Time on current axis          
            aaPlot[0][P_SET][S_XDATA][0]~= 
              sprint("set ", sXYZ, "data time");
            aaPlot[0][P_SET][S_XDATA][1]~= 
              sprint("set ", sXYZ, "data");
          
            // Set format for time in data file           
            sFormat= "'";
            for (i= 0; i < 6; ++i)
              sFormat~= vSpec[i] 
                ? {"%Y ", "%m ", "%d ", "%H ", "%M ", "%S"}[i] : "";
            sFormat~= "'";
            aaPlot[0][P_SET][S_TIMEFMT]= 
              {sprint("set timefmt ", sFormat)};

            // Set format for time on axis
            sFormat= (vL[iAxis] < 5) ? "'%H:%M'" :
                     (vL[iAxis] < 30) ? "'%d %H:%M'" :
                     (vL[iAxis] < 365) ? "'%m/%y'" : 
                     (vL[iAxis] < 5*365) ? "'%Y'" :
                     "'%y'";

            if (!aaPlot[0][P_AXISSCALE][iAxis][3] && 
                !sizeof(aaPlot[0][P_SET][S_FORMAT][0]))
              {
                aaPlot[0][P_SET][S_FORMAT][0]~= 
                  sprint("set format ", sXYZ, " ", sFormat);
                aaPlot[0][P_SET][S_FORMAT][1]~= 
                  sprint("set format ", sXYZ, " '%g'");
              }    
//             else
//               println ("Not setting \"set format ", 
//                 sXYZ, " ", sFormat, "\""); 

            // Get rid of excessive tics
            // 21/10/10: No clue why I did this...
  //        aaPlot[0][P_TICSAXISBORDER][6][0]= -1;
  //            isnan(imxTics) ? -1 : imxTics;
          }
      }
}

/*
**  s_SetSurf(const aaPlot)
**
**  Purpose:
**    Set the surface parameters; also change fan-plot if necessary,
**    adding a third coordinate for simple lines
**
**  Inputs:
**    aaPlot    Plot, including surface information
**
**  Outputs:
**    aaPlot    Plot, adapted settings
**
**  Return value:
**    None
*/
s_SetSurf(const aaPlot)
{
  decl iN, i, iX, nSize, bCont, bFan, bGrid, bLog, iCont, aCont,
  asColor, sColor, iColor, dS, mY, sSet, vView, sView;
  
  if (!aaPlot[0][P_TYPE][T_XYZ])
    return;     // No three-dimensional plot

  bCont= aaPlot[0][P_TYPE][T_CONTOUR];
  bFan= aaPlot[0][P_TYPE][T_FAN];
  bGrid= aaPlot[0][P_TYPE][T_GRID];
  vView= aaPlot[0][P_3DVIEW];
  iN= aaPlot[0][P_NLINES];
  aCont= {};
  for (i= iCont= iX= 0; i < iN; ++i)
    if (sizerc(aaPlot[0][P_LINE][i][L_XYZ][2]))
      {
        if (ismatrix(aaPlot[0][P_LINE][i][L_TYPE][3]) &&
            (sizerc(aaPlot[0][P_LINE][i][L_TYPE][3]) == 1))
          // Number of contours  
          iCont= max(iCont, aaPlot[0][P_LINE][i][L_TYPE][3][0]);
        else if (isarray(aaPlot[0][P_LINE][i][L_TYPE][3]))
          // Contour start, increment, possibly end
          aCont= aaPlot[0][P_LINE][i][L_TYPE][3];
        iX= max(iX, columns(aaPlot[0][P_LINE][i][L_XYZ][2]));
      }  
  
  sView= "set view 180,0,1.3";      // For fan plots
  if (!bFan)
    {
      sView= "set view ";
      for (i= 0; i < 4; ++i)
        sView~= sprint(ismissing(vView[i]) ? "" : vView[i], i < 3 ? "," : "");
    }         
  aaPlot[0][P_SET][S_VIEW]= 
    {sView, ""};
  aaPlot[0][P_SET][S_SURFACE]= 
    {"set surface", "unset surface"};
  aaPlot[0][P_SET][S_CLABEL]= 
    {"unset clabel", ""};
  aaPlot[0][P_SET][S_CONTOUR]= 
    {(bCont ? "" : "un")~"set contour", ""};
  if (bCont)
    {
      sSet= "set cntrparam levels";
      if (sizeof(aCont) == 3)
        sSet~= sprint(" incremental ", aCont[0], ",", aCont[1], 
           isdotmissing(aCont[2]) ? "" : sprint(",", aCont[2]));
      else                  
        sSet~= sprint(" auto ", iCont ? iCont : "");
      aaPlot[0][P_SET][S_CNTRPARAM]= 
        {sSet, "unset cntrparam"};
    }                  
  if (bFan)
    {
      // Find fan-plot colors in use; adapt z-value if necessary
      iColor= 2; 
      for (i= 0; i < iN; ++i)
        {
          if (sizerc(aaPlot[0][P_LINE][i][L_XYZ][2]) == 0)
            // Original 2-dim lines are enlarged with zero third dimension
            aaPlot[0][P_LINE][i][L_XYZ][2]= zeros(aaPlot[0][P_LINE][i][L_XYZ][1]);
          if (aaPlot[0][P_LINE][i][L_TYPE][0] == "pm3d")
            iColor= aaPlot[0][P_LINE][i][L_TYPE][1];
        }
      asColor= {"white", "black", "red", "blue", 
                "sea-green", "purple", "green", "brown", "dark-magenta", 
                "orange", "dark-green", "dark-blue", "black", "grey", 
                "dark-grey", "light-grey"};
// OxDraw only uses 8 colors, instead of the full list for fan-plots               
//      asColor= {"white","dark-gray","dark-red","dark-blue","sea-green","purple","dark-green","brown"};
      if (isstring(iColor))
        sColor= iColor;
      else
        sColor= asColor[fmod(iColor, sizerc(asColor))];                
      
      aaPlot[0][P_SET][S_PM3D]= 
        {"set pm3d explicit at b", "unset pm3d"};
      aaPlot[0][P_SET][S_COLORBOX]= 
        {"unset colorbox", ""};
      sSet= sprint("set palette model RGB defined (0 \"white\", 255 \"", sColor, "\")");
      aaPlot[0][P_SET][S_PALETTE]= 
        {sSet, ""},     // 8/8/10: Not resetting palette, for windows compatibility
//        {sSet, "set palette"},
      aaPlot[0][P_SET][S_HIDDEN3D]= 
        {"unset hidden3d", ""};  
    }  
  else
    {
// Ticslevel is deprecated, starting gnuplot 4.4 it would be xyplane.
// Keep old version for now, until 4.4 settles in.    
//       aaPlot[0][P_SET][S_XYPLANE]= 
//         {"set xyplane relative 0", ""};
      aaPlot[0][P_SET][S_TICSLEVEL]= 
        {"set ticslevel 0", ""};
      aaPlot[0][P_SET][S_HIDDEN3D]= 
        {"set hidden3d", ""};  
    }  
  if (bGrid)
    {
      // Approximate grid using quadratic weighting scheme, 
      // and on average 5 items per cell
      nSize= min(floor(sqrt(iX/5)), 25);
      aaPlot[0][P_SET][S_HIDDEN3D]= 
        {sprint("set dgrid3d ", nSize, ",", nSize, ",2"), ""};
    }    
}

/*
**  s_SetText(const aaPlot, const iOuttype)
**
**  Purpose:
**    Prepare the labels 
**
**  Inputs:
**    aaPlot    Plot, including surface information
**    iOuttype  integer, type of output
**
**  Outputs:
**    aaPlot    Plot, adapted settings
**
**  Return value:
**    none
*/
s_SetText(const aaPlot, const iOuttype)
{
  decl i, j, i0, iS, aLabels, iLabel, sLabel, vYMD, sText, vXYZ, bTim, b3Dim, iJust,
    iTitle, iFont, iSize, iRotation, sFont, iFSize;

  aLabels= aaPlot[0][P_LABEL];
  bTim= aaPlot[0][P_TYPE][T_DMY];
  b3Dim= ((sizeof(aaPlot[P_LINE]) > 0) && 
          (sizeof(aaPlot[P_LINE][0]) > 0) && 
          (sizeof(aaPlot[P_LINE][0][L_XYZ]) == 3));
  iS= sizeof(aLabels[0]);  

  aaPlot[0][P_SET][S_LABEL+TEXT_TEXT]= {{}, "", TRUE};        // Force reset
  for (i= TEXT_TEXT+1; i <= TEXT_ZLABEL; ++i)
    aaPlot[0][P_SET][S_LABEL+i]= {{}, ""};
      
  for (i= 0; i < iS; ++i)
    {
      sText= aLabels[0][i];
      [sFont, iFSize]= s_GetFontTypeSize(aLabels[1][i][3:4], 1, iOuttype, FALSE);
      [vXYZ, iTitle, iJust, iRotation]= {aLabels[1][i][0:2],
        aLabels[1][i][5], aLabels[1][i][6], aLabels[1][i][7]};

      // Take care with limits of iTitle  
      iLabel= (iTitle < TEXT_TEXT) || (iTitle > TEXT_ZLABEL) ? 0 : iTitle;
      sLabel= "set ";
      sLabel~= {"label", "title", "xlabel", "ylabel", "zlabel"}[iLabel];
      aaPlot[0][P_SET][S_LABEL+iLabel][1]= "un"~sLabel;  // Reset label later

      if (iFSize > 0)
        sLabel~= sprint (" '{/=", round(iFSize), " ", sText, "}'");
      else
        sLabel~= sprint (" '", sText, "'");
      if (!(iTitle != <TEXT_TEXT, TEXT_AREA, TEXT_PANEL>))
        { // Check coordinate type
          sLabel~= sprint (" at ",
                        iTitle == TEXT_AREA ? "graph " : 
                        iTitle == TEXT_PANEL ? "screen " : "");
// Not adapting xyz for sizes; graph is scaled through canvas size.                        
//           if (iTitle == TEXT_PANEL)
//             {
//               vXYZ[0]*= vSize[0]*vSize[2];
//               vXYZ[1]*= vSize[1]*vSize[3];
//             }  

          i0= 0;                      
          if (bTim && (iTitle == TEXT_TEXT))
            {
              // Check for date in format of 2011.5 instead of 2011/6/1
              if ((vXYZ[0] > 1000) && (vXYZ[0] < 3000)) 
                vXYZ[0]= dayofcalendar(floor(vXYZ[0]), floor(fmod(vXYZ[0], 1)*12)+1, 1);
              sLabel~= s_WriteTime(vXYZ[0], FALSE);
              i0= 1;
            }
          else
            sLabel~= sprint(vXYZ[0]);
          sLabel~= sprint(",", vXYZ[1]);
          if (b3Dim)                                    
            sLabel~= sprint(",", vXYZ[2]);
      
          if (!ismissing(iJust) && (iJust >= -1) && (iJust <= 1))
            sLabel~= sprint({" left", " center", " right"}[iJust+1]);
        }

      if (ismissing(iRotation) && (iTitle == TEXT_YLABEL))
        iRotation= 90;      // Rotate y-label by 90 degrees
      if (!ismissing(iRotation) && (iRotation != 0))
        sLabel~= sprint(" rotate by ", "%.0f", iRotation);
//       if (iFSize > 0)
//         sLabel~= sprint(" font ',", "%.0f", iFSize, "'");
      if (sizeof(sFont) > 0)
        sLabel~= sprint(" font '", sFont, "'");

// Temporary measure (which no longer works with GnuPlot >= 4.2): labels on top
//      sLabel~= " front";
      aaPlot[0][P_SET][S_LABEL+iLabel][0]~= sLabel;

      if (iTitle == TEXT_PANEL)
        { // Scale graph down to allow space for text, if needed
          if (vXYZ[0] >= 1)
            s_Fig[F_SIZES][2]= min(s_Fig[F_SIZES][2], .97/vXYZ[0]);
          if (vXYZ[1] >= 1)
            s_Fig[F_SIZES][3]= min(s_Fig[F_SIZES][3], .97/vXYZ[1]);
        }      
    }
}

/*
**  s_DropMissingLines()
**
**  Purpose:
**    Drop empty lines from the plot
**
**  Inputs:
**    s_Fig     the global figure variable
**
**  Return value:
**    bEmpty    boolean, TRUE if one of the plots is empty
*/
s_DropMissingLines()
{
  decl aPlot, i, j, iNPlots, iN, bEmpty;

  bEmpty= FALSE;
  iNPlots= s_Fig[F_NPLOTS];
  // Get rid of missing lines
  for (i= 0; i < iNPlots; ++i)
    { // Note: Using pointer to plot in this block
      aPlot= &(s_Fig[F_PLOT][i]);
      for (j= 0; j < aPlot[0][P_NLINES]; ++j)
        if (aPlot[0][P_LINE][j][L_SKIP])
          {
            iN= aPlot[0][P_NLINES];
            aPlot[0][P_LINE]= (j > 0 ? aPlot[0][P_LINE][:j-1] : {}) ~
                   (j < iN-1 ? aPlot[0][P_LINE][j+1:] : {});
            --(aPlot[0][P_NLINES]);
            --j;
          }
      bEmpty= bEmpty || (aPlot[0][P_NLINES] == 0);
    }
  return bEmpty;  
}

/*
**  s_WriteSet(const fh, const aSetOld, const aSetNew)
**
**  Purpose:
**    Print the unset-commands for the old plot, and the set-commands
**    for the new plot
**
**  Input:
**    fh        File handle to opened file
**    aSetOld   Array aPlot[P_SET] with set-commands (disregarded)
**              and unset-commands, which are applied as needed.
**    aSetNew   Array aPlot[P_SET] with set-commands 
**              for the new plot. If an option is not set explicitly
**              the unset command (if available) is applied.
*/
s_WriteSet(const fh, const aSetOld, const aSetNew)
{
  decl i, j, aNew, aOld, bForce, bDiff;
  
  #ifdef GNUDRAW_DEBUG
    println ("\nOld settings ", aSetOld);
    println ("\nNew settings ", aSetNew);
  #endif  

  // Set settings, including whether this is time data
  for (i= 0; i < S_LAST; ++i)
    {
      bForce= bDiff= FALSE;
      if (sizeof(aSetOld) > i)
        {
          bDiff= (sizeof(aSetOld[i]) > 1) && (sizeof(aSetNew[i]) > 0) &&  // Old and new are defined 
                 (sizeof(aSetOld[i][0]) > 0) && sizeof(aSetOld[i][1]) &&  // There is something to reset
                 !((sizeof(aSetNew[i][0]) > 0) &&                         // And old value doesn't happen to be new value
                   (array(aSetNew[i][0])[0] == array(aSetOld[i][0])[0]));
          bForce= (sizeof(aSetOld[i]) == 3) && aSetOld[i][2];
        }
      // Reset only if needed
      aOld= (sizeof(aSetOld) > i) && (sizeof(aSetOld[i]) > 1) 
              ? array(aSetOld[i][1]) : "";  
      if (bForce || bDiff)
        for (j= 0; j < sizeof(aOld); ++j)
          if (sizeof(aOld[j]))
            fprintln(fh, aOld[j]);
          
      // Make sure to have an array  
      aNew= array(aSetNew[i][0]);  
      aOld= sizeof(aSetOld) > i ? array(aSetOld[i][0]) : "";  
      for (j= 0; j < sizeof(aNew); ++j)
        if (sizeof(aNew[j]) && 
            (!isarray(aOld) || (sizeof(aOld) <= j) || bForce || 
             (aNew[j] != aOld[j])))
        {
  #ifdef GNUDRAW_DEBUG
          println (aNew[j]);        
  #endif  
            fprintln(fh, aNew[j]);
        }            
    }
}

/*
**  s_WriteBlock(const fh, const mXYZ, const bDel, const bCol)
**
**  Purpose:
**    Write a block of data to the output file
**
**  Inputs:
**    fh        file handle
**    mXYZ      k x n matrix with coordinates, written out over n lines
**    bDel      boolean, if true, delete missings
**    bCol      boolean, if true write full integer number, for color
*/
s_WriteBlock(const fh, const mXYZ, const bDel, const bCol)
{
  decl i, j, iK, iN, mXYZL; 
  
  mXYZL= bDel ? deletec(mXYZ) : mXYZ;
  if (sumc(isdotmissing(mXYZ)) >= 1)
    mXYZL= constant(M_NAN, rows(mXYZ), 1);

  iK= rows(mXYZL);
  iN= columns(mXYZL);
  // Plot elements with missing values
  fprint(fh, "\n");
  for (j= 0; j < iN; ++j)
    {
      for (i= 0; i < iK; ++i)
        if (ismissing(mXYZL[i][j]))
          fprint(fh, "NaN ");
        else if (bCol && (i > 1))
//          // Color code, z-value, write as hexadecimal
//          fprint(fh, "0x", "%06X", mXYZL[i][j], " ");
          // Don't write hexadecimal, doesn't work on windows
          fprint(fh, "%i", mXYZL[i][j], " ");
        else if ((round(mXYZL[i][j]) == mXYZL[i][j]) && 
                 (fabs(mXYZL[i][j]) < INT_MAX))
          // Special case, for increment in a vector plot, a 
          //   time-plot specifies x-delta in integer seconds
          // Extra check is meant to catch extremely large doubles,
          //   which might seem like integers
          fprint(fh, "%i", mXYZL[i][j], " ");
        else
          fprint(fh, mXYZL[i][j], " ");
      fprint(fh, "\n");
    }   
}

/*
**  s_WriteLines(const fh, const aPlot, const vSpec, const vSkip)
**
**  Purpose:
**    Write the lines in this plot. It mainly puts all data in the file,
**    including the missings.
**
**  Inputs:
**    fh        file handle
**    aPlot     array with the plot
**    vSpec     vector of size 6, indicating use of YY, MM, DD, hh, mm, ss
**    vSkip     1 x iN vector with booleans, indicating if
**              line should be skipped (if it was a contour plot from
**              a table file prepared earlier)
**
**  Return value:
**    None
*/
s_WriteLines(const fh, const aPlot, const vSpec, const vSkip)
{
  decl amXYZ, mXYZ, i, j, k, iRX, iCX, iRY, iCY, iRZ, iCZ, vSel, bDel,
    bCol, vI, dDay;
  
  vSel= vecrindex(vSpec);
  for (i = 0; i < aPlot[P_NLINES]; ++i)
    if (!vSkip[i])
      { 
        amXYZ= aPlot[P_LINE][i][L_XYZ];
        bDel= (aPlot[P_LINE][i][L_TYPE][0] == "points") ||
              (aPlot[P_LINE][i][L_TYPE][0] == "impulses");
        bCol= ismissing(aPlot[P_LINE][i][L_TYPE][1]);      
        [iRX, iCX, iRY, iCY, iRZ, iCZ] = {rows(amXYZ[0]), columns(amXYZ[0]),
          rows(amXYZ[1]), columns(amXYZ[1]), rows(amXYZ[2]), columns(amXYZ[2])};

        for (j= 0; j < 3; ++j)
          {
//            iAxis= j+3*aPlot[P_LINE][i][L_AXES][j];
//            if (aPlot[P_AXISSCALE][iAxis][0] == AXIS_DATE)
           
            // Translate timings of dayofcalendar specifications 
            if (binand(aPlot[P_TYPE][T_DMY], 2^j))
              { 
                if (ismissing(amXYZ[j]))
                  {
                    vI= !isdotmissing(amXYZ[j]);
//                     // Get rid of missings relating to time axis
//                     for (k= 0; k < sizeof(amXYZ); ++k)
//                       if (columns(amXYZ[k]) == sizerc(vI))
//                         {
//                           amXYZ[k]= selectifc(amXYZ[k], vI);    // Get rid of missings 
//                         }                      

                    // Place missings on other axes, when time axis is missing
                    for (k= 0; k < sizeof(amXYZ); ++k)
                      if (columns(amXYZ[k]) == sizerc(vI))
                        amXYZ[k]= vI .* ones(rows(amXYZ[k]), 1) .? amXYZ[k] .: M_NAN;    // Place missings everywhere

                    // Replace missing time by last non-missing
                    dDay= min(amXYZ[j]);                    
                    for (k= 0; k < sizerc(vI); ++k)
                      if (ismissing(amXYZ[j][k]))
                        amXYZ[j][k]= dDay;
                      else
                        dDay= amXYZ[j][k];  
                  }

                // Time series data, each element fully specified
                amXYZ[j]= dayofcalendar(amXYZ[j]')'|timeofday(fmod(amXYZ[j]', 1))';
                amXYZ[j][5][]+= amXYZ[j][6][]/100;
                amXYZ[j]= amXYZ[j][vSel][];    // Get rid of elements not used
              }  
          }  
        // Ugly hack, to get vector time-plots right; used in case
        //   the line type is a vector, and there are dates on the
        //   x-axis, see tests/axis/adjtime_vector.ox
        for (j= 0; j < 3; ++j)
          if ((aPlot[P_LINE][i][L_TYPE][0] == "vector") && 
              (rows(amXYZ[3]) > j) && 
              binand(aPlot[P_TYPE][T_DMY], 2^j))
            amXYZ[3][j][]*= 24*60*60;  // Translate difference in days to seconds...

        if (iRZ == 0)      // && (columns(amXYZ[0]) == columns(amXYZ[1])))
          {
            // Standard plot, XY
//            println ("Standard plot, XY");
            s_WriteBlock(fh, amXYZ[0]|amXYZ[1]|amXYZ[3], bDel, bCol);
          }              
        else if ((iCX == iCY) &&     // x: . x k, y: . x k
                 (iCX == iCZ) &&     // x: . x k, z: . x k
                 (iRZ == 1))         // z: 1 x .
          {
//             println ("Writing standard XYZ");
            // Standard plot, XYZ
            s_WriteBlock(fh, amXYZ[0]|amXYZ[1]|amXYZ[2], bDel, FALSE);
          }            
        else if ((iCX == iCZ) &&     // x: . x k, z: . x k
                 (iCY == iRZ) &&     // y: . x n, z: n x .
                 (iRY == iCZ))       // y: k x ., z: . x k
          { // Single X, multiple Y
//             println ("Writing Single X, multiple Y");
            for (j= 0; j < iCX; ++j)
              {
                mXYZ= (amXYZ[0][][j] .* ones(1, iCY))|
                       amXYZ[1][j][]|amXYZ[2][][j]';
                s_WriteBlock(fh, mXYZ, bDel, FALSE);
              }
          }    
        else if ((iCX == iCZ) &&  // x: . x k, z: . x k
                 (iCY == iRZ) &&  // y: . x n, z: n x .
                 (iRX == iRZ))    // x: n x ., z: n x .
          { // Multiple X, single Y
//             println ("Writing multiple X, single Y");
            for (j= 0; j < iCX; ++j)
              {
                mXYZ= amXYZ[0][j][]|
                      (amXYZ[1][][j] .* ones(1, iCX))|amXYZ[2][j][];
                s_WriteBlock(fh, mXYZ, bDel, FALSE);
              }
          }    
        else if ((iCX == iCZ) &&  // x: 1 x k, z: . x k
                 (iCY == iRZ))    // y: 1 x n, z: n x .
          { // Single X, single Y
//             println ("Writing Single X, single Y");
            for (j= 0; j < iCX; ++j)
              {
                mXYZ= (amXYZ[0][][j] .* ones(1, iCY))|
                       amXYZ[1][][]| amXYZ[2][][j]';
                s_WriteBlock(fh, mXYZ, bDel, FALSE);
              }
          }  
        else
          {
            oxwarning("Didn't cover this case of sizes");
            print ("%c", {"X", "Y", "Z"}, "%r", {"r", "c"},
                   iRX~iRY~iRZ|iCX~iCY~iCZ);
          }
        fprint(fh, "e");
      }     

  fprintln(fh, "");
}

/*
**  s_WriteFilePlot(const fh, const aPlot, const bTab, const sBasename, 
**                  const iLine, const iSize, cons iSpec)
**
**  Purpose:
**    Write the plot statement of the output file, for either a univariate
**    or bivariate graph
**
**  Inputs:
**    fh          file handle of opened output file
**    aPlot       Array with information on a set of lines
**    bTab        boolean, if true the table in the log-file is used with 
**                contour data, else the data is printed
**    sBasename   string, base of filename, for loading contour plot
**                tables
**    iLine       integer, if > -1, just prepare for one line (used
**                  for preparing a contour plot table)
**    iSize       integer, font size
**    iSpec       integer, size of specification of time axis
**
**  Return value:
**    vContour    1 x iN vector with booleans, indicating if this
**                was a contour plot
*/
s_WriteFilePlot(const fh, const aPlot, const bTab, const sBasename,
                const iLine, const iSize, const iSpec)
{
  decl i, j, k, pl, i1, i2, iColor, iLineWidth, iFont, vX, iD, b3Dim,
    bSurf, bCont, iPt, iPs, vContour, sKey, mScale, iAxis, vScale;

  vContour= zeros(1, aPlot[P_NLINES]);
  if (!aPlot[P_NLINES])
    return vContour;

  bSurf= aPlot[P_TYPE][T_XYZ];
  bCont= aPlot[P_TYPE][T_CONTOUR];    
  iFont= aPlot[P_LEGEND][8];
  mScale= aPlot[P_AXISSCALE];
  b3Dim= (bSurf || bCont) && !(bCont && !bSurf && bTab);
  pl = b3Dim ? "splot " : "plot ";
  i1= iLine == -1 ? 0 : iLine;
  i2= iLine == -1 ? aPlot[P_NLINES] : iLine+1;
  for (i= i1; i < i2; ++i)
    {
      // If this is a contour plot without a surface, and tab is true,
      //   we're plotting the log-file
      if (bCont && !bSurf && bTab)
        {
          pl= sprint(pl, "'", sBasename, "_", i, ".log'");
          vContour[i]= TRUE;
        }  
      else
        {
          iD= 1;          
          pl= sprint(pl, "'-' using ");
          
          if (!(mScale[][0] != AXIS_DATE) || 
              !(mScale[][1] == 1) ||
              !(mScale[][2] == 0) || 
              rows(aPlot[P_LINE][i][L_XYZ][3]))
            {
              for (j= 0; j < 3; ++j)
                {
                  if ((j == 2) && (!b3Dim))
                    continue;

                  if (j > 0)
                    pl~= ":";  

                  iAxis= j+3*aPlot[P_LINE][i][L_AXES][j]; // first or second axis
                  vScale= mScale[iAxis][1:2];
                  if (vScale != <1, 0>)  // Translated scale
                    pl~= sprint("(", "%g", vScale[0], "*$", iD, "+", "%g", vScale[1], ")");
                  else if (j == 2 - !b3Dim)  // Last element; if missing, interrupt line
                    pl~= sprint("($", iD, ")");
                  else                       // Normal coordinate
                    pl~= sprint(iD);

                  iD+= mScale[iAxis][0] == AXIS_DATE ? iSpec : 1;
                 }
              for (j= 0; j < rows(aPlot[P_LINE][i][L_XYZ][3]); ++j)
                pl= sprint(pl, ":", iD++);
            }
          else // Default using specification
            {
              iD= aPlot[P_TYPE][1] ? int(sumr(aPlot[P_DATESPEC])) : 1;  

              // Use iD dates, plus the y-value
              pl= sprint(pl, "1:($", ++iD, ")");
              //   plus the z-value, if there is one
              if (b3Dim)
                pl= sprint(pl, ":($", ++iD, ")");
            }
        }    

      // add the axes
      if (!b3Dim && !(aPlot[P_LINE][i][L_AXES][:1] == 0))
        pl= sprint(pl, " axes x", aPlot[P_LINE][i][L_AXES][0]+1, 
                             "y", aPlot[P_LINE][i][L_AXES][1]+1);
      // add the key
      //   add the fontsize iFont using the output type iOuttype
      sKey= aPlot[P_LINE][i][L_KEY];
      if ((iFont > 0) && (iSize > 0) && sizeof(sKey))
        sKey= sprint("{/=", iSize, " ", sKey, "}");
      pl= sprint(pl, " title '", sKey, "'");
      // add the linestyle (lines, points etc)
      
      if (aPlot[P_LINE][i][L_TYPE][0] != "")
        pl= sprint(pl, " with ", aPlot[P_LINE][i][L_TYPE][0]);

      // add the linetype (color: 1 less then Ox-linetype)
      iColor= aPlot[P_LINE][i][L_TYPE][1];
      iPt= aPlot[P_LINE][i][L_POINT][0];
      iPs= aPlot[P_LINE][i][L_POINT][1];
      
      if (isstring(iColor))
        pl~= sprint(" lc rgb '", iColor, "'");
      else if (ismissing(iColor))
        {
          pl~= " lc rgb variable";      
          if (aPlot[P_LINE][i][L_TYPE][0] == "linespoints")
            oxwarning ("Gnuplot doesn't like linespoints in combination with ZMODE_COLOR");
        }    
      else
        {
//          iColor= iColor <= 0 ? "bgnd"  :   // Translate white to background
          iColor= iColor <= 0 ? 0 :         // Translate white to dotted lines
                  round(iColor) == 1 ? -1 : // Translate dotted lines to black
                  int(iColor-1);            // Other colors are OxColor-1
          pl~= sprint(" lt ", iColor);
        }

      if (strfind({"points", "linespoints", "yerrorbars"},
                  aPlot[P_LINE][i][L_TYPE][0]) > -1)
        {   
          if (!ismissing(iPt))      // Make pointtype correspond to linetype
            {
              // Translate to gnuplot symbols
              iPt= max(min(fabs(iPt), PL_DOT), 0);
              // 2/12/10: Got rid of the 'nosymbol' by replacing -1 by 10
              iPt= <5, 4, 1, 3, 6, 10,  7, 8, 9, 12, 13, 2, 0>[iPt];
              pl= sprint(pl, " pt ", iPt);
            }  
          if (!ismissing(iPs))
            pl= sprint(pl, " ps ", iPs >= 0 ? iPs : "variable");
          if (rows(aPlot[P_LINE][i][L_XYZ][3]) > 2)
            oxwarning("DrawZ(ZMODE_SYMBOL/ZMODE_COLOR, ...) not robust against multi-use");  
        }        

      iLineWidth= ismissing(aPlot[P_LINE][i][L_TYPE][2]) 
        ? s_Fig[F_GLOB][G_DEFLINE][2] 
        : aPlot[P_LINE][i][L_TYPE][2];

      if (!ismissing(iLineWidth))
        pl= sprint(pl, " lw ", max(round(iLineWidth/10), 1));

      if (i < i2-1)
        pl= sprint(pl, ", \\\n");
      fprint(fh, pl);
      pl= "     ";
    }

  return vContour;  
}

/*
**  s_WriteTables(const fh, const sBasename)
**
**  Purpose:
**    Write separate tables for the contour plots
**
**  Inputs:
**    fh        file index
**    sBasename string, base of filename
**
**  Return value:
**    bCont     boolean, if TRUE contour was plotted
*/
s_WriteTables(const fh, const sBasename)
{
  decl bCont, i, j, iN, iNPlots, aaPlot, sBase, iColorModel, vL, vSpec; 
  
  bCont= FALSE;
  iNPlots= s_Fig[F_NPLOTS];
  for (i= 0; i < iNPlots; ++i)
    {
      aaPlot= &(s_Fig[F_PLOT][i]);
      iN= aaPlot[0][P_NLINES];
      if (aaPlot[0][P_TYPE][T_CONTOUR] && !aaPlot[0][P_TYPE][T_XYZ])
        {
          s_DateSpec(&vL, &vSpec, aaPlot);
          bCont= TRUE;
          for (j= 0; j < iN; ++j)
            if (sizerc(aaPlot[0][P_LINE][j][L_XYZ][2]))
              {
                // Prepare the table, with correct number of
                //   contourlines
                sBase= sprint(sBasename, i, "_", j);
                iColorModel= aaPlot[0][P_LINE][j][L_TYPE][3];

                fprintln(fh, "# Plotting contour ", sBase, ".log");
                fprintln(fh, "set table '", sBase, ".log'");
                fprintln(fh, "set contour");
                fprintln(fh, "set nosurface");
                fprint(fh, "set cntrparam levels");
          
                if (isarray(iColorModel))
                  { 
                    // iColorModel specifies the start, increment and possibly end
                    if (isdotmissing(iColorModel[2]))
                      fprintln (fh, " incremental ", 
                                iColorModel[0], ",", iColorModel[1]);
                    else if (sizeof(iColorModel) == 3)
                      fprintln (fh, " incremental ", 
                                iColorModel[0], ",", iColorModel[1], ",", iColorModel[2]);
                  }
                else if (sizerc(iColorModel) > 1)
                  {
                    // iColorModel specifies the vector of contours in this case
                    fprint(fh, " discrete");
                    for (i= 0; i < sizerc(iColorModel); ++i)
                      fprint(fh, i == 0 ? " " : ",", iColorModel[i]);
                    fprintln(fh, "");
                  }  
                else
                  // iColorModel specifies the number of contours in this case
                  fprintln(fh, " auto ", 
                    iColorModel[0] > 0 ? iColorModel[0] : "");

                // Special action: Set logscale if needed...                         
                if (sizeof(aaPlot[0][P_SET][S_AXISSCALE]) == 2)
                  fprintln (fh, aaPlot[0][P_SET][S_AXISSCALE][0]);
                s_WriteFilePlot(fh, aaPlot[0], FALSE, "", j, -1,
                                int(sumr(vSpec)));
                s_WriteLines(fh, aaPlot[0], vSpec, range(0, iN-1) .!= j);
                fprintln (fh, "unset table");
              }                
        }
    }
  return bCont;  
}
