/*
**  GnuPipe
**
**  Purpose:
**    Open up a pipe using popen for gnuplot
**
**  Date:
**    18/11/2006
**
**  Author:
**    Charles Bos       
*/
#define _POSIX_C_SOURCE 2    // In order to get popen() recognized in
                             // stdio.h, see /usr/include/stdio.h

#include <stdio.h>
#include <stdlib.h>
#include "oxexport.h"
#include "jdmath.h"

#define OxSetOpenFile(pv,i,d,f) (pv)[i].type = OX_FILE, (pv)[i].t.ioval.fp = d, (pv)[i].t.ioval.fmode= f

/*
**  popen_C(const sGnuExec, const sType)
**
**  Purpose:
**    Open a pipe to gnuplot
**
**  Input:
**    sGnuExec  string, executable name
**    sType     string, type of pipe, either "r" or "w"
**
**  Return value:
**    fh    file handle to opened pipe, or 0
*/
void OXCALL popen_C(OxVALUE *rtn, OxVALUE *pv, int cArg)
{
  int fmode;
  char *sGnuExec, *sType;
  FILE *fh;
  
  if ((cArg != 2) && (cArg != 3))
    OxRunError(ER_ARGS, NULL);
 
  OxLibCheckType(OX_STRING, pv, 0, 1);
  sGnuExec= OxStr(pv, 0);
  sType= OxStr(pv, 1);
  fmode= (sType[0] == 'r') ? 161 :
         (sType[0] == 'w') ? 162 : -1;
  
  if (cArg == 3)
    {
      OxLibCheckType(OX_INT, pv, 2, 2);
      fmode= OxInt(pv, 2);
    }  
//   printf ("Read arguments gnuplot=%s, type=%s, mode=%i.\n", sGnuExec, sType, fmode);

  if (fmode > 0)
    {
//      printf ("Opening pipe to %s of type %s\n", sGnuExec, sType);
      fh= popen(sGnuExec, sType);
      if (fh == NULL)
        OxRunErrorMessage("Oops, I can't find gnuplot");
        
      OxSetOpenFile(rtn, 0, fh, fmode);
    }
  else
    {
//      printf ("Not opening pipe, some error...\n");
      OxSetInt(rtn, 0, 1);
    }  
}

/*
**  fopen_C(const sFile, const sType)
**
**  Purpose:
**    Open a file to whereever
**
**  Input:
**    sFile     string, file to be opened
**    sType     string, type of file, either "r" or "w"
**
**  Return value:
**    fh    file handle to opened pipe, or 0
*/
void OXCALL fopen_C(OxVALUE *rtn, OxVALUE *pv, int cArg)
{
  int fmode;
  char *sFile, *sType;
  FILE *fh;
  
  if ((cArg != 2) && (cArg != 3))
    OxRunError(ER_ARGS, NULL);
 
  OxLibCheckType(OX_STRING, pv, 0, 1);
  sFile= OxStr(pv, 0);
  sType= OxStr(pv, 1);
  fmode= (sType[0] == 'r') ? 161 :
         (sType[0] == 'w') ? 162 : -1;
  
  if (cArg == 3)
    {
      OxLibCheckType(OX_INT, pv, 2, 2);
      fmode= OxInt(pv, 2);
    }  
//   printf ("Read arguments file=%s, type=%s, mode=%i.\n", sFile, sType, fmode);

  if (fmode > 0)
    {
//       printf ("Opening file %s of type %s\n", sFile, sType);
      fh= fopen(sFile, sType);
      if (fh == NULL)
        OxRunErrorMessage("Oops, I can't find open file");
        
      OxSetOpenFile(rtn, 0, fh, fmode);
    }
  else
    {
//       printf ("Not opening pipe, some error...\n");
      OxSetInt(rtn, 0, 1);
    }  
}

/*
**  fflush_C(const fh)
**
**  Purpose:
**    Flush the pipe
**
**  Input:
**    fh    file handle to opened pipe
**
**  Return value:
**    none
*/
void OXCALL fflush_C(OxVALUE *rtn, OxVALUE *pv, int cArg)
{
  int ir;
  FILE *fh;
  
  if (cArg != 1)
    OxRunError(ER_ARGS, NULL);
  OxLibCheckType(OX_FILE, pv, 0, 0);
  
  fh= (pv)[0].t.ioval.fp;
  ir= fflush(fh);
  
  OxSetInt(rtn, 0, ir);
}

/*
**  pclose_C(const fh)
**
**  Purpose:
**    Closes the pipe to gnuplot
**
**  Input:
**    fh    file handle to opened pipe
**
**  Return value:
**    none
*/
void OXCALL pclose_C(OxVALUE *rtn, OxVALUE *pv, int cArg)
{
  int ir;
  FILE *fh;
  
  if (cArg != 1)
    OxRunError(ER_ARGS, NULL);
 
  OxLibCheckType(OX_FILE, pv, 0, 0);
  fh= (pv)[0].t.ioval.fp;
  ir= pclose(fh);
  
  OxSetInt(rtn, 0, ir);
}

