/*
**  ScatCol
**
**  Purpose:
**    Scatter plot, varying colors
**
**  Date:
**    23/11/07
**
**  Author:
**    Charles Bos, James Davidson
*/
#include <oxstd.h>
#include <packages/gnudraw/gnudraw.h>

/*
**  rgb(const mR, const mG, const mB)
**
**  Purpose:
**    Translate a color triplet to RGB
**
**  Inputs:
**    mR, mG, mB    n x k matrix of colors red/green/blue, in range
**                  0-255
**
**  Return value:
**    mRGB          n x k matrix with RGB value between 0-(2^24-1)
*/
rgb(const mR, const mG, const mB)
{
  return imod(mR, 256)*2^16 + imod(mG, 256)*2^8 + imod(mB, 256);
}

/*
**  rgbN(const iN)
**
**  Purpose:
**    Provide a range of iN values, smoothly changing color from red to
**    blue
**
**  Inputs:
**    iN        integer, size of color vector
**
**  Return value:
**    vCol      vector of size 1 x iN of colors
*/
rgbN(const iN)
{
  decl vC, vCol;

  // Construct a range from 0-255, of size iN/2
  vC= round(range(0, iN/2)*255*2/iN);
  
  // Construct colors from red to green, and from green to blue
  vCol = rgb(reverser(vC), vC, 0)~rgb(0, reverser(vC), vC);
  
  return vCol[:iN-1];
}

main()
{
  decl vX, vY, iN, i, vC;
  
  iN= 1000;
  vX= range(0, iN-1);
  vY= rann(1, iN);
  vC= round(vX / (iN-1) * 255);       // Color range from 0-255

  DrawXMatrix(0, vY, "Red", vX, "", 1, 2);
  DrawZ(rgb(vC, 0, 0), "col", ZMODE_COLOR);

  DrawXMatrix(1, vY, "Green", vX, "", 1, 2);
  DrawZ(rgb(0, vC, 0), "col", ZMODE_COLOR);

  DrawXMatrix(2, vY, "Blue", vX, "", 1, 2);
  DrawZ(rgb(0, 0, vC), "col", ZMODE_COLOR);

  DrawXMatrix(3, vY, "Black", vX, "", 1, 2);
  DrawZ(rgb(vC, vC, vC), "col", ZMODE_COLOR);

  DrawXMatrix(4, vY, "Smooth", vX, "", 1, 2);
  DrawZ(rgbN(iN), "col", ZMODE_COLOR);

  SaveDrawWindow("graphs/scatcol.png");
  ShowDrawWindow();
}
