/*
**  PenTest
**
**  Purpose:
**    Compute test statistics according to Pena03A
**
**  Date:
**    10/12/03
**
**  Author:
**    Charles Bos
*/
#include <oxstd.h>      // Include the Ox standard library header
#include <packages/gnudraw/gnudraw.h>
#include <packages/oxutils/oxutils.h>

extern "bdsccode/bdstest_ox,TestBds_C" TestBds_C(const vX, const iM, const dEps);
extern "bdsccode/bdstest_ox,BdsFree_C" BdsFree_C();

GenerateData(const iT, const iM, const bGraph)
{
  decl vE, vY, i;
  
  if (iT < 0)
    oxrunerror("Length t not specified");
    
  vE= rann(iT+5, 1);
  vY= zeros(vE);
  for (i= 5; i < iT+5; ++i)
    if (iM == 0)
      vY[i]= vE[i];
    else if (iM == 1)
      vY[i]= .4*vY[i-1] + 0.8*vY[i-1]*vE[i-1] + vE[i];
    else if (iM == 2)
      vY[i]= vY[i-1] <= 1 
        ? 1 - 0.5*vY[i-1] + vE[i]
        : 1 + 0.5*vY[i-1] + vE[i];
    else if (iM == 3)
      vY[i]= vY[i-1] <= 1 
        ? 1 - 0.5*vY[i-1] + vE[i]
        : 1 + vE[i];
    else if (iM == 4)
      vY[i]= -0.4*vE[i-1] +0.3*vE[i-2] + 0.5*vE[i]*vE[i-2] +  vE[i];
    else if (iM == 5)
      vY[i]= -0.3*vE[i-1] +0.2*vE[i-2] + 0.4*vE[i-1]*vE[i-2] 
        - 0.25*sqr(vE[i-2]) + vE[i];
    else if (iM == 6)
      vY[i]= 0.4*vY[i-1] -0.3*vY[i-2] + 0.5*vY[i-1]*vE[i-1] +  vE[i];
    else if (iM == 7)
      vY[i]= 0.4*vY[i-1] -0.3*vY[i-2] + 0.5*vY[i-1]*vE[i-1] + 0.8*vE[i-1] +  vE[i];
    else if (iM == 8)
      vY[i]= 0.2*vE[i-1]^3 + vE[i];
    else if (iM == 9)
      vY[i]= 0.6*vE[i-1]*(sqr(vE[i-2]) + 0.8*sqr(vE[i-3]) 
        + 0.8^2*vE[i-4]^2 + 0.8^3*vE[i-5]^2) + vE[i];
    else if (iM == 10)
      vY[i]= 0.5*vY[i-1] + vE[i];
    else if (iM == 11)
      vY[i]= 0.3*vY[i-1] + 0.5*vY[i-2] - 0.5*vY[i-3] + vE[i];
    else
      oxrunerror ("Model not recognized");

  if (bGraph)
    {      
      SetDrawWindow("Data");
      DrawMatrix(0, vY[5:]', "Y", 1, 1);
      ShowDrawWindow();    
    }
      
  return vY[5:];    
}

FitAR(const vY, const iM)
{
  decl mX, i, vRSS, ir, vBeta;
  
  mX= ones(vY)~lag0(vY, range(1, iM));
  vRSS= new matrix [1][iM+1];
  for (i= 0; i < iM+1; ++i)
    {
      ir= olsc(vY, mX[][:i], &vBeta);
      vRSS[i]= sumsqrc(vY-mX[][:i]*vBeta);
    }
    
  return vRSS;
}      

/*
**  TestTsay(const vE, const iM)
**
**  Purpose:
**    Perform the Tsay test on residuals vE, with lag iM
**
**  Inputs:
**    vE    iT x 1 vector of residuals
**    iM    integer, lag
**
**  Return value:
**    vR    1 x 3 vector with test statistic, p-value and conclusion
*/
TestTsay(const vE, const iM)
{
  decl mZ, mELag, mX, vBeta, vEP, dF, dP, iT, iH, i, j, k, ir;
  
  iT= rows(vE);
  iH= iM*(iM+1)/2;
  mELag= lag0(vE, range(1, iM));
  mX= zeros(iT, iH);
  ir= 1;
  dF= M_NAN;
  
  for (i= k= 0; i < iM; ++i)
    for (j= i; j < iM; ++j)
      {
        mZ= mELag[][i] .* mELag[][j];     // lag (i+1) * lag(j+1)
        ir= ir && olsc(mZ, mELag, &vBeta);
        mX[][k++]= mZ - mELag*vBeta;
      }
  ir= ir && olsc(vE, 1~mX, &vBeta);
  
  if (ir)
    {
      vEP= vE- (1~mX)*vBeta;
  
      dF= ((vE'vE-vEP'vEP)/iH)/(vEP'vEP/(iT-iH-1));
    }
  dP= tailf(dF, iH, iT-iH-1);
  
  return dF~dP~(dP < .05);
}

/*
**  TestBDS(const vE, const iK, const dC)
**
**  Purpose:
**    Perform the BDS test on residuals vE, with distance dC
**
**  Inputs:
**    vE    iT x 1 vector of residuals
**    iK    integer, block size
**    dC    double, distance measure
**
**  Return value:
**    vR    1 x 3 vector with test statistic, p-value and conclusion
*/
TestBDS(vE, const iK, const dC)
{
  decl dT, dCkT, dC1Tk, iC, dP, i, j, iT, vDist;
  
  vE/= sqrt(varc(vE));      // Scale residuals

  dCkT= dC1Tk= iC= 0;
  iT= sizerc(vE);
  vDist= new matrix [1][(iT-iK)*(iT-iK-1)/2];
  for (i= 0; i < iT-iK-1; ++i)
    for (j= i+1; j < iT-iK; ++j)
      {
        vDist[iC]= max(fabs(vE[i:i+iK-1] - vE[j:j+iK-1]));
        dC1Tk+= (fabs(vE[i] - vE[j]) <= dC);
        ++iC;
      }

  dCkT= sumr(vDist/sqrt(varr(vDist)) .<= dC);

  // Adapt for iC= (iT-iK)*(iT-iK-1)/2
  dCkT= 2*dCkT/((iT-iK)*(iT-iK-1));
  dC1Tk= 2*dC1Tk/((iT-iK)*(iT-iK-1));
  dT= sqrt(iT-iK-1)*(dCkT-dC1Tk^iK)/sqrt(varr(vDist));
  dP= 2*tailn(fabs(dT));
  
  return dT~dP~(dP < .05);
}

/*
**  TestBDS_Ox(const vE, const iK, const dC)
**
**  Purpose:
**    Perform the BDS test on residuals vE, with distance dC, 
**
**  Inputs:
**    vE    iT x 1 vector of residuals
**    iK    integer, block size
**    dC    double, distance measure
**
**  Return value:
**    mR    iK-1 x 3 vector with test statistic, p-value and conclusion,
**          for lag lengths 2,..,iK
*/
TestBDS_Ox(vE, const iK, const dC)
{
  decl mBds, vP;
  
  vE= vecr(vE)' / sqrt(varc(vE));      // Scale residuals
  mBds= TestBds_C(vE, iK, dC);
  vP= 2*tailn(fabs(mBds));
  
  return (mBds|vP|(vP < .05))';
}

/*
**  TestIC(const vRSS, const iT, const amIC)
**
**  Purpose:
**    Test using the information criteria which model performs best
**
**  Input:
**    vRSS  1 x iM+1 matrix with RSS
**    iT    integer, number of observations
**
**  Output:
**    amIC  (optional) 4 x iM+1 matrix with AIC, BIC, AICc and HQ criteria
**
**  Return value:
**    vIC   1 x 4 vector with booleans, indicating if the IC's select
**          the model with p=0 lags. Note that this means that if IC=1,
**          the linear model is accepted.
*/
TestIC(const vRSS, const iT, ...)
{
  decl mIC, vI, vPen, iM, va;
  
  iM= columns(vRSS)-1;
  vI= 1+range(0, iM);
  // AIC, BIC, AICc, HQ
  vPen= (vI .* 2/iT)|
        (vI .* log(iT)/iT)|
        (2*(vI+1)./(iT-(vI+2)))|
        (vI .* iM * log(log(iT))/iT);
        
  mIC= log(vRSS/iT) + vPen;
//  print ("Pen: ", vPen, "mIC: ", mIC);        
  
  va= va_arglist();
  if (sizeof(va) && isarray(va[0]))
    va[0][0]= mIC;

  return (mincindex(mIC') .== 0);  
}

/*
**  TestBLj(const vE, const iM)
**
**  Purpose:
**    Perform the Box-Ljung test on residuals vE, with lag iM
**
**  Inputs:
**    vE    iT x 1 vector of residuals
**    iM    integer, lag
**
**  Return value:
**    vR    1 x 3 vector with test statistic, p-value and conclusion
*/
TestBLj(const vE, const iM)
{
  decl vACF, vBLj, vE2, iT, dBLj, dP;
  
  iT= sizerc(vE);
  vE2= sqr(vE);  
  vACF= acf(vE2, iM)[1:];
  dBLj= iT*(iT+2)*sumc(sqr(vACF) ./ range(iT-1, iT-iM, -1)');

  dP= tailchi(dBLj, iM);
  
  return dBLj~dP~(dP < .05);
}


main()
{
  decl vY, mX, dC, iMod, iT, iM, iR, iSeed, iP, vBeta, vE, vE2, vRSS, 
       i, k, ir, vIC, mIC, vBLj, mBDS, vFTsay, mRes, mRej, vP, sOutbase;

  println ("PenTest\n=======");
  
  dC= 1.5;      // Distance in BDS test
  iMod= iT= -1;
  iR= 100;
  iSeed= 999;
  ReadArg(&iMod, "model", 1);
  ReadArg(&iT, "t", 1);
  ReadArg(&iR, "r", 1);
  ReadArg(&iSeed, "seed", 1);
  if (ReadArg(&i, "nogr", 0))
    DrawAdjust(ADJ_SHOW, FALSE);
  iM= floor(sqrt(iT));
  
  setseed(iSeed);  

  println ("Generating ", iT, " observations from model ", iMod,
           " with seed ", iSeed);
  
  mRes= new matrix [9][iR];
  vP= new matrix [1][iR];
  infoinit(100, 10);
  for (i= 0; i < iR; ++i)
    {
      info(i, iR);
      TrackTime(0);
      vY= GenerateData(iT, iMod, FALSE);
      TrackTime(1);
      vRSS= FitAR(vY, 4);
      TrackTime(2);
      vIC= TestIC(vRSS, iT, &mIC);

      // Select only from p= 1, .., 4 using AIC
      vP[i]= iP= mincindex(mIC[0][1:]')+1;
      mX= iP ? ones(vY)~lag0(vY, range(1, iP)) : ones(vY);
      ir= olsc(vY, mX, &vBeta);
      vE= vY-mX*vBeta;
      vE2= sqr(vE);  

      // Select AR order for model on squared residuals
      TrackTime(3);
      vRSS= FitAR(vE2, iM);

      TrackTime(4);
      mBDS= TestBDS_Ox(vE, 4, dC);

      TrackTime(5);
      vBLj= TestBLj(vE, iM);
      
      TrackTime(6);
      vIC= TestIC(vRSS, iT, &mIC);
      
      TrackTime(7);
      vFTsay = TestTsay(vE, 5);

      if (iR < 10)
        print (mBDS, vBLj, vIC, vFTsay, vRSS, mIC, mincindex(mIC')');

      mRes[][i]= vFTsay[1]|mBDS[][1]|vBLj[1]|vIC';
    }      
  info(iR, iR);
  BdsFree_C();  
  
  print ("%r", {"p", "frac"}, "%cf", {"%6.2f"}, 
         range(0, 4)|countr(vP, range(0, 3))/iR);
         
  mRej= mRes .< .05;
  print ("%c", {"FTsay", "BDS2", "BDS3", "BDS4", "Qml", "AIC", "BIC", "AICc", "HQ"}, 
         "%r", {"Av rej", "V rej", "Av res", "V res"},
         "%cf", {"%6.3f"},
         meanr(mRej)'|varr(mRej)'|meanr(mRes)'|varr(mRes)');
  TrackReport();       
         
  sOutbase= sprint("excl/ptm", iMod, "t", iT, "r", iR);
  
  DrawTitle(-1, sprint("Model ", iMod, " with t=", iT, " and r= ", iR));
  DrawDensity(0, mRes, {"FTsay", "BDS2", "BDS3", "BDS4", "Qml", "AIC", "BIC", "AICc", "HQ"}, 
              TRUE, 0, 0);
  SaveDrawWindow(sOutbase~"uni.plb");
  ShowDrawWindow();  
  
  DrawTitle(-1, sprint("Model ", iMod, " with t=", iT, " and r= ", iR));
  DrawBivDensity(0, mRes[3:4][], {"BDS4", "Qml"}, 1, 0, 0, 2);
  DrawDensity(1, mRes[3:4][], {"BDS4", "Qml"}, 1, 0, 0);
  SaveDrawWindow(sOutbase~"biv.plb");
  ShowDrawWindow();
  
  savemat(sOutbase~"res.fmt", mRes);
}
