/*
**  Library MaxIt
**
**  Contents:
**    Routines for maximizing a function, for calculation and printing 
**    of the standard deviation, and for plotting the full conditionals
**    around the optimum.
**
**  Note:
**    Library GnuDraw should be loaded beforehand.
*/
#ifndef MAXIT_INCLUDED
#define MAXIT_INCLUDED

#import <maximize>      // The optimization routines 
#ifndef OXDRAW_INCLUDED
  #include "oxdraw.h"
#endif

#include <oxfloat.h>

/*
**  First routines provide the numerical derivative of the analytical
**  derivative, thus computing the second derivative more precisely
*/
const decl SQRT_EPS =1E-8;        /* appr. square root of machine precision */
const decl DIFF_EPS =1E-8; /* Rice's formula: log(DIFF_EPS)=log(MACH_EPS)/2 */
const decl DIFF_EPS1=5E-6; /* Rice's formula: log(DIFF_EPS)=log(MACH_EPS)/3 */
const decl DIFF_EPS2=1E-4; /* Rice's formula: log(DIFF_EPS)=log(MACH_EPS)/4 */

dFiniteDiff1_Hess(const x)
{
    return max( (fabs(x) + SQRT_EPS) * SQRT_EPS, DIFF_EPS1);
}

/*
**  Num1Derivative_Hess
**
**  Purpose:
**    Compute the numerical first derivative of the analytical score
**    function, as to get the second derivative
**
**  Based on:
**    Num1Derivative, in maximize.ox
*/
Num1Derivative_Hess(const func, vP, const amHess)
{
    decl i, cp = rows(vP), left, right, fknowf = FALSE, p, h, 
         f, fm, fp, m, vf, vfm, vfp;

    m = new matrix[cp][cp];
    vfm= vfp= vf= new matrix[cp][1];

    for (i = 0; i < cp; i++)    /* get 1st derivative by central difference */
    {
        p = double(vP[i][0]);
        h = dFiniteDiff1_Hess(p);

        vP[i][0] = p + h;
        right = func(vP, &fp, &vfp, 0);
        vP[i][0] = p - h;
        left = func(vP, &fm, &vfm, 0);
        vP[i][0] = p;                         /* restore original parameter */

        if (left && right)
            m[][i] = (vfp - vfm) / (2 * h);       /* take central difference */
        else if (left)
        {
            if (!fknowf)					    /* see if we already know f */
			{	fknowf = func(vP, &f, &vf, 0);	      /* not: try to get it */
				if (!fknowf)
                	return FALSE;
			}
            m[i][] = (vf - vfm) / h;                 /* take left difference */
        }
        else if (right)
        {
            if (!fknowf)
			{	fknowf = func(vP, &f, &vf, 0);
				if (!fknowf)
                	return FALSE;
			}
            m[i][] = (vfp - vf) / h;                /* take right difference */
        }
        else
            return FALSE;
    }
    amHess[0] = m;

return TRUE;
}

/*
**  MaxStdErr(const fnFunc, const vP, const iT, const avS, const amS2)
**  MaxStdErr(const fnFunc, const vP, const iT, const avS, const amS2, const bNum)
**
**  Purpose:
**    Calculate the numerical standard errors
**
**  Inputs:
**    ...
**    bNum  boolean, TRUE (default): Use numerical derivatives
*/
MaxStdErr(const fnFunc, const vP, const iT, const avS, const amS2, ...)
{
  decl ir, covar, va, bNum;

  va= va_arglist();
  bNum= sizeof(va) ? va[0] : TRUE;

  amS2[0]= avS[0]= 0;
  if (bNum)
    {
      ir= Num2Derivative(fnFunc, vP, &covar);
      if (ir)
        amS2[0]= invertgen(-covar, 30)/iT;
//      print ("Numer: ", amS2[0], determinant(amS2[0]));  
    }  
  else  
    {
      ir= Num1Derivative_Hess(fnFunc, vP, &covar);  
      covar= (covar + covar')/2;
      if (ir)
        amS2[0]= invertgen(-covar, 30)/iT;
//      print ("Anal: ", amS2[0], determinant(amS2[0]), amS2[0]-result);   
    }

  avS[0]= sqrt(diagonal(amS2[0]))';
  ir= !(amS2[0] == 0);
  if (!ir)
    println("Covar() failed in numerical second derivatives");

  return ir;
}

/*
**  Max_It()
**
**  Purpose:
**    Maximize the likelihood function
**
*/
MaxIt(const fnFunc, const avP, const amS2, const adLnPdf,
      const asVarNames, const iT, ...)
{
  decl ir, i, vS, asVN, va, bNum;
  
  va= va_arglist();
  bNum= sizeof(va) ? va[0] : TRUE;

  ir= MaxBFGS(fnFunc, avP, adLnPdf, 0, bNum);
  println("\n", MaxConvergenceMsg(ir),
          (bNum ? " using numerical derivatives" : 
          " using analytical derivatives"),
          "\nLog-likelihood = ", 
          "%.8g", double(adLnPdf[0]*iT),
          "; n = ", iT);
  MaxStdErr(fnFunc, avP[0], iT, &vS, amS2, bNum);

  asVN= asVarNames;
  if (sizeof(asVN) < sizerc(avP[0]))
    {
      asVN= new array [sizerc(avP[0])];
      for (i= 0; i < sizerc(avP[0]); ++i)
        asVN[i]= sprint("Par ", "%2i", i);
    }
  print("Parameters with standard errors:",
        "%r", asVN, "%cf", {"%12.5g", "  (%7.5f)"}, 
        avP[0] ~ vS);

  return ir;
}

/*
**  Max_PlotConds(const fnFunc, const vP, const vS, const asVarNames, 
**                const iK, const sFilebase)
**
**  Purpose:
**    Plot the univariate full conditionals
*/
Max_PlotConds(const fnFunc, const vP, const vS, const asVarNames,
              const iT, const iK, const sFilebase, ...)
{
  decl nDim, vLL, i, j, vX, vlP, ir, dLL, vP_True, va, bShow, sFile;
  
  va= va_arglist();
  if (isstring(iK))
    oxrunerror("Error: Pass number of observations to this version of Max_PlotConds", 0);
  bShow= (sizeof(va) > 0) ? va[0] : (sizeof(sFilebase) == 0);
  
  nDim= rows(vP);
  vP_True= (columns(vP) > 1) ? vP[][1] : M_NAN;
  vLL= new matrix [1][iK];
  
  for (i= 0; i< nDim; ++i)
    {
      vX= range(0, iK-1)/(iK-1) * 4 * vS[i] + vP[i][0] - 2*vS[i];
      vlP= vP[][0];
      for (j= 0; j < iK; ++j)
        {
          vlP[i]= vX[j];
          ir= fnFunc(vlP, &dLL, 0, 0);
          vLL[j]= dLL*iT;
        }
      if (sizerc(deletec(vLL)))
        {
          vLL= exp(vLL-max(deletec(vLL)));
          vLL= vLL/(sumr(deletec(vLL))*(vX[1]-vX[0]));
          DrawXMatrix(i, vLL, asVarNames[i], vX, "");
          if (!isnan(vP_True))
            DrawLine(i, vP_True[i], 0, vP_True[i], max(vLL), 0);      
        }
      else
        println("Warning: All likelihood NaN");   
    }

  #ifdef GNUDRAW_INCLUDED
    sFile= sFilebase~"cond.plb";
  #else  
    sFile= sFilebase~"cond.eps";
  #endif
  if (sizeof(sFilebase) > 0)
    SaveDrawWindow(sFile);
  if (bShow)
    ShowDrawWindow();
  CloseDrawWindow();
}

/*
**  Max_PlotBivConds(const fnFunc, const vP, const vS, const asVarNames, 
**                   const iK, const sFilebase)
**
**  Purpose:
**    Plot the bivariate full conditionals
*/
Max_PlotBivConds(const fnFunc, const vP, const vS, const asVarNames,
                 const iT, const iK, const sFilebase, ...)
{
  decl nDim, vLL, mLL, i1, i2, i, j, vX, vY, vlP, ir, dLL, vP_True, va, bShow, sFile;
  
  va= va_arglist();
  if (isstring(iK))
    oxrunerror("Error: Pass number of observations to this version of Max_PlotBivConds", 0);
  bShow= (sizeof(va) > 0) ? va[0] : (sizeof(sFilebase) == 0);
  
  nDim= rows(vP);
  vP_True= (columns(vP) > 1) ? vP[][1] : M_NAN;
  vLL= new matrix [1][iK];
  mLL= new matrix [iK][iK];
  
  for (i1= 0; i1< nDim; ++i1)
    {
      vX= range(0, iK-1)/(iK-1) * 4 * vS[i1] + vP[i1][0] - 2*vS[i1];
      vlP= vP[][0];
      for (j= 0; j < iK; ++j)
        {
          vlP[i1]= vX[j];
          ir= fnFunc(vlP, &dLL, 0, 0);
          vLL[j]= dLL*iT;
        }
      if (sizerc(deletec(vLL)))
        {
          vLL= exp(vLL-max(deletec(vLL)));
          vLL= vLL/(sumr(deletec(vLL))*(vX[1]-vX[0]));
          DrawXMatrix(i1*nDim+i1, vLL, asVarNames[i1], vX, "");
          if (!isnan(vP_True))
            DrawLine(i1, vP_True[i1], 0, vP_True[i1], max(vLL), 0);      
        }
      else
        println("Warning: All likelihood NaN");   
      for (i2= 0; i2< i1; ++i2)
        {
          vY= range(0, iK-1)/(iK-1) * 4 * vS[i2] + vP[i2][0] - 2*vS[i2];
          vlP= vP[][0];
          for (i= 0; i < iK; ++i)
            {
              vlP[i1]= vX[i];
              for (j= 0; j < iK; ++j)
                {
                  vlP[i2]= vY[j];
                  ir= fnFunc(vlP, &dLL, 0, 0);
                  mLL[i][j]= dLL*iT;
                }
            }
          mLL/= fabs(max(deleter(vecr(mLL))));  
          if (sizerc(deleter(vecr(mLL))))
            DrawXYZ(i1*nDim+i2, vX, vY, mLL, 0, asVarNames[i1],
                    asVarNames[i2], "");
          else
            println("Warning: All likelihood NaN");   
        }        
    }

  #ifdef GNUDRAW_INCLUDED
    sFile= sFilebase~"bcond.plb";
  #else  
    sFile= sFilebase~"cond.eps";
  #endif
  if (sizeof(sFilebase) > 0)
    SaveDrawWindow(sFile);
  if (bShow)
    ShowDrawWindow();
  CloseDrawWindow();
}

#endif /* MAXIT_INCLUDED */
