/*
**  Library info.ox
**
**  Purpose:
**    Time the duration of a program
**
**  Contents:
**    info(const it, const nofexps)
**      Provide screen info number of (elapsed) iterations and 
**      elapsed/remaining clock time.
**
**    infobugcheck()
**      Check for the bug in the timer() function on Unix, which
**      switches to negative values after 214747 hundredths of a second,
**      approximately half an hour. This routine should be called every
**      30 minutes for correct output of infotimer(). It is called 
**      at every call to info().
**      For Ox-versions >= 3.10, this routine is no longer needed as
**      Ox itself adjusts for the bug.
**
**    infotimer()
**      Provide a value of the timer() function, adjusted for the
**      timer() bug.
**
**    infotimespan(const dTime, ...)
**      Provide a string with the processor time spent, like timespan()
**  
**  Author:
**    Charles Bos
**
**  Version:
**    3     Using today() instead of timer(), to get clock time
**    4     Included infotimer(), a replacement for timer(), which
**          adjusts for the timer() bug.
**    5     Also checking for the amount of time passed since the
**          last call
**    6     Skipping bug-check routines, not needed since Ox 3.10
**
**  Date:
**    29/12/2002
*/
#ifndef INFO_INCLUDED
  #define INFO_INCLUDED
  #include <oxstd.h>
  #include <oxfloat.h>
  
static decl s_info_start= -1, s_info_lasttime= 1, s_info_buground= 0,
            s_info_timpos= TRUE, s_info_deltatime= M_NAN, s_info_deltait= M_NAN, 
            s_info_count= M_NAN, s_info_lastit= 0;


/*
**  infoinit(const it, const dTime)
**
**  Purpose:
**    Initialize the info-routine, indicating that output should only
**    be given every it iterations or every dTime seconds, whichever
**    comes first. Other calls to info give no output.
**
**  Inputs:
**    it          integer, number of iterations to skip before
**                giving screen output.
**    dTime       double, number of seconds to skip before
**                giving screen output.
*/
infoinit(const it, const dTime)
{
  // Initialize
  s_info_start= today();
  s_info_count= 0;
  s_info_deltatime= dTime;
  s_info_lasttime= s_info_start - s_info_deltatime - 1;
  s_info_deltait= it;
  s_info_lastit= -s_info_deltait-1;
}

/*
**  info(const it, const nofexps)
**
**  Inputs:
**    it          integer, number of present iteration (0= initialize)
**    nofexps     integer, total number of iterations to go
**
**  Output:
**                Screen info with number of (elapsed) iterations and
**                elapsed/remaining time
**
*/
info(const it, const nofexps)
{
    decl meantime, esttime, tim, timt, it_loc, bDisplay;
    
    tim= today();
    if (((s_info_start == -1) .|| (it == 0)) && (s_info_count != 0))
      // Initialize
      infoinit(0, 0);

    it_loc= isnan(it) ? s_info_count : it;  

    // Show if it is time, if the requested number of iterations has
    // passed, or if I'm not checking for those
    bDisplay= (tim - s_info_lasttime >= s_info_deltatime) ||
              (it - s_info_lastit >= s_info_deltait) ||
              (it == nofexps);                

    if (bDisplay)
      {
        print ("\n");
        println ("-------------------------------------------------------------");
        println ("   Number of elapsed iterations: ", "%9i", int(it_loc));
        if (!isnan(nofexps))
          println ("   Number of iterations to go:   ", "%9i", int(nofexps - it_loc));
        println ("   Elapsed time:                 ", "%9s", timespan(s_info_start*100, tim*100));

        if (it_loc .!= 0)
          {
            meantime = 100*(tim-s_info_start) / it_loc;
            esttime = meantime * (nofexps - it_loc);
            if (meantime .> 50)
                println("   Time per iteration:           ", "%12.2f", double(meantime/100));
            else
                println("   Time per 100 iterations:      ", "%12.2f", double(meantime));
		    if (esttime == 0)
              println("   Estimate of remaining time:   ", "%12s", "0.00"); 
		    else if (!isnan(esttime))
              println("   Estimate of remaining time:   ", 
                      "%12s", timespan(esttime, 0));
          }
        else
          {
            println("   Time per iteration:                     .");
            println("   Estimate of remaining time:             .");
          }
        s_info_lasttime= tim;  
        s_info_lastit= it_loc;  
      }  
//     else
//       println ("Not printing with ", tim, " and lasttim=",
//                s_info_lasttime, " and ", s_info_deltatime, " with difference ",
//                tim-s_info_lasttime, " counting for ", s_info_deltait, 
//                " between iterations");  

    ++s_info_count;  

    return bDisplay;
}

/*
**  infobugcheck()
**
**  Deprecated
*/
infobugcheck()
{
  println ("Warning: This routine is deprecated.");
}

/*
**  infotimer()
**
**  Deprecated
*/
infotimer()
{
  println ("Warning: This routine is deprecated, use timer() directly");
  return timer();
}

/*
**  infotimespan(const time);
**  infotimespan(const time, const time0);
**  
**  Deprecated
*/
infotimespan(const dTime, ...)
{
  decl va;
  
  va= va_arglist();
  println ("Warning: This routine is deprecated, use timespan(dTime, ...) directly");

  return timespan(dTime, va);
}
#endif
