#ifndef LIBKERN_INCLUDED

#define LIBKERN_INCLUDED

/*
**  meanisc(const ma, const vw)
**
**  Inputs:
**    ma    iT x iN matrix A
**    vw    iT x  1 vector with weights w
**
**  Return value
**    vmean 1 x iN vector with mean of columns of A, adjusted for
**          weights
*/
meanisc(const ma, const vw)
{
  return meanc(ma .* vw)/meanc(vw);
}

/*
**  meanisr(const ma, const vw)
**
**  Inputs:
**    ma    iN x iT matrix A
**    vw    1 x iT vector with weights w
**
**  Return value
**    vmean iN x 1 vector with mean of rows of A, adjusted for
**          weights
*/
meanisr(const ma, const vw)
{
  return meanr(ma .* vw)/meanr(vw);
}

/*
**  varisc(const ma, const vw, ...)
**  varisc(const ma, const vw, const vmean)
**
**  Inputs:
**    ma    iT x iN matrix A
**    vw    iT x  1 vector with weights w
**    vmean (optional) 1 x iN vector with mean of columns of A, adjusted 
**          for weights
**
**  Return value:
**    vvar  1 x iN vector with variances of columns of A, adjusted for 
**          weights
*/
varisc(const ma, const vw, ...)
{
  decl va, vmean, ve;

  va= va_arglist();  
  vmean= (sizeof(va) > 0) ? va[0] : meanisc(ma, vw);
    
  ve= ma - vmean;
  return meanc(sqr(ve) .* vw)/meanc(vw);
}

/*
**  varisr(const ma, const vw, ...)
**  varisr(const ma, const vw, const vmean)
**
**  Inputs:
**    ma    iN x iT matrix A
**    vw     1 x iT vector with weights w
**    vmean (optional) iN x 1 vector with mean of rows of A, adjusted 
**          for weights
**
**  Return value:
**    vvar  iN x 1 vector with variances of rows of A, adjusted for 
**          weights
*/
varisr(const ma, const vw, ...)
{
  decl va, vmean, ve;

  va= va_arglist();  
  vmean= (sizeof(va) > 0) ? va[0] : meanisr(ma, vw);
    
  ve= ma - vmean;
  return meanr(sqr(ve) .* vw)/meanr(vw);
}

/*
**  varianceis(const ma, const vw, ...)
**
**  Inputs:
**    ma    iN x iP matrix with observations
**    vw    iN x  1 vector with weights
**
**  Optional input:
**    vmean  1 x iP vector with mean of ma
**
**  Return value:
**    mS2   iP x iP matrix with variance of Theta adjusted for 
**          weights
*/
varianceis(const ma, const vw, ...)
{
  decl va, vmean, ve, i, mS2, iN;

  if (sizerc(vw) == 1)
    mS2= variance(ma);
  else
    {
      va= va_arglist();
      vmean= (sizeof(va) > 0) ? vecr(va[0])' : meanisc(ma, vw);
 
      iN= rows(ma);
      ve= ma - vmean;
      mS2= 0;
      for (i= 0; i < iN; ++i)
        mS2+= ve[i][]' * ve[i][] * vw[i];
      mS2= mS2/(iN*meanc(vw));
    }

  return mS2;
}

/*
**  s_KernelUni(const vX, const vY, const avYK)
**
**  Purpose:
**    Fit a kernel through a dataset
**
**  Inputs:
**    vX    1 x iK vector with support points
**    vY    1 x iN vector of data
**
**  Outputs:
**    avYK  1 x iK vector of corresponding y-values for the kernel
**
**  Remark:
**    A Gaussian kernel is applied, with an automatic bandwidth
**    selection based on Silverman's rule. 
**
**  This library is based on code by Ruud Koning, Gary King, Mico Loretan,
**    Geoffrey Shuetrim and Berwin Turlach, and got adapted by
**    Dick van Dijk (in Gauss) and Charles Bos (in Ox)
*/
s_KernelUni(const vX, const vY, const avYK)
{
  decl iT, iqr, ds, dh, ir, vArg, vK;

  iT= columns(vY[0][]);
  iqr= quantiler(vY[0][], .75) - quantiler(vY[0][], .25);
  ds = sqrt(varr(vY[0][]));
  dh= 0.9 * min(ds, (iqr / 1.34)) / iT .^ 0.2;

  avYK[0] = zeros(vX);

  ir= 0;
  if (dh > 0)
    {
      /* Compute density estimates */
      vArg= (vX' - vY[0][]) / dh;
      vK= densn(vArg);

      avYK[0] = meanr(vK)'/dh;
      ir= 1;
    }

  return ir;
}

/*
**  s_KernelMult(const vX, const vY, const avYK)
**
**  Purpose:
**    Fit a kernel through a multivariate dataset
**
**  Inputs:
**    mX    iP x iK matrix with support points
**    mY    iP x iN matrix of observed data points
**
**  Outputs:
**    amYK  1 x iK matrix of corresponding y-values for the kernel
**
**  Remark:
**    A Gaussian kernel is applied, with an automatic bandwidth
**    selection based on Silverman's rule. 
**
**  Based on routine mKernel, by Dick van Dijk.
**
**  Author:
**    Charles Bos
*/
s_KernelMult(const mX, const mY, const amYK)
{
  decl iN, iP, iK, iqr, ds, mh, mih, ddeth, ir, mArg, mK, i, 
       dLogDet, dSignDet;

  iP= rows(mX);
  iK= columns(mX);
  iN= columns(mY);

  if (iP == 1)
    {
      iqr= quantiler(mY[0][], .75) - quantiler(mY[0][], .25);
      ds = sqrt(varr(mY[0][]));
      mh= 0.9 * min(ds, (iqr / 1.34)) / iN .^ 0.2;
      mih= 1/mh;
      ddeth= mh;
    }
  else
    {
      mh= variance(mY');
      mh= choleski(mh).*iN^(-1/(iP+4));
      mih=invert(mh, &dLogDet, &dSignDet);
      ddeth=exp(dLogDet)*dSignDet;
    }

  amYK[0] = zeros(1, iK);

  ir= 0;
  if (!(mh == 0))
    {
      /* Compute density estimates */
      for (i= 0; i < iK; ++i)
        {
          mArg= mih * (mX[][i] - mY);
          mK= densn(mArg);
          mK= prodc(mK) ./ ddeth;

          amYK[0][0][i] = meanr(mK);
        }
      ir= 1;
    }

  return ir;
}

/*
**  s_KernelUniIS(const vX, const vY, const vW, const avYK)
**
**  Purpose:
**    Fit a kernel through a dataset
**
**  Inputs:
**    vX    1 x iK vector with support points
**    vY    1 x iN vector of data
**    vW    1 x iN vector of weights
**
**  Outputs:
**    avYK  1 x iK vector of corresponding y-values for the kernel
**
**  Remark:
**    A Gaussian kernel is applied, with an automatic bandwidth
**    selection based on Silverman's rule. 
**
**  This library is based on code by Ruud Koning, Gary King, Mico Loretan,
**    Geoffrey Shuetrim and Berwin Turlach, and got adapted by
**    Dick van Dijk (in Gauss) and Charles Bos (in Ox)
*/
s_KernelUniIS(const vX, const vY, const vW, const avYK)
{
  decl iT, iqr, ds, dh, ir, vArg, vK;

  iT= columns(vY[0][]);
  iqr= quantiler(vY[0][], .75) - quantiler(vY[0][], .25);
  ds = sqrt(varisr(vY[0][], vW));
  dh= 0.9 * min(ds, (iqr / 1.34)) / iT .^ 0.2;

  avYK[0] = zeros(vX);

  ir= 0;
  if (dh > 0)
    {
      /* Compute density estimates */
      vArg= (vX' - vY[0][]) / dh;
      vK= densn(vArg);

      avYK[0] = meanr(vK .* vW)'/(dh * meanr(vW));
      ir= 1;
    }

  return ir;
}

/*
**  s_KernelMultIS(const mX, const mY, const vw, const avYK)
**
**  Purpose:
**    Fit a kernel through a multivariate dataset
**
**  Inputs:
**    mX    iP x iK matrix with support points
**    mY    iP x iN matrix of observed data points
**    vw    1 x iN  vector of weights
**
**  Outputs:
**    avYK  1 x iK vector of corresponding y-values for the kernel
**
**  Remark:
**    A Gaussian kernel is applied, with an automatic bandwidth
**    selection based on Silverman's rule. 
**
**  Based on routine mKernel, by Dick van Dijk.
**
**  Author:
**    Charles Bos
*/
s_KernelMultIS(const mX, const mY, const vw, const avYK)
{
  decl iN, iP, iK, iqr, ds, mh, mih, ddeth, ir, mArg, mK, i, 
       dLogDet, dSignDet;

  iP= rows(mX);
  iK= columns(mX);
  iN= columns(mY);
  
  mh= varianceis(mY', vw');
  if (iP == 1)
    {
      iqr= quantiler(mY[0][], .75) - quantiler(mY[0][], .25);
      ds = sqrt(mh);
      mh= 0.9 * min(ds, (iqr / 1.34)) / iN .^ 0.2;
      mih= 1/mh;
      ddeth= mh;
    }
  else
    {
      mh= choleski(mh).*iN^(-1/(iP+4));
      mih=invert(mh, &dLogDet, &dSignDet);
      ddeth=exp(dLogDet)*dSignDet;
    }

  avYK[0] = zeros(1, iK);

  ir= 0;
  if (!(mh == 0))
    {
      /* Compute density estimates */
      for (i= 0; i < iK; ++i)
        {
          mArg= mih * (mX[][i] - mY);
          mK= densn(mArg);
          mK= prodc(mK) ./ ddeth;

          avYK[0][0][i] = meanr(mK .* vw);
        }
      ir= 1;
    }
  avYK[0] /= meanr(vw);

  return ir;
}

#endif // LIBKERN_INCLUDED
