/*
**  LoadTab
**
**  Purpose:
**    Read and write a table with column and row headers
**
**  Date:
**    7 October 2005
**
**  Author:
**    Charles Bos
*/
#include <oxstd.h>

/*
**  combinetab(const aasrlab0, const aasclab0, const amX0, 
**             const asrlab1, const asclab1, const mX1, const bHor, const bVer)
**
**  Purpose:
**    Add mX1 to mX0
**
**  Inputs:
**    amX0      pointer to matrix with start
**    mX1       matrix to concatenate
**    bHor      boolean, if true use horizontal concatenation
**    bVer      boolean, if true use vertical concatenation
**
**  Output:
**    amX0      pointer to matrix with mX1 added to it
*/
static combinetab(const aasrlab0, const aasclab0, const amX0, 
                  const asrlab1, const asclab1, const mX1, 
                  const bHor, const bVer)
{
  decl i0, j0, i1, j1, vROrd, vCOrd;
  
//   print ("Combining ", "%r", aasrlab0[0], "%c", aasclab0[0], amX0[0],
//          "%r", asrlab1, "%c", asclab1, mX1);

// Build in: Manner of keeping the sorting order correct
//   vROrd= range(1, sizeof(aasrlab0[0]));
//   vCOrd= range(1, sizeof(aasclab0[0]));

  for (i1= 0; i1 < sizeof(asrlab1); ++i1)
    {
      i0= strifindr(aasrlab0[0], asrlab1[i1]);
      if (asrlab1[i1] == "")
        oxwarning("Missing labels not expected");
      if ((i0 < 0) || (bVer == 1))
        {
          aasrlab0[0]~= asrlab1[i1];
          amX0[0]|= M_NAN;
          i0= sizeof(aasrlab0[0])-1;
          // print ("Adding row");
        }  
      for (j1= 0; j1 < sizeof(asclab1); ++j1)
        {
          if (asclab1[j1] == "")
            oxwarning("Missing labels not expected");
          j0= strifindr(aasclab0[0], asclab1[j1]);
          if ((j0 < 0) || ((bHor == 1) && (i1 == 0)))
            {
              aasclab0[0]~= asclab1[j1];
              amX0[0]~= M_NAN;
              j0= sizeof(aasclab0[0])-1;
              // print ("Adding column");
            }  
          // print (i0~j0~i1~j1, size(amX0[0], mX1));  
          amX0[0][i0][j0]= mX1[i1][j1];  
        } 
    }    
//   print ("to ", "%r", aasrlab0[0], "%c", aasclab0[0], amX0[0]);

  return 1;
}

/*
**  readblock(const fh, const sLab, const iX, const aX, const bNAL)
**
**  Purpose:
**    Read a block of data
*/
static readblock(const fh, const sLab, const iX, const aX, const bNAL)
{
  decl ir, sLine, i;
  
  // Skip lines until labels
  do
    {
      ir= (fscan(fh, "%z", &sLine) >= 0)
         && (sLine != sLab);
    }     
  while (ir);
  
  if (sLine != sLab)
    return 0;

  ir= 1;
  aX[0]= new array [iX];
  for (i= 0; i < iX; ++i)
    {
      ir= ir && (fscan(fh, ">%z", &sLine) > 0);
      if (ir)
        aX[0][i]= (!bNAL && (sizeof(sLine) > 3) && (sLine[0:2] == "NAL")) 
          ? "" : sLine;
    }
  #ifdef DEBUG  
    print ("Reading block ", sLab);
    println (": ", ir);
  #endif
    
  return ir;  
}

loadtab(const sFile, const aasrlab, const aasclab, const amX, ...)
{
  decl i, j, sBase, sDir, dVer, fh, ir, iR, iC, sData, sLine, bNAL;
 
  bNAL= sizeof(va_arglist()) > 0;
  
  // Get file names  
  i= strfind(sFile, ".tab");
  j= max(strfindr(sFile, "/"), strfindr(sFile, "\\"));
  if (i-1 < j+1)  // Return if filename has no base-part
    {
      oxwarning("Incorrect filename, returning");
      return 0;
    }  
  
  sBase= sFile[j+1:i-1];
  sDir= j > 0 ? sFile[0:j] : "";

//   println ("Reading tab");

  // Read tab file
  fh= fopen(sFile, "r");
  if (!isfile(fh))
    {
      #ifdef DEBUG
        oxwarning(sprint("File ", sFile, " not found, returning"));
      #endif        
//       aasrlab[0]= {};
//       aasclab[0]= {};
//       amX[0]= <>;

      return 0;
    }  
  // Check for the correct version
  ir= fscan(fh, "%*s", "%*s", "%d", &dVer, "%*z");
  if (!ir || (dVer != 100))
    {
      oxwarning("Incorrect version of table-file, returning");
      return 0;
    }  
    
//  println ("Reading rows");    

  // Read rows and columns
  ir= fscan(fh, "%d", &iR, "%d", &iC) &&
      readblock(fh, "data", 1, &sData, FALSE) &&
      readblock(fh, "row labels", iR, aasrlab, bNAL) &&
      readblock(fh, "column labels", iC, aasclab, bNAL) &&
      (fscan(fh, "%z", &sLine) > 0);

  sData= sData[0];      
//  if (!(ir && (sData == sBase~".fmt") && (sLine == "end")))
  if (!ir)
    {
      print (ir~iR~iC, sData, aasrlab, aasclab);
      #ifdef DEBUG
        oxwarning("Could not read row/column labels");
      #endif        
      return 0;     
    }
//   println ("Reading data matrix");    
  amX[0]= loadmat(sDir~sData);
  
  return ((rows(amX[0]) == iR) && (columns(amX[0]) == iC));
}

savetab(const sFile, const asrlab, const asclab, const mX, ...)
{
  decl va, bAppend, bHor, bVer, sFmt, asrlab0, asclab0, mX0,  
       sBase, sDir, i, j, fh, ir, iR, iC;
  
  iR= rows(mX);
  iC= columns(mX);
  asrlab0= array(asrlab);
  asclab0= array(asclab);
  mX0= mX;
  
  #ifdef DEBUG
    print ("In savetab, saving ", sFile, " with ", "%c", asclab0, "%r",
           asrlab0, mX, "size: ", iR, " x ", iC);
  #endif  

  // Prepare labels
  if (sizeof(asrlab0) < iR)
    asrlab0~= new array [iR-sizeof(asrlab0)];
  for (i= j= 0; i < iR; ++i)
    if (sizeof(asrlab0[i]) < 1)
      asrlab0[i]= sprint("NAL", j++);    
  if (sizeof(asclab0) < iC)
    asclab0~= new array [iC-sizeof(asclab0)];
  for (i= j= 0; i < iC; ++i)
    if (sizeof(asclab0[i]) < 1)
      asclab0[i]= sprint("NAL", j++);
    
  if ((iR > sizeof(asrlab0)) || (iC > sizeof(asclab0)))
    {
      oxwarning("Incorrect sizes of labels vs matrix, returning");
      return 0;
    }  
  
  bAppend= bHor= bVer= FALSE;
  va= va_arglist();
  if (sizeof(va) > 0)
    {
      sFmt= va[0];
      bAppend= (strifind(sFmt, "a") > -1);
      bHor= (strifind(sFmt, "h") > -1);
      bVer= (strifind(sFmt, "v") > -1);
    }      

  ir= 1;
  if (bAppend)  
    ir= loadtab(sFile, &asrlab0, &asclab0, &mX0, TRUE) 
          && combinetab(&asrlab0, &asclab0, &mX0, 
               array(asrlab), array(asclab), mX, bHor, bVer);
    
  // Get file names  
  i= strfind(sFile, ".tab");
  j= max(strfindr(sFile, "/"), strfindr(sFile, "\\"));
  if (i-1 < j+1)  // Return if filename has no base-part
    {
      oxwarning("Incorrect filename, returning");
      return 0;
    }  
  
  sBase= sFile[j+1:i-1];
  sDir= j > 0 ? sFile[0:j] : "";
  
  // Write labels
  fh= fopen(sFile, "w");
  if (!isfile(fh))
    {
      oxwarning("Cannot open file for writing, returning");
      return 0;
    }  
//   println ("Writing tab file");    
  fprintln (fh, "oxutils tab 100");  
  fprintln (fh, "%8.0f", rows(mX0), "%8.0f", columns(mX0));
  fprintln (fh, "time ", timestr(today()));
  fprintln (fh, "data");
  fprintln (fh, ">", sBase, ".fmt");
  fprintln (fh, "row labels");
  for (i= 0; i < sizeof(asrlab0); ++i)
    fprintln (fh, ">", asrlab0[i]);
  fprintln (fh, "column labels");
  for (i= 0; i < sizeof(asclab0); ++i)
    fprintln (fh, ">", asclab0[i]);
  fprintln (fh, "end");
  fclose(fh);
    
//   println ("Writing data file");    
  // Write data  
  return savemat(sDir~sBase~".fmt", mX0);
}
