/*
**  PrintMatrix(const fh, const s0, const aPrFmt, const acLabs, const arLabs,
**              const mX, const iHLine)
**  PrintMatrix(const fh, const s0, const aPrFmt, const acLabs, const arLabs,
**              const mX, const iHLine, const iHLine2, const bEnvironment)
**
**  Purpose:
**    Print a latex matrix to a file and/or to the screen
**
**  Inputs:
**    fh           File handle (if 0, writing to screen)
**    s0           String, for including in upper left corner of table
**    aPrFmt       Array with printing format, see print(). No & needed
**    acLabs       Array with column labels
**    arLabs       Array with row labels
**    mX           Data matrix
**    iHLine       Integer, indicating whether table should start & end with 
**                 one or more lines
**    iHLine2      Integer, indicating whether columns labels should be 
**                 separated with one or more lines from the matrix elements
**    bEnvironment Boolean indicating whether begin & end tabular environment
**                 should be printed
**
**  For both iHLine* inputs: Value indicates the number of lines to use
**
**  Output:
**    <file>  fh!=0: File pointed to by fh has a latex table appended
**            fh==0: Matrix gets printed on the screen
**
**  Authors:
**    Charles Bos and Patrick Houweling
** 
**  History:
**    26/10/2000: 1st version by Charles
**    16/06/2001: improved by Patrick:
**                - \begin{tabular} and \end{tabular} can be written to the file
**                - lines can be single or double
**                - lines can appear at the begin of the table, at the end, 
**                  and/or below the column headers
**                - only write to the screen if fh==0, otherwise only write
**                  to the file
**    17/10/2001: changed by Charles to get rid of a typo
*/
#ifndef PRINTMAT_INCLUDED
  #define PRINTMAT_INCLUDED
  #include <oxstd.h>

// Function declarations, small routines included below
printmat_hline(const fh, const iHLine);
printmat_printline(const fh, const sLine);

// Main routine
PrintMatrix(const fh, const s0, const aPrFmt, const acLabs, const arLabs,
            const mX, const iHLine, ...)
{
  decl sLine, i, j, nC, nR, nF, alPrFmt, alcLabs, alrLabs, iwL, srFMT,
       iHLine2, bEnvironment, va;

  iHLine2= bEnvironment= 0;       
  va= va_arglist();
  if (sizeof(va) > 0)
    iHLine2= va[0];
  if (sizeof(va) > 1)
    bEnvironment= va[1];

  // get matrix size
  nC= sizec(mX);
  nR= sizer(mX);
  
  // get print format & labels
  alPrFmt= aPrFmt;
  alcLabs= acLabs;
  alrLabs= arLabs;
  if (isstring(aPrFmt))
    alPrFmt= {aPrFmt};
  if (isstring(acLabs))
    alcLabs= {acLabs};
  if (isstring(arLabs))
    alrLabs= {arLabs};
  nF= sizeof(alPrFmt);

  // fix width of row label
  iwL= sizeof(s0);     
  for (i= 0; i < min(nR, sizeof(alrLabs)); ++i)
    iwL= max(1, iwL, sizeof(alrLabs[i]));
  srFMT= sprint("%", iwL, "s");

  // begin environment
  if (bEnvironment)
    {
      sLine= "";
      sLine = "\\begin{tabular}{l";
      for(i= 0; i < nC; ++i)
        sLine~= "r";
      sLine~="}";  
      printmat_printline(fh, sLine);
    }

  // line(s) above column headers
  printmat_hline(fh, iHLine);

  // column headers
  sLine= sprint(srFMT, s0);
  for (i= 0; i < min(sizeof(alcLabs), nC); ++i)
    sLine~=sprint(" & ", alcLabs[i]);
  if (sizeof(sLine))
    sLine ~= " \\\\";

  // print begin stuff
  printmat_printline(fh, sLine);
  
  // line(s) below column headers
  printmat_hline(fh, iHLine2);

  // matrix elements
  for (i= 0; i < nR; ++i)
    {
      sLine= "";
      if (i < sizeof(alrLabs))
        sLine~= sprint(srFMT, alrLabs[i]);
 
      for (j= 0; j < nC; ++j)
        if (isnan(mX[i][j]))
          sLine~=" & ";
        else
          sLine~= sprint(" & ", alPrFmt[imod(j, nF)], mX[i][j]);
      if (sizeof(sLine))
        sLine ~= " \\\\";
      printmat_printline(fh, sLine);
    }

  // bottom line
  printmat_hline(fh, iHLine);

  // end environment
  if (bEnvironment)
    printmat_printline(fh, "\\end{tabular}");
}

/*
**  printmat_printline(const fh, const sLine)
**
**  Purpose:
**    Print the line to the file or the screen
*/
printmat_printline(const fh, const sLine)
{
  if (sizeof(sLine) > 0)
    {
      if (isfile(fh))
        fprintln (fh, sLine);
      else
        println (sLine);
    }
}

/*
**  printmat_hline(const fh, const iHLine)
**
**  Purpose:
**    Print the latex hline commands
*/
printmat_hline(const fh, const iHLine)
{
  decl sLine, i;

  // line(s) below column headers
  sLine= "";
  for (i= 0; i < iHLine; ++i)
    sLine~= "\\hline";

  printmat_printline(fh, sLine);
}
#endif
