/*
**  Library TrackTime()
**
**  Purpose:
**    Measure time spent in separate routines
**
**  Date:
**    29/12/2002
**
**  Author:
**    Charles Bos
*/
#ifndef TRACKTIME_INCLUDED
  #define TRACKTIME_INCLUDED
  #include <oxstd.h>

enum
  { TT_NAMES, TT_TIME, TT_TIMEOLD, TT_I, TT_ROUTINE, 
    TT_FMT, TT_TOTAL }
// Static declarations
static decl s_TT= {}, s_TT_c= 0;

/*
**  s_Track_AddClass(const iI)
**
**  Purpose:
**    Enlarge the set of tracking classes
*/
s_Track_AddClass(const iI)
{
  decl i;
  for (i= s_TT_c; i <= iI; ++i)
    {
      s_TT~= {new array [TT_TOTAL]};
      s_TT[i][TT_I]= -1;
      s_TT[i][TT_TIME]= <>;
      s_TT[i][TT_ROUTINE]= FALSE;
      s_TT[i][TT_NAMES]= {};
      s_TT[i][TT_FMT]= 8;
      ++s_TT_c;
    }  
}

/*
**  s_Track_AddFunc(const iI, const x)
**
**  Purpose:
**    Add an extra function
*/
s_Track_AddFunc(const iI, const x)
{
  decl i;
  
  s_Track_AddClass(iI);

  for (i= sizeof(s_TT[iI][TT_NAMES]); i <= x; ++i)
    {
      s_TT[iI][TT_NAMES]~= {sprint("Fn ", i+1)};  
      s_TT[iI][TT_TIME]~= 0;
      s_TT[iI][TT_FMT]= max(s_TT[iI][TT_FMT], sizeof(s_TT[iI][TT_NAMES][i]));
    }
}
/*
**  s_Track_Readva(const va)
**
**  Purpose:
**    Read the variable argument list
*/
s_Track_Readva(const va)
{
  decl iI;

  iI= sizeof(va) ? va[0] : 0;
  
  s_Track_AddClass(iI);
  
  return iI;
}

/*
**  TrackInit(const aSNames, ...)
**
**  Purpose:
**    Initialise the tracking of the routines
**
**  Input:
**    asNames   if array: Series of names of routines. Initializes
**              if integer: Reset timings, do not change names
**    iI    integer, optional index of set of routines to count
*/
TrackInit(const asNames, ...)
{
  decl i, iI;

  iI= s_Track_Readva(va_arglist());

  if (isarray(asNames))
    {
      s_TT[iI][TT_NAMES]= asNames;
      s_TT[iI][TT_TIME]= zeros(1, sizeof(asNames));
      s_TT[iI][TT_I]= -1;
      s_TT[iI][TT_FMT]= 8;
      for (i= 0; i < sizeof(asNames); ++i)
        s_TT[iI][TT_FMT]= max(s_TT[iI][TT_FMT], sizeof(s_TT[iI][TT_NAMES][i]));
    }
  else
    {
      s_TT[iI][TT_TIME]*= 0;
      s_TT[iI][TT_I]= -1;
    }
}

/*
**  TrackTime(const x, ...)
**
**  Purpose:
**    Measure time spent in separate routines
**
**  Input:
**	  x     if array: Series of names of routines. Initializes
**          if integer: Index for routine which is entered
**    iI    integer, optional index of set of routines to count
**
**  Output:
**    return value      Integer, previous routine index number
*/
TrackTime(const x, ...)
{
  decl iOld, tim, i, iI;

  iI= s_Track_Readva(va_arglist());
  iOld= -1;

  if (isarray(x))
    TrackInit(x, iI);
  else
    {
      s_Track_AddFunc(iI, x);
        
      iOld= s_TT[iI][TT_I];
      tim= timer();

      if ((iOld >= 0) && (iOld < sizerc(s_TT[iI][TT_TIME])))
        s_TT[iI][TT_TIME][iOld] += tim - s_TT[iI][TT_TIMEOLD];

      if (s_TT[iI][TT_ROUTINE])
		if (x >= 0)
          println ("Entering ", s_TT[iI][TT_NAMES][x]);
		else if (iOld > 0)
          println ("Leaving ", s_TT[iI][TT_NAMES][iOld]);

      s_TT[iI][TT_I]= x;
      s_TT[iI][TT_TIMEOLD]= tim;
    }

  return iOld;
}

/*
**  TrackRoutine(bTrack, ...)
**
**  Purpose:
**    Indicate if routines which are entered should be tracked
**    through a message on the screen
**
**  Input:
**    bTrack    Boolean, 1 if message is to appear in TrackTime each time
**          a new routine is started, 0 otherwise.
**
*/
TrackRoutine(const bTrack, ...)
{
  decl iI;

  iI= s_Track_Readva(va_arglist());
  s_TT[iI][TT_ROUTINE]= bTrack;
}

/*
**  TrackReportI(const iI)
**
**  Purpose:
**    Report the time spent in the routines
**
**  Inputs:
**    None
**
**  Outputs:
**    Screen	Information on total and fraction of time spent in routines
*/
TrackReportI(const iI)
{
  decl i, bR, viSel, aPrFmt;

  if (sizerc(s_TT[iI][TT_NAMES]) > 0)
    {
      // Flush time
      bR= s_TT[iI][TT_ROUTINE];
      TrackRoutine(FALSE, iI);
      i= TrackTime(-1, iI);
      TrackTime(i, iI);
      TrackRoutine(bR, iI);
 
      viSel= range(0, sizerc(s_TT[iI][TT_NAMES])-1);
      if (sumr(s_TT[iI][TT_TIME]) > 0)
        viSel= selectifc(viSel, s_TT[iI][TT_TIME] .> 0);

      println ("Time spent in routines ");
      aPrFmt= sprint("%", s_TT[iI][TT_FMT], "s");
      for (i= 0; i < sizerc(viSel); ++i)
        print (aPrFmt, s_TT[iI][TT_NAMES][viSel[i]]);
      print ("\n");  
      aPrFmt= sprint("%", s_TT[iI][TT_FMT], ".2f");
      for (i= 0; i < sizerc(viSel); ++i)
        print (aPrFmt, s_TT[iI][TT_TIME][viSel[i]]/100);
      print ("\n");  
      for (i= 0; i < sizerc(viSel); ++i)
        print (aPrFmt, double(s_TT[iI][TT_TIME][viSel[i]]/sumr(s_TT[iI][TT_TIME])));
      print ("\n");  
    }
}

/*
**  TrackReport()
**
**  Purpose:
**    Report the time spent in the routines
**
**  Inputs:
**    None
**
**  Outputs:
**    Screen    Information on total and fraction of time spent in routines
*/
TrackReport()
{
  TrackReportI(0);
}  
#endif
