/*
**  SSFSVEst
**
**  Purpose:
**    Estimate the models for the inflation or ip data using SSFSV
**
**  Date:
**    18/3/2004, update 21/6/09, 22/3/11
**
**  Author:
**    Charles Bos
*/
#include <oxstd.h>      // Include the Ox standard library header
#include <include/choosedraw.h>

// Import the full (adapted) oxutils package from here
#import "./packages/oxutils/oxutils"
#import "include/ssfsv"

DrawStates(const ssfsv, const sOutbase, const mStsm);

#include "simox.dec"

main()
{
  decl ssfsv, vY, vP, vP_New, ir, dLnPdf, sOutbase, iSim, 
       dTime, vStat, asLab, mP, mRes, vSel, i, iZoom, va, bEstim, mLU;

  dTime= timer();
  va= arglist();
  iSim= 250;
  bEstim= TRUE;
  ReadArg(&g_bTrans, "trans", 1);
  ReadArg(&iSim, "sim", 1);
  ReadArg(&bEstim, "noest", 0);
  bEstim= !bEstim;

  sOutbase= sprint(g_Outbase, "r", g_bTrans, "s", iSim);
  fopen(sOutbase~".log", "l");
  println ("SSFSVEst\n-------");
  println ("Arguments:");
  println ("  noest   skip estimation of model");
  println ("  trans 1/0    Use (1) or don't (0) use transformation");
  println ("  sim xxx      Use xxx simulations in the importance sampler");

  setseed(g_iSeed);
  ssfsv= new SSFSV();
  ssfsv.LoadIn7(g_Datafile);

  ssfsv.Select(Y_VAR, {g_yVar, 0, 0});
  ssfsv.SetSelSample(g_Sample[0], g_Sample[1], g_Sample[2], g_Sample[3]);

  vY= ssfsv.GetVarByIndex(Y_VAR);
  ssfsv.Info();

  ssfsv.SetMethod(g_iMethod);
  ssfsv.ChooseTransform(g_bTrans);
  ssfsv.ChooseModel(g_mStsm);
  ssfsv.ChooseSVRep(-1, iSim);

  vP= ssfsv.GetFreePar();
  ir= ssfsv.SSFSVLikelihood(vP, &dLnPdf, 0, 0);
  println ("Estimating model ", g_sModel);
  print ("Initial likelihood: ", sizerc(vY)*dLnPdf, " at ",
         "%c", ssfsv.GetFreeParNames(), vP');

  if (bEstim)
    ssfsv.Estimate();

  ssfsv.SaveGraphics(sOutbase);
  mRes= ssfsv.TestAuxResiduals(ones(1, 4), ones(1,4)~0, 3.5, FALSE);
  mRes= ssfsv.TestGraphics(1, 1, 0, 12);
  mRes= ssfsv.TestGraphics(1, 0, 0, 12);
  mRes= ssfsv.TestResiduals(1~1~0~1~1~1~1~1~1~1, 60, TRUE);

  SetDrawWindow("");
  DrawAcf(0, mRes|sqr(mRes), {"Residuals", "Squared residuals"}, 60,
          TRUE, FALSE, TRUE, 2, FALSE, TRUE);
  SaveDrawWindow(sOutbase~"res2"~CH_FIGEXT);
  ShowDrawWindow();
  SetDrawWindow("");  
  
  vP= ssfsv.GetFreePar();
  if (rows(g_mWeight) == 0)
    g_mWeight= ssfsv.ObsYear(sizerc(vY)/2)~ssfsv.ObsPeriod(sizerc(vY)/2);
  for (i= 0; i < rows(g_mWeight); ++i)
    {
      iZoom= ssfsv.GetIndex(g_mWeight[i][0], g_mWeight[i][1]);
      ssfsv.SaveGraphics(sprint(sOutbase, "w", i));
      ssfsv.SetSampleZoom(iZoom, iZoom);
      ssfsv.TestWeightGraphs(vP, 0);
    }
  // Reset the sample  
  ssfsv.SetSampleZoom(-1, -1);
  DrawStates(ssfsv, sOutbase, g_mStsm);
  
  // Plot the parameters at the end, as it disturbs the computations
  //   of the vhFiltSV
  ssfsv.TestParameterGraphs();

//  TrackReport();
  println ("Time passed: ", timespan(dTime));
}  


/*
**  DrawStates(const ssfsv, const sOutbase, const mStsm)
**
**  Purpose:
**    Get the elements I want...
*/
DrawStates(const ssfsv, const sOutbase, const mStsm)
{
  decl i, im, vP, vY, mR, amStates, vhMeanSV, vSample, mOut;

  vP= ssfsv.GetFreePar();
  vY= ssfsv.GetGroup(Y_VAR)';
  i= ssfsv.GetSelStart();
  mOut= <>;
  vSample= ssfsv.ObsYear(i)~ssfsv.ObsPeriod(i)~ssfsv.GetFrequency();

  vhMeanSV= ssfsv.GetHMean();
  amStates= ssfsv.ExtractStates(ST_SMO);

  // if no AR1, Mu is first state element, else last
  i= (mStsm[][0] != CMP_AR1) ? CMP_LEVEL : CMP_AR1;
  SetDrawWindow("States");
  if (sizeof(amStates[0][i]) > i)
    {
	  DrawTMatrix(0, amStates[0][i], "{/Symbol m}", vSample[0], vSample[1], vSample[2]);
	  mOut|= amStates[0][i];
	}
	
  // if no AR1, season is second state element, else first
  i= (mStsm[][0] != CMP_SEAS_TRIG) ? CMP_SEAS_DUMMY : CMP_SEAS_TRIG;
  DrawTMatrix(1, amStates[0][i], "{/Symbol g}", vSample[0], vSample[1], vSample[2]);
  mOut|= amStates[0][i];

  i= strfind(ssfsv.GetFreeParNames(), "s(Irr)");
  DrawTMatrix(2, vP[i]*exp(vhMeanSV/2), "{/Symbol s_e} exp(h_t/2)",
              vSample[0], vSample[1], vSample[2]);
  mOut|= vP[i]*exp(vhMeanSV/2);

  DrawAdjust(ADJ_AREAMATRIX, 1, 3);              
  SaveDrawWindow(sOutbase~"st"~CH_FIGEXT);
  ShowDrawWindow();
  SetDrawWindow("");
  
  return mOut;
}
