/*
**  SSFSVSVIS
**
**  Purpose:
**    Estimate an SV model using Importance sampling through SSFSV
**
**  Date:
**    22/3/2002
**
**  Author:
**    Charles Bos
*/
#include <oxstd.h>      // Include the Ox standard library header
#include <include/choosedraw.h>

#import "packages/oxutils/oxutils"
#import "include/ssfsv"

#include "simox.dec"

Initialize(const va);

main()
{
  decl ssfsv, vSV, nI, vY, vH, ir, iK, vP, vP0, vPInit, dPhi, mRes, i, j, mSum, 
       fh, vS, asLab, asLabSel, mOut, asNames, vPSel, vLSel, iSSV, iPhiSV, 
       sOutbase, sCommand, va, nRep, mGenrSV, mEstSV, iT;

  [ssfsv, mGenrSV, mEstSV, sOutbase, nRep, iT]= Initialize(arglist());
  asLabSel= {"Normality-DH:", "Q:", "ARCH:", sprint("H(",
             double(round(iT/3)), "):"), "LL"};

  ssfsv.ChooseModel(g_mStsm|mGenrSV);
  vPInit= ssfsv.GetFreePar();

  ssfsv.InitPar();

  // Find indices of parameters
  ssfsv.ChooseModel(g_mStsm);
  asNames= ssfsv.GetFreeParNames();
  ssfsv.ChooseModel(g_mStsm|mEstSV);
  vPSel= strfind(ssfsv.GetFreeParNames(), asNames);
  iSSV= strfind(ssfsv.GetFreeParNames(), "s(SV)");
  iPhiSV= strfind(ssfsv.GetFreeParNames(), "phi(SV)");
    
  ssfsv.SetPrint(2);
  infoinit(nRep, 5);
  i= 0;
  while (i < nRep)
    {
      info(i, nRep);
      println ("Choosing model with vSV ", mGenrSV, "and parameters ",
               vPInit');
      ssfsv.ChooseModel(g_mStsm|mGenrSV);
      ssfsv.GenerateSsf(vPInit, "Y");
//      ssfsv.SetFreePar(vPInit);

      if (sizerc(g_bEstSV))
        {
          println ("Estimating model without SV");
          ssfsv.ChooseModel(g_mStsm);
          ir= ssfsv.Estimate();
          vP0= ssfsv.GetFreePar();

          ssfsv.ChooseModel(g_mStsm|mEstSV);
          vP= ssfsv.GetFreePar();
          vP[vPSel]= vP0;
          
          ssfsv.SetFreePar(vP);
        }
      else
        ssfsv.ChooseModel(g_mStsm|mEstSV);

      println ("Estimating model with bEstSV ", g_bEstSV);
      ir= ssfsv.Estimate();
      
      if (ir)
        {
          // Retrieve the parameter estimates  
          vP= ssfsv.GetFreePar();
          vS= ssfsv.MapParToFree(ssfsv.GetStdErr());
          [asLab, mSum]= ssfsv.TestSummary(0);
          vLSel= strfind(asLab, asLabSel);
          if (!(vLSel > 0))
            {
              unlockfile(sOutbase);
              oxrunerror(sprint("Labels ", asLabSel, " not found, Sel=",
                         vLSel, "SSFSV returns ", asLab), 1);  
            }

          mRes= vP|vS|vec(mSum[vLSel][0])|ir;
          fh= fopen(sOutbase~"res.fmt", "abf");
          if (!columns(fh) || (columns(fh) == sizerc(mRes)))
            fwrite(fh, mRes');
          else
            {
              print ("Size of fh, mRes: ", columns(fh)~sizerc(mRes));
              unlockfile(sOutbase);
              oxrunerror(sprint("Error: Incorrect size of results matrix ",
                         sOutbase, "res.fmt"), 1);  
            }  
          fh= fclose(fh);
//           println ("Added results", mRes', "with indices ", vLSel, 
//                    "from labels ", "%r", asLab, mSum);
          
          ++i;
        }
      else
        println("No summary statistics computed");    
    }
  info(nRep, nRep);

  ssfsv.SetPrint(TRUE);
  if (nRep > 0)
    unlockfile(sOutbase);
}  

/*
**  Initialize
**
*/
Initialize(const va)
{
  decl nI, ssfsv, mGenrSV, mEstSV, mRes, nRep, sOutbase, iT, mPhi,
       mOmega, mSigma;
  
  println("SVISSim V7\n----------");
  println("Program calling: \n  oxl svissim [nRep [nData [bGenrSV [bEstSV]]]]");
  setseed(0);
  if ((sizeof(va) > 1) && sscan(va[1], "%i", &nI))
    g_nRep= nI;
  if ((sizeof(va) > 2) && sscan(va[2], "%i", &nI))
    g_Sample[3]= nI;
  if ((sizeof(va) > 3) && sscan(va[3], "%i", &nI))
    g_bGenrSV= nI;
  if ((sizeof(va) > 4) && sscan(va[4], "%i", &nI))
    g_bEstSV= nI;
  if ((sizeof(va) > 5) && (va[5] == "nofig"))
    DrawAdjust(ADJ_PAPERSCALE+4, FALSE);
  if ((g_Sample < 100) || (max(g_bGenrSV, g_bEstSV) > 2))  
    oxrunerror(sprint("Error: Incorrect parameters, nRep= ", g_nRep,
      ", nSample= ", g_Sample[3], ", bGenrSV= ", g_bGenrSV, ", bEstSV= ", 
      g_bEstSV), 1);
    
  ssfsv= new SSFSV();
  ssfsv.Debug(0);
  ssfsv.Create(g_Sample[0], g_Sample[1], g_Sample[2],
               ceil(g_Sample[3]/g_Sample[0]), g_Sample[4]);
  ssfsv.SetSelSample(-1, -1, -1, -1);
  ssfsv.Deterministic(0);
  ssfsv.Append(ssfsv.GetVar("Constant"), {"Y"});
  ssfsv.Select(Y_VAR, {"Y", 0, 0});
  ssfsv.Info();
  ssfsv.SetMethod(ESTM_CML);

  mGenrSV= g_bGenrSV == 0 ? <> :                           //      No SV
           g_bGenrSV == 1 ? CMP_SV~g_vSV[0]~0~g_vSV[1] :   //      SV
           CMP_SV~g_vSV[0]~0~1;                            //      ISV
  mEstSV= g_bEstSV == 0 ? <> :                             //      No SV
          (g_bEstSV == 1) && !g_bGenrSV ? 
                          CMP_SV~.5~0~0.01 :               //      SV
          (g_bEstSV == 1) && g_bGenrSV ? 
                          CMP_SV~g_vSV[0]~0~g_vSV[1] :     //      SV
          (g_bEstSV == 2) && !g_bGenrSV ? 
                          CMP_SV~.5~0~1 :                  //      ISV
          CMP_SV~g_vSV[0]~0~1;                             //      ISV
          
  sOutbase= sprint(g_Outbase, g_bGenrSV, g_bEstSV, "_", g_Sample[3]);
  mRes= loadmat(sOutbase~"res.fmt");
  if (mRes == 0)
    fclose(fopen(sOutbase~"res.fmt", "wbf"));
  nRep= max(g_nRep-rows(mRes), 0);

  if (nRep == 0)
    {
      println("Nothing left to do, quitting...");
      exit(0);
    }  
  else if (!lockfile(sOutbase))
    {
      // Check if the file is in use
      print("Quitting...");
      exit(0);
    }  

  println ("Running ", nRep, " estimations from ", rows(mRes), " to ", g_nRep);
  println(g_sModel);
  println(g_bGenrSV==0 ? "Not generating SV" : 
          g_bGenrSV==1 ? "Generating SV" : "Generating ISV");
  println(g_bEstSV==0 ? "Not estimating SV" : 
          g_bEstSV==1 ? "Estimating SV" : "Estimating ISV");
  println("Output in file ", sOutbase, "res.fmt");

  if (g_mStsm[][0] != CMP_ARMA)
    GetSsfStsm(g_mStsm, &mPhi, &mOmega, &mSigma);
  else
    GetSsfArma(g_mStsm[][2], g_mStsm[][3], g_mStsm[0][1], 
               &mPhi, &mOmega, &mSigma);
  iT= sizerc(ssfsv.GetVar("Constant")) - sumc(vecr(mSigma .== -1));

  return {ssfsv, mGenrSV, mEstSV, sOutbase, nRep, iT};          
}
