#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
estnorm.py

Purpose:
    Estimate a normal regression model, using lambda function

Version:
    1       Following estnorm.ox, using 1d parameter vectors
    2       Getting rid of ppectr.py/GetCovML
    3       Using vector of loglikelihood, with score

Date:
    2017/8/21, 2019/6/3, 2019/8/30

Author:
    Charles Bos
"""
###########################################################
### Imports
import numpy as np
import pandas as pd
#import matplotlib.pyplot as plt
import scipy.optimize as opt
import math

###########################################################
### Get hessian and related functions
from lib.grad import *

###########################################################
def GetPars(vP):
    """
    Purpose:
      Read out the parameters from the vector

    Inputs:
      vP        iK+1 vector with sigma and beta's

    Return value:
      dS        double, sigma
      vBeta     iK vector, beta's
    """
    iK= np.size(vP)-1
    # Force vP to be a 1D matrix
    vP= vP.reshape(iK+1,)
    dS= vP[0]   # np.fabs(vP[0])
    vBeta= vP[1:]

    return (dS, vBeta)

###########################################################
def GetParNames(iK):
    """
    Purpose:
      Construct names for the parameters from the vector

    Inputs:
      iK        integer, number of beta's

    Return value:
      asP       iK array, with strings "sigma", "b1", ...
    """
    asP= ["B"+str(i+1) for i in range(iK)]
    asP= ["Sigma"] + asP

    return asP

###########################################################
def GenrX(iN, iK):
    """
    Purpose:
      Generate regressors, constant + uniforms

    Inputs:
      iN        integer, number of observations
      iK        integer, number of regressors

    Return values:
      mX        iN x iK matrix of regressors, constant + uniforms
    """
    mX= np.hstack([np.ones((iN, 1)), np.random.rand(iN, iK-1)])

    return mX
###########################################################
def GenrY(vP, mX):
    """
    Purpose:
      Generate regression data

    Inputs:
      vP        iK+1 vector of parameters, sigma and beta
      mX        iN x iK matrix of regressors

    Return values:
      vY        iN vector of data
    """
    iN= mX.shape[0]
    (dS, vBeta)= GetPars(vP)
    vY= mX@vBeta + dS * np.random.randn(iN)

    return vY

###########################################################
### vLL= LnLRegr(vP, vY, mX)
def LnLRegr(vP, vY, mX):
    """
    Purpose:
        Compute loglikelihood of regression model

    Inputs:
        vP      iK+1 1D-vector of parameters, with sigma and beta
        vY      iN 1D-vector of data
        mX      iN x iK matrix of regressors

    Return value:
        vLL     iN vector, loglikelihood
    """
    (iN, iK)= mX.shape
    if (np.size(vP) != iK+1):         # Check if vP is as expected
        print ("Warning: wrong size vP= ", vP)

    (dSigma, vBeta)= GetPars(vP)
    vLL= -math.inf*np.ones(iN)    # Value, just in case
    if (dSigma <= 0):
        print ("x", end="")
        return vLL

    vE= vY - mX @ vBeta
    vLL= -0.5*(np.log(2*np.pi) + 2*np.log(dSigma) + np.square(vE/dSigma))

    print (".", end="")             # Give sign of life

    return vLL

###########################################################
### mLLS= LnLRegr_Sc(vP, vY, mX)
def LnLRegr_Sc(vP, vY, mX):
    """
    Purpose:
        Compute score of loglikelihood of regression model

    Inputs:
        vP      iK+1 1D-vector of parameters, with sigma and beta
        vY      iN 1D-vector of data
        mX      iN x iK matrix of regressors

    Return value:
        mLLS    iN x iK+1 matrix, score
    """
    (iN, iK)= mX.shape
    if (np.size(vP) != iK+1):         # Check if vP is as expected
        print ("Warning: wrong size vP= ", vP)

    (dSigma, vBeta)= GetPars(vP)
    mLLS= -math.inf*np.ones((iN, iK+1))
    if (dSigma <= 0):
        print ("x", end="")
        return mLLS

    vE= vY - mX @ vBeta
    # vLL= -0.5*(np.log(2*np.pi) + 2*np.log(dSigma) + np.square(vE/dSigma))

    print ("g", end="")             # Give sign of life

    mLLS[:,0]= -1/dSigma + np.square(vE)/dSigma**3
    mLLS[:,1:]= vE.reshape(-1, 1)*mX / np.square(dSigma)

    return mLLS

###########################################################
### (vP, vS, dLL, sMess)= EstimateRegr(vY, mX)
def EstimateRegr(vY, mX):
    """
    Purpose:
      Estimate the regression model

    Inputs:
      vY        iN vector of data
      mX        iN x iK matrix of regressors

    Return value:
      vP        iK+1 vector of optimal parameters sigma and beta's
      vS        iK+1 vector of standard deviations
      dLL       double, loglikelihood
      sMess     string, output of optimization
    """
    (iN, iK)= mX.shape
    vP0= np.ones(iK+1)        # Get (bad...) starting values

    # vB= np.linalg.lstsq(mX, vY)[0]
    # vP0= np.vstack([[[1]], vB])

    # Create lambda function returning NEGATIVE AVERAGE LL, as function of vP only
    AvgNLnLRegr= lambda vP: -np.mean(LnLRegr(vP, vY, mX), axis=0)
    AvgNLnLRegr_Sc= lambda vP: -np.mean(LnLRegr_Sc(vP, vY, mX), axis=0)

    dLL= -iN*AvgNLnLRegr(vP0)
    print ("Initial LL= ", dLL, "\nvP0=", vP0)

    # Check score
    vSc0= AvgNLnLRegr_Sc(vP0)
    vSc1= opt.approx_fprime(vP0, AvgNLnLRegr, 1e-5*np.fabs(vP0))
    vSc2= gradient_2sided(AvgNLnLRegr, vP0)
    print ("\nScores:\n",
           pd.DataFrame(np.vstack([vSc0, vSc1, vSc2]), index=["Analytical", "grad_1sided", "grad_2sided"]))

    dErr= np.sqrt(np.mean((vSc0 - vSc1)**2, axis= 0))
    if (dErr > 1e-3):
        print ("Warning: Implementation of gradient gives error= ", dErr)

    # res= opt.minimize(AvgNLnLRegr, vP0, method="BFGS")
    # Optimize with jacobian
    res= opt.minimize(AvgNLnLRegr, vP0, method="BFGS",
            jac=AvgNLnLRegr_Sc)

    vP= res.x
    sMess= res.message
    dLL= -iN*res.fun
    print ("\nBFGS results in ", sMess, "\nPars: ", vP, "\nLL= ", dLL, ", f-eval= ", res.nfev)

    mHn= hessian_2sided(AvgNLnLRegr, vP)
    mH= -iN*mHn
    mS2= -np.linalg.inv(mH)
    vS= np.sqrt(np.diag(mS2))

    return (vP, vS, dLL, sMess)

###########################################################
### Output(mPPS, dLL, sMess, iN)
def Output(mPPS, dLL, sMess, iN):
    """
    Purpose:
      Provide output on screen
    """
    iK= mPPS.shape[1]-1
    print ("\n\nEstimation resulted in ", sMess)
    print ("Using ML with LL= %g, n=%i" % (dLL, iN) )

    print ("Parameter estimates:\n",
           pd.DataFrame(mPPS.T, index=GetParNames(iK), columns=["PTrue", "PHat", "s(P)"]))


###########################################################
### main
def main():
    vP0= [.1, 5, 2, -2]    #dSigma and vBeta together
    iN= 100
    iSeed= 1234

    #Generate data
    np.random.seed(iSeed)
    vP0= np.array(vP0)

    iK= vP0.size - 1
    mX= GenrX(iN, iK)
    vY= GenrY(vP0, mX)

    (vP, vS, dLnPdf, sMess)= EstimateRegr(vY, mX)
    Output(np.vstack([vP0, vP, vS]), dLnPdf, sMess, iN);

###########################################################
### start main
if __name__ == "__main__":
    main()
