#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
estnorm_tr.py

Purpose:
    Estimate a normal regression model, using lambda function

Version:
    1       Following estnorm.ox, using 1d parameter vectors
    tr      Use transformation

Date:
    2017/8/21, 2019/6/3

Author:
    Charles Bos
"""
###########################################################
### Imports
import numpy as np
import pandas as pd
#import matplotlib.pyplot as plt
import scipy.optimize as opt
import math

###########################################################
### Get hessian and related functions
from lib.grad import *

###########################################################
def GetPars(vP):
    """
    Purpose:
      Read out the parameters from the vector

    Inputs:
      vP        iK+1 vector with sigma and beta's

    Return value:
      dS        double, sigma
      vBeta     iK vector, beta's
    """
    iK= np.size(vP)-1
    # Force vP to be a 1D matrix
    vP= vP.reshape(iK+1,)
    dS= vP[0]
    vBeta= vP[1:]

    return (dS, vBeta)

###########################################################
def GetParNames(iK):
    """
    Purpose:
      Construct names for the parameters from the vector

    Inputs:
      iK        integer, number of beta's

    Return value:
      asP       iK array, with strings "sigma", "b1", ...
    """
    asP= ["B"+str(i+1) for i in range(iK)]
    asP= ["Sigma"] + asP

    return asP

###########################################################
def TransPar(vP):
    """
    Purpose:
      Transform the parameters for restrictions

    Inputs:
      vP        iP array of parameters, with sigma and beta's

    Return value:
      vPTr      iP array, with log(sigma) and beta's
    """
    vPTr= np.copy(vP)
    vPTr[0]= np.log(vP[0])

    return vPTr

###########################################################
def TransBackPar(vPTr):
    """
    Purpose:
      Transform the parameters back from restrictions

    Inputs:
      vPTr      iP array of parameters, with log(sigma) and beta's

    Return value:
      vP        iP array, with sigma and beta's
    """
    vP= np.copy(vPTr)
    vP[0]= np.exp(vPTr[0])

    return vP

###########################################################
### mX= GenrX(iN, iK)
def GenrX(iN, iK):
    """
    Purpose:
      Generate regressors, constant + uniforms

    Inputs:
      iN        integer, number of observations
      iK        integer, number of regressors

    Return values:
      mX        iN x iK matrix of regressors, constant + uniforms
    """
    mX= np.hstack([np.ones((iN, 1)), np.random.rand(iN, iK-1)])

    return mX

###########################################################
### vY= GenrY(vP, mX)
def GenrY(vP, mX):
    """
    Purpose:
      Generate regression data

    Inputs:
      vP        iK+1 vector of parameters, sigma and beta
      mX        iN x iK matrix of regressors

    Return values:
      vY        iN vector of data
    """
    iN= mX.shape[0]
    (dS, vBeta)= GetPars(vP)
    vY= mX@vBeta + dS * np.random.randn(iN)

    return vY

###########################################################
### vLL= LnLRegr(vP, vY, mX)
def LnLRegr(vP, vY, mX):
    """
    Purpose:
        Compute loglikelihood of regression model

    Inputs:
        vP      iK+1 1D-vector of parameters, with sigma and beta
        vY      iN 1D-vector of data
        mX      iN x iK matrix of regressors

    Return value:
        vLL     iN vector, loglikelihood
    """
    (iN, iK)= mX.shape
    if (np.size(vP) != iK+1):         # Check if vP is as expected
        print ("Warning: wrong size vP= ", vP)

    (dSigma, vBeta)= GetPars(vP)
    vLL= -math.inf*np.ones(iN)    # Value, just in case
    if (dSigma <= 0):
        print ("x", end="")
        return vLL

    vE= vY - mX @ vBeta
    vLL= -0.5*(np.log(2*np.pi) + 2*np.log(dSigma) + np.square(vE/dSigma))

    print (".", end="")             # Give sign of life

    return vLL

###########################################################
### (mPSS, dLL, sMess)= EstimateRegr(vY, mX)
def EstimateRegr(vY, mX):
    """
    Purpose:
      Estimate the regression model

    Inputs:
      vY        iN vector of data
      mX        iN x iK matrix of regressors

    Return value:
      mPSS      iK+1 matrix of optimal parameters sigma and beta's, standard errors based on hessian,
                and standard errors based on hessian + delta method
      dLL       double, loglikelihood
      sMess     string, output of optimization
    """
    (iN, iK)= mX.shape
    vP0= np.ones(iK+1)        # Get (bad...) starting values

    # Create lambda function returning NEGATIVE AVERAGE LL, as function of vP only
    AvgNLnLRegr= lambda vP: -np.mean(LnLRegr(vP, vY, mX), axis=0)
    # Use lambda function to transform vPTr back in place
    # AvgNLnLRegrTr= lambda vPTr: AvgNLnLRegr(TransBackPar(vPTr))                       # Option 1
    AvgNLnLRegrTr= lambda vPTr: -np.mean(LnLRegr(TransBackPar(vPTr), vY, mX), axis=0)   # Option 2

    vP0Tr= TransPar(vP0)
    dLL= -iN*AvgNLnLRegrTr(vP0Tr)
    print ("Initial LL= ", dLL, "\nvP0=", vP0)

    res= opt.minimize(AvgNLnLRegrTr, vP0Tr, method="BFGS")
    vPTr= np.copy(res.x)
    vP= TransBackPar(vPTr)   # Remember to transform back!

    # Get standard errors, straight from LL without transformation
    mHn= hessian_2sided(AvgNLnLRegr, vP)
    mH= -iN*mHn
    mS2h= -np.linalg.inv(mH)                      # Non-robust covariance

    # Notation of EQRM course...
    # mA= -hessian_2sided(AvgNLnLRegr, vP)        # Asymptotic E(H), adapt for negative LL
    # mAi= np.linalg.inv(mA)
    # mS2h= -mAi/iN                               # Non-robust covariance
    vSh= np.sqrt(np.diag(mS2h))
    # print ("\nst.dev, no transformation: ", vS)

    # Get standard errors, using delta method
    mHnTr= hessian_2sided(AvgNLnLRegrTr, vPTr)
    mHTr= -iN*mHnTr
    mS2Tr= -np.linalg.inv(mHTr)
    mG= jacobian_2sided(TransBackPar, vPTr)  # Evaluate jacobian at vPTr
    mS2hd= mG @ mS2Tr @ mG.T                 # Cov(vP)

    # Notation of EQRM course...
    # mA= -hessian_2sided(AvgNLnLRegrTr, vPTr)
    # mAi= np.linalg.inv(mA)
    # mS2Tr= -mAi/iN
    # mG= jacobian_2sided(TransBackPar, vPTr)  # Evaluate jacobian at vPTr
    # mS2hd= mG @ mS2Tr @ mG.T                 # Cov(vP)
    vShd= np.sqrt(np.diag(mS2hd))            # s(vP)

    sMess= res.message
    dLL= -iN*res.fun
    print ("\nBFGS results in ", sMess, "\nPars: ", vP, "\nLL= ", dLL, ", f-eval= ", res.nfev)
    print ("\nJacobian of transformation, mG=\n", mG, "\nmS2h:\n", mS2h, "\nmS2hd, delta method:\n", mS2hd)

    return (np.vstack((vP, vSh, vShd)).T, dLL, sMess)

###########################################################
### Output(mPPS, dLL, sMess, iN)
def Output(mPPSS, dLL, sMess, iN):
    """
    Purpose:
      Provide output on screen
    """
    iK= mPPSS.shape[1]-1
    print ("\n\nEstimation resulted in ", sMess)
    print ("Using ML with LL= %g, n=%i" % (dLL, iN) )

    print ("Parameter estimates:\n",
           pd.DataFrame(mPPSS, index=GetParNames(iK), columns=["PTrue", "PHat", "s(P)-h", "s(P)-hd"]))

###########################################################
### main
def main():
    vP0= [.1, 5, 2, -2]    #dSigma and vBeta together
    iN= 100
    iSeed= 1234

    # Generate data
    np.random.seed(iSeed)
    vP0= np.array(vP0)

    iK= vP0.size - 1
    mX= GenrX(iN, iK)
    vY= GenrY(vP0, mX)

    (mPSS, dLnPdf, sMess)= EstimateRegr(vY, mX)
    Output(np.hstack([vP0.reshape(-1, 1), mPSS]), dLnPdf, sMess, iN);

###########################################################
### start main
if __name__ == "__main__":
    main()
