#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
prettyprint.py

Purpose:
    Several options for pretty printing a matrix

Version:
    1       First start

Date:
    2017/9/29

Author:
    Charles Bos
"""
###########################################################
### Imports
import numpy as np
import pandas as pd

###########################################################
### printtex(mX, index=None, columns=None, formats="%.6g"):
def printtex(mX, index=None, columns=None, formats="%.6g"):
    """
    Purpose:
        Print a latex matrix, with column formatting

    Inputs:
        mX      iN x iK matrix of data
        index   iN array of row names
        columns iK array of column names
        formats single format string, or iK array of column formats, default= "%.6g"

    Author:
        Charles Bos
    """
    [iN, iK]= mX.shape

    bInd= (not index == None) and (len(index) >= iN)
    bCol= (not columns == None) and (len(columns) >= iK)

    if (isinstance(formats, str)):
        formats= [formats]*iK
    bFor= len(formats) >= iK

    iWl= 0
    if (bInd):
        vWl= [len(index[i]) for i in range(iN)]
        iWl= max(vWl)

    if (bCol):
        if (bInd):
            print (" "*iWl, " & ", end="")
        for j in range(iK):
            print (columns[j], " & " if j < iK-1 else "\\\n", end="")

    for i in range (iN):
        if (bInd):
            print (index[i], " & ", end= "")
        for j in range(iK):
            print (formats[j] % mX[i,j], " & " if j < iK-1 else "\\\n", end="")

###########################################################
### main
def main():
    # Magic numbers
    iN= 3
    iK= 4

    # Initialisation
    mX= np.random.randn(iN, iK)

    # Output
    print ("Standard output mX=\n", mX)
    print ("Pandas output mX=\n", pd.DataFrame(mX, columns=list('abcd'), index=list('ABC')))

    pd.options.display.float_format = '${:,.3f}'.format
    print ("Pandas formatted output mX=\n", pd.DataFrame(mX, columns=list('abcd'), index=list('ABC')))

    pd.options.display.float_format = None

    fnB= lambda x: "%6.3f" % x          # Format for parameter
    fnS= lambda x: "(%.2f)" % x         # Format for st.dev

    print ("Pandas latex formatted output mX=\n",
           pd.DataFrame(mX, columns=list('abcd'), index=list('ABC')).to_latex(formatters=[fnB, fnS, fnB, fnS]))

    print ("Local function, simplified printing")
    printtex(mX, formats=["%6.3f", "(%.2f)", "%.3f", "[%.3f]"], index= list('ABC'), columns= list('abcd'))



###########################################################
### start main
if __name__ == "__main__":
    main()
