%
%  EstStack_ML
%
%  Purpose:
%    Estimate a regression model using ML on the stackloss data set
%    using octave (not matlab...)
%
%  Inputs:
%    The program expects the file data/stackloss.txt to contain
%    the data
%
%  Dependencies:
%    This program depends on the optim package of 
%    octave. Download it, from
%      https://octave.sourceforge.io/optim/
%    start octave, and run e.g.
%       pkg install optim-XXX.tar.gz    
%    or install them through the package manager of your system, e.g.
%       apt-get install octave-optim
%        
%  Run:
%    source eststack_ml.m 
%
%  Author:
%    Charles Bos
%
%  Date:
%    8/9/2008, 22/8/2017
%
global s_vY s_mX;      % Globals for passing the data

% 
% [vY, mX]= LoadData(sData)
%
% Purpose:
%   Read the dataset, extract columns of data
% 
% Inputs:
%   sData       string, name of existing matrix-file
%
% Outputs:
%   vY          iN x 1 vector, stack loss data, from last row of data file
%   mX          iN x iK matrix, explanatory variable, remaining columns
%
function [vY, mX]= LoadData(sData)
  mData= load (sData);                          % Load the data
  iD= rows(mData);
  vY= mData(iD,:)';                             % Read out last row
  iN= rows(vY);
  mX= [ ones(iN, 1), mData(1:iD-1, :)'] ;       % Read out other rows
endfunction

%   AvgLnLiklRegr(vP)
% 
%   Purpose:
%     Compute the average log likelihood for a regression model
% 
%   Inputs:
%     vP        iK+1 x 1 vector of parameters; last parameter is standard
%               dev
%     s_vY      iN x 1 vector of data
%     s_mX      iN x iK matrix of explanatory variables
% 
%   Return value:
%     dLnPdf    negative average loglikelihood
function dLnPdf= AvgLnLiklRegr(vP)

  global s_vY s_mX;
  
  dLnPdf= Inf;
  iK= columns(s_mX);
  iN= rows(s_vY);
  vBeta= vP(1:iK);
  dS2= vP(iK+1)^2;
  
  vE= s_vY - s_mX*vBeta;
  if (vP(iK+1) > 0)
    dLnPdf= 0.5*(vE' * vE/(iN*dS2) + log(2*pi*dS2));
  endif
  
  % printf ("%8.2f\n", vP, dLnPdf);
  
endfunction

% InEqRestrP(vP)
% 
% Purpose:
%   Check the inequality restrictions on vP
function vQ= InEqRestrP(vP)

  iK= rows(vP)-1;
  vQ= vP(iK+1);        % Make sure dS > 0
endfunction

%  Estimate
%  Purpose:
%    Estimate the model using sqp maximization
%
function [mPS, dLnPdf]= EstimateSQP(vY, mX, vP0)
  global s_vY s_mX;

  s_vY= vY;
  s_mX= mX;

  [vP, dLnPdf, iInfo, iIter, iNFn, vLambda] = sqp (vP0, @AvgLnLiklRegr, [], @InEqRestrP);
  mPS= vP;

endfunction

%  Estimate
%  Purpose:
%    Estimate the model using maximization
%
function [mPS, dLnPdf]= EstimateBFGS(vY, mX, vP0)
  global s_vY s_mX;       % Indicate you'll be using the globals here

  s_vY= vY;
  s_mX= mX;
  iN= rows(vY);
  iK= columns(mX);

  aControl= {100,1,1,1};  % maxiters, verbosity, conv. reg., arg_to_min
  [vP, dLnPdf, iR] = bfgsmin ("AvgLnLiklRegr", {vP0}, aControl);
  mH= numhessian("AvgLnLiklRegr", {vP});
  mS2= inv(iN * mH);    % Notice you're minimizing, hence mH is pos def
  mPS= [vP, sqrt(diag(mS2))];
endfunction

function [mPS, dSigma2] = EstimateOLS(vY, mX)
  [vBeta, dSigma2, vR]= ols(vY, mX);                 % Run OLS on columns

  mS2= dSigma2 * inv(mX' * mX);
  mPS= [vBeta, sqrt(diag(mS2))];
endfunction

function Output(sMethod, mPS, dLnPdf, dSigma2)
  printf ("%s gives\n", sMethod);
  if (dLnPdf != 0)
    printf ("Likelihood:        %8.3f\n", dLnPdf);
  else    
    printf ("Residual sdev:     %8.3f\n", sqrt(dSigma2));
  endif    
  printf ("    Pars (sdev)\n");
  printf ("%8.3f (%.3f)\n", mPS');
endfunction

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% Main
% Load the data
[vY, mX]= LoadData("data/stackloss.txt");
iN= rows(vY);

% Get a starting value for vP
vP0= [zeros(columns(mX), 1) ; 1];

% Estimate using BFGS
[mPS, dLnPdf]= EstimateBFGS(vY, mX, vP0);
Output("BFGS", mPS, -iN*dLnPdf, 0);

% Estimate using OLS
[mPS, dSigma2]= EstimateOLS(vY, mX);
Output("OLS", mPS, 0, dSigma2);
