#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
np_newtonm.py

Purpose:
    Evaluate newton steps

Version:
    1       Simplifying without graphical output, from np_newton_show2.ox
    m       minimising instead of maximising

Date:
    2018/8/27

Author:
    Charles Bos
"""
###########################################################
### Imports
import numpy as np

###########################################################
### (vF, vG, vH)= fnF(vTheta)
def fnF(vTheta):
    """
    Purpose:
      Calculate target function, plus first and second derivative

    Inputs:
      vTheta    vector, value for evaluation

    Return value:
      vF        vector, function value at each location of vTheta
      vG        vector, first derivative
      vH        vector, second derivative
    """
    vF= -np.exp(-(vTheta - 1)**2) - 1.5 * np.exp(-(vTheta - 3)**2) - .2*np.sqrt(vTheta)
    vG= 2*(vTheta-1)*np.exp(-(vTheta - 1)**2) + 3 * (vTheta-3) * np.exp(-(vTheta - 3)**2) - .1 / np.sqrt(vTheta)
    vH= 2*np.exp(-(vTheta - 1)**2) * (1-2*(vTheta-1)**2) + 3*np.exp(-(vTheta - 3)**2) * (1 - 2*(vTheta-3)**2) + .05 / (vTheta * np.sqrt(vTheta))

    return (vF, vG, vH)

###########################################################
### StepNR(fnF, iR, dTheta0)
def StepNR(fnF, iR, dTheta0):
    """
    Purpose:
        Show steps of the Newton-Raphson algorithm

    Inputs:
        fnF         function to minimise
        iR          integer, number of steps
        dTheta0     double, initial estimate

    Return value:
        None
    """
    dTheta= dTheta0
    print ("Starting NR at theta= %g" % dTheta0)
    for i in range(iR):
        (dF, dG, dH)= fnF(dTheta)

        # Optimize approximating function q(h), finding new theta
        dStep= -dG/dH
        dTheta= dTheta + dStep
        print ("Step %i: g= %g, H= %g, h= %g, theta= %g" % (i, dG, dH, dStep, dTheta))

    print ("Final estimate: %g" % dTheta)

    return dTheta

###########################################################
### main
def main():
    # Magic numbers
    dTheta= 0.4             # Local minimum
    # dTheta= .1              # Local maximum
    # dTheta= 1               # Local maximum
    # dTheta= 5.9             # Quick global optimum

    iR= 10                  # Number of steps

    # Estimation
    dThetaEst= StepNR(fnF, iR, dTheta)

###########################################################
### start main
if __name__ == "__main__":
    main()
