#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
np_newton_show.py

Purpose:
    Show newton steps

Version:
    1       First start, following np_newton_show.ox

Date:
    2018/8/23

Author:
    Charles Bos
"""
###########################################################
### Imports
import numpy as np
import matplotlib.pyplot as plt

###########################################################
### (vF, vG, vH)= fnF(vTheta)
def fnF(vTheta):
    """
    Purpose:
      Calculate target function, plus first and second derivative

    Inputs:
      vTheta    vector, value for evaluation

    Return value:
      vF        vector, function value at each location of vTheta
      vG        vector, first derivative
      vH        vector, second derivative
    """
    vF= np.exp(-(vTheta - 1)**2) + 1.5 * np.exp(-(vTheta - 3)**2) + .2*np.sqrt(vTheta)

    vG= -2*(vTheta-1)*np.exp(-(vTheta - 1)**2) - 3 * (vTheta-3) * np.exp(-(vTheta - 3)**2) + .1 / np.sqrt(vTheta)
    vH= -2*np.exp(-(vTheta - 1)**2) * (1-2*(vTheta-1)**2) - 3*np.exp(-(vTheta - 3)**2) * (1 - 2*(vTheta-3)**2) - .05 / (vTheta * np.sqrt(vTheta))

    return (vF, vG, vH)

###########################################################
### vA= SolveQ(dF, dG, dH)
def SolveQ(dF, dG, dH):
    """
    Purpose:
      Find quadratic equation q(h)= a0 + a1 h + a2 h^2, fitting
          q(h)= f + g h + 0.5 H h^2

    Inputs:
      dF, dG, dH    scalars, function in h=0, gradient, hessian

    Return value:
      vA        1 x 3 vector with a0, a1, a2
    """
    return [dF, dG, 0.5*dH]

###########################################################
### vQ= EvalQ(vH, vA)
def EvalQ(vH, vA):
    """
    Purpose:
      Compute the value of the approximating function q(h)

    Inputs:
      vH        iN vector of locations, for evaluating q(h)
      vA        1 x 3 vector of parameters of quadratic function

    Return value:
      vQ        1 x iN vector, q(h)= a0 + a1*h + a2*h^2
    """
    return vA[0] + vA[1]*vH + vA[2]*vH**2

###########################################################
### StepNR(fnF, iR, dTheta0)
def StepNR(fnF, iR, dTheta0):
    """
    Purpose:
        Step over the Newton-Raphson algorithm
    """
    dTheta= dTheta0

    vFI= np.zeros(iR)
    vThetaI= np.zeros(iR+1)
    vThetaI[0]= dTheta
    mA= np.zeros((iR, 3))
    i=0
    for i in range(iR):
        (dF, dG, dH)= fnF(dTheta)

        # Find approximating function
        vA= SolveQ(dF, dG, dH)

        # Optimize Q, finding new theta
        dTheta= dTheta - 0.5*vA[1]/vA[2]

        vThetaI[i+1]= dTheta
        mA[i,:]= vA


    return (vThetaI, mA)

###########################################################
### ShowNR(vThetaI, mA, vTheta)
def ShowNR(fnF, vThetaI, mA, vTheta):
    """
    Purpose:
        Show steps of the Newton-Raphson algorithm

    Inputs:
        fnF         function to maximize
        vThetaI     iR+1 vector of thetas
        mA          iR x 3 matrix with coefficients of approximating model
        vTheta      iN vector, points for evaluation

    Return value:
        None
    """
    iR= np.size(vThetaI)-1

    vF= fnF(vTheta)[0]

    plt.ioff()          # Don't redraw until explicit draw() command
    fig=plt.figure(figsize=(8,3))   # Prepare a figure
    ax= fig.subplots()              # Extract the axes
    ax.plot(vTheta, vF, "r-")       # Plot the true f(theta) function
    plt.ylim(np.min(vF), np.ceil(np.max(vF)))  # Fix scale

    for i in range(iR):
        dTheta= vThetaI[i]
        dF= fnF(dTheta)[0]
        vA= mA[i,:]

        vQ= EvalQ(vTheta-dTheta, vA)
        dThetaOpt= dTheta - 0.5*vA[1]/vA[2]

        # Plot present approximation
        ax.plot(vTheta, vQ, "b-")

        # Indicate current estimate
        ax.plot(dTheta, dF, "rx")

        # Show, save, pause
        plt.draw()
        plt.savefig("graphs/np_newton_th%.0fi%i.png" % (10*vThetaI[0], i))
        plt.pause(1)

        # Overwrite current approximation in light colour, for next plot
        ax.plot(vTheta, vQ, "y-")
    plt.show()          # Keep last plot on screen

###########################################################
### main
def main():
    # Magic numbers
    vTheta= np.arange(0, 6, .05)
    dTheta= 0.4             # Local minimum
    # dTheta= .1              # Local maximum
    # dTheta= 1               # Local maximum
    # dTheta= 5.9             # Quick global optimum

    iR= 10                  # Number of steps

    # Estimation
    (vThetaI, mA)= StepNR(fnF, iR, dTheta)

    # Output
    ShowNR(fnF, vThetaI, mA, vTheta)

###########################################################
### start main
if __name__ == "__main__":
    main()
