#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
estnorm_tr.py

Purpose:
    Estimate a normal regression model, using lambda function

Version:
    1       Following estnorm.ox, using 1d parameter vectors
    tr      Use transformation

Date:
    2017/8/21, 2019/6/3

Author:
    Charles Bos
"""
###########################################################
### Imports
import numpy as np
import pandas as pd
#import matplotlib.pyplot as plt
import scipy.optimize as opt
import math

###########################################################
### Get hessian and related functions
from lib.grad import *

###########################################################
def GetPars(vP):
    """
    Purpose:
      Read out the parameters from the vector

    Inputs:
      vP        iK+1 vector with sigma and beta's

    Return value:
      dS        double, sigma
      vBeta     iK vector, beta's
    """
    iK= np.size(vP)-1
    # Force vP to be a 1D matrix
    vP= vP.reshape(iK+1,)
    dS= vP[0]
    vBeta= vP[1:]

    return (dS, vBeta)

###########################################################
def GetParNames(iK):
    """
    Purpose:
      Construct names for the parameters from the vector

    Inputs:
      iK        integer, number of beta's

    Return value:
      asP       iK array, with strings "sigma", "b1", ...
    """
    asP= ["B"+str(i+1) for i in range(iK)]
    asP= ["Sigma"] + asP

    return asP

###########################################################
def TransPar(vP):
    """
    Purpose:
      Transform the parameters for restrictions

    Inputs:
      vP        iP array of parameters, with sigma and beta's

    Return value:
      vPTr      iP array, with log(sigma) and beta's
    """
    vPTr= np.copy(vP)
    vPTr[0]= np.log(vP[0])

    return vPTr

###########################################################
def TransBackPar(vPTr):
    """
    Purpose:
      Transform the parameters back from restrictions

    Inputs:
      vPTr      iP array of parameters, with log(sigma) and beta's

    Return value:
      vP        iP array, with sigma and beta's
    """
    vP= np.copy(vPTr)
    vP[0]= np.exp(vPTr[0])

    return vP

###########################################################
def Generate(vP, iN):
    """
    Purpose:
      Generate regression data

    Inputs:
      vP        iK vector of parameters
      iN        integer, number of observations

    Return values:
      vY        iN vector of data
      mX        iN x iK matrix of regressors, constant + uniforms
    """
    (dS, vBeta)= GetPars(vP)

    iK= len(vBeta);

    mX= np.hstack([np.ones((iN, 1)), np.random.rand(iN, iK-1)])

    vY= mX@vBeta + dS * np.random.randn(iN)
    #print ("Y: ", vY.shape)

    return (vY, mX)

###########################################################
### dLL= AvgNLnLRegr(vP, vY, mX)
def AvgNLnLRegr(vP, vY, mX):
    """
    Purpose:
        Compute negative average loglikelihood of regression model

    Inputs:
        vP      iK+1 vector of parameters, with sigma and beta
        vY      iN vector of data
        mX      iN x iK matrix of regressors

    Return value:
        dNALL   double, negative average loglikelihood
    """
    (iN, iK)= mX.shape
    if (np.size(vP) != iK+1):         # Check if vP is as expected
        print ("Warning: wrong size vP= ", vP)

    (dSigma, vBeta)= GetPars(vP)
    if (dSigma <= 0):
        print ("x", end="")
        return math.inf

    vE= vY - mX @ vBeta

    vLL= -0.5*(np.log(2*np.pi) + 2*np.log(dSigma) + np.square(vE/dSigma))
    dALL= np.mean(vLL, axis= 0)

    print (".", end="")             # Give sign of life

    return -dALL

###########################################################
### (vP, vS, dLL, sMess)= EstimateRegr(vY, mX)
def EstimateRegr(vY, mX):
    """
    Purpose:
      Estimate the regression model

    Inputs:
      vY        iN vector of data
      mX        iN x iK matrix of regressors

    Return value:
      vP        iK+1 vector of optimal parameters sigma and beta's
      vS        iK+1 vector of standard deviations
      dLL       double, loglikelihood
      sMess     string, output of optimization
    """
    (iN, iK)= mX.shape
    vP0= np.ones(iK+1)        # Get (bad...) starting values

    dLL= -iN*AvgNLnLRegr(vP0, vY, mX)
    print ("Initial LL= ", dLL, "\nvP0=", vP0)

    # Use lambda function to transform back in place
    AvgNLnLRegrTr= lambda vPTr, vY, mX: AvgNLnLRegr(TransBackPar(vPTr), vY, mX)

    vP0Tr= TransPar(vP0)
    res= opt.minimize(AvgNLnLRegrTr, vP0Tr, args=(vY, mX), method="BFGS")

    vPTr= np.copy(res.x)
    vP= TransBackPar(vPTr)   # Remember to transform back!
    sMess= res.message
    dLL= -iN*res.fun
    print ("\nBFGS results in ", sMess, "\nPars: ", vP, "\nLL= ", dLL, ", f-eval= ", res.nfev)

    # Get standard errors, straight from LL without transformation
    mH= hessian_2sided(AvgNLnLRegr, vP, vY, mX)
    mS2= np.linalg.inv(mH)/iN
    # mS2= GetCovML(AvgNLnLRegr, vP, iN, vY, mX)
    vS= np.sqrt(np.diag(mS2))
    print ("\nst.dev, no transformation: ", vS)

    # Get standard errors, using delta method
    mH= hessian_2sided(AvgNLnLRegrTr, vPTr, vY, mX)
    mS2Th= np.linalg.inv(mH)/iN
    mG= jacobian_2sided(TransBackPar, vPTr)  # Evaluate jacobian at vPTr
    mS2= mG @ mS2Th @ mG.T                # Cov(vP)
    vS= np.sqrt(np.diag(mS2))             # s(vP)

    print ("\nJacobian, mG=\n", mG, "\nst.dev, delta method: ", vS)

    return (vP, vS, dLL, sMess)

###########################################################
### Output(mPPS, dLL, sMess)
def Output(mPPS, dLL, sMess):
    """
    Purpose:
      Provide output on screen
    """
    iK= mPPS.shape[1]-1
    print ("\n\nEstimation resulted in ", sMess)
    print ("Using ML with LL= ", dLL)

    print ("Parameter estimates:\n",
           pd.DataFrame(mPPS.T, index=GetParNames(iK), columns=["PTrue", "PHat", "s(P)"]))


###########################################################
### main
def main():
    vP0= [.1, 5, 2, -2]    #dSigma and vBeta together
    iN= 100
    iSeed= 1234

    #Generate data
    np.random.seed(iSeed)
    vP0= np.array(vP0)
    (vY, mX)= Generate(vP0, iN)

    (vP, vS, dLnPdf, sMess)= EstimateRegr(vY, mX)
    Output(np.vstack([vP0, vP, vS]), dLnPdf, sMess);

###########################################################
### start main
if __name__ == "__main__":
    main()
