#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
SolveMacro.py

Purpose:
    Solve problem from Macro, finding the equilibrium

Date:
    16/8/16, 17/8/22

Version:
    1       In python

Author:
    Charles Bos
"""
###########################################################
### Imports
import numpy as np
# import pandas as pd
# import matplotlib.pyplot as plt
import scipy.optimize as opt

###########################################################
### vF= EquilMacro(vP, vS)
def EquilMacro(vP, vS):
    """
    Purpose:
      Check the equilibrium in some specific problem from TI-Macro I

    Inputs:
      vP        2 vector with pH and Nu1
      vS        2 vector, relative risk aversions

    Return value:
      vF        2 vector, with distance from equilibrium
    """
    vF= np.ones_like(vP)

    (dpH, dNu1)= vP
    (dS1, dS2)= vS

    # if ((dNu1 < 0) or (dNu1 > 1) or (dpH == 0)):       # Check limits
    #     return None

    vF[0]= (1.0 / dpH)**(1.0 /dS1)*dNu1 + (1.0 / dpH)**(1.0 / dS2)*(1.0-dNu1)-2
    vF[1]= dpH**( (dS1-1)/dS1)*dNu1+dNu1-dpH-(1/2)

    # print ("f: ", vF)
    print (".", end="")         # Give sign of life

    return vF

###########################################################
### main
def main():
    # Magic numbers
    vS= [2, 2]        # Data
    vP= [10, .9]      # Initial parameters

    # Estimation
    vF= EquilMacro(vP, vS)
    print ("\nInitial distance vF= ", vF, "at vP= ", vP)

    EquilMacroL= lambda vP: EquilMacro(vP, vS)
    res= opt.least_squares(EquilMacroL, vP)

    # Output
    # print ("Full set of results: ", res)
    print ("\nNLS returns ", res.message, "\nParameters: ", res.x)
    print ("The distance to equilibrium is ", res.fun)

###########################################################
### start main
if __name__ == "__main__":
    main()
