#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
speed_pool.py

Purpose:
    Check speed of computation using loop with/without multiprocessing

Version:
    1       First start
    2       Use ThreadPool, which will lock CPU tasks, no use
    3       Use Pool, true multiprocessing

Date:
    2019/8/29

Author:
    Charles Bos
"""
###########################################################
### Imports
import numpy as np
from multiprocessing import Pool
from lib.inctime import *

###########################################################
### Loop(mX, iR)
def Loop(mX, iR):
    """
    Purpose:
        Pass time computing R times X'X, in a loop

    Inputs:
        mX  iN x iK matrix
        iR  integer, number of repetitions

    Return value:
        mXtX    iK x iK matrix of X'X
    """
    (iN, iK)= mX.shape
    for r in range(iR):
        mXtX= np.zeros((iK, iK))
        for i in range(iK):
            for j in range(i+1):
                for k in range(iN):
                    mXtX[i,j]+= mX[k,i] * mX[k,j]
                mXtX[j, i]= mXtX[i, j]
    return mXtX

###########################################################
### mXtX= LoopG(r)
def LoopG(r):
    """
    Purpose:
        Call Loop(), using the global version of mX, for a single iteration

    Inputs:
        r       integer, iteration
        g_mX    global iN x iK matrix, input data

    Return value:
        mXtX    iK x iK matrix
    """
    global g_mX
    return Loop(g_mX, 1)

###########################################################
### mXtX= LoopJ(mX, iR)
def LoopJ(mX, iR):
    """
    Purpose:
        Pass time computing R times X'X, in a loop, using inner function

    Inputs:
        mX  iN x iK matrix
        iR  integer, number of repetitions

    Return value:
        mXtX    iK x iK matrix of X'X
    """
    global g_mX             # Prepare a global for passing mX
    g_mX= mX                # Fill the global with the value of mX

    pool= Pool()            # Open the pool of processors, as many as possible
    lXtX= pool.map(LoopG, range(iR))        # Call LoopG, for each value r= 0, .., iR-1
                        # Store all results in the list lXtX

    # close the pool and wait for the work to finish
    pool.close()
    pool.join()

    return lXtX[0]       # Return only a single of those results

###########################################################
### MatMult(mX, iR)
def MatMult(mX, iR):
    """
    Purpose:
        Pass time computing R times X'X, using numpy

    Inputs:
        mX  iN x iK matrix
        iR  integer, number of repetitions

    Return value:
        mXtX    iK x iK matrix of X'X
    """
    for r in range(iR):
        mXtX= mX.T@mX
    return mXtX

###########################################################
### main
def main():
    # Magic numbers
    iN= 1000
    iK= 10
    iR= 100

    # Initialisation
    mX= np.hstack([np.ones((iN, 1)), np.random.randn(iN, iK-1)])

    print ("Calculation X'X for (%i x %i) matrix X, repeating R=%i times" % (iN, iK, iR))

    # Estimation
    with Timer("Loop, Rx"):
        mXtX= Loop(mX, iR)
    with Timer("LoopJ, multiprocessing, Rx"):
        mXtX= LoopJ(mX, iR)
    with Timer("MatMult, Rx"):
        mXtX= MatMult(mX, iR)

    # Output
    print ("Quite a difference...\n")

###########################################################
### start main
if __name__ == "__main__":
    main()
