/*
**  EstStack_ML
**
**  Purpose:
**    Estimate a regression model on the stackloss data set using
**    a maximum likelihood method
**
**  Inputs:
**    The program expects the file eststack.dec to contain
**    information on the data file to use, the variables, etc.
**
**  Author:
**    Charles Bos
**
**  Date:
**    3/10/06
*/
#include <oxstd.h>
#include <oxfloat.h>
#include <packages/oxutils/oxutils.h>
#import <database>
#import <maximize>

#include "eststack.dec"

static decl s_vY, s_mX;     // Two globals for likelihood function

/*
**  Initialise(const asYVar, const aasXVar, const asData)
**
**  Purpose:
**    Initialise the settings, from declaration file and parameter
**    arguments
*/
Initialise(const avP, const asYVar, const aasXVar, const asData)
{
  avP[0]= g_vPInit;
  asData[0]= g_sData;
  asYVar[0]= g_sYVar;
  aasXVar[0]= g_asXVar;
  
  if (ReadArg(avP, "p", 5))     // Read at most 5 parameters from data file
    avP[0]= vecr(avP[0]);       //   Make sure it's a column
  ReadArg(asData, "data", -1);  // Read string with data file
  ReadArg(asYVar, "y", -1);     // Read string with y-variable
  ReadArg(aasXVar, "x", -5);    // Read array of strings with x-variable
  
  ReadArgUsed();
}

/*
**  InitData(const avY, const amX, const sYVar, const asXVar, const sData)
**
**  Purpose:
**    Read the data
**
**  Inputs:
**    sYVar     string, y-var to use
**    asXVar    string, x-var to use
**    sData     string, data file to use
**
**  Outputs:
**    avY       iN x 1 vector with y-var
**    amX       iN x iK matrix with x-var
**
**  Return value:
**    1 if succeeded, zero otherwise
*/
InitData(const avY, const amX, const sYVar, const asXVar, const sData)
{
  decl db;
  
  db= new Database();           // Prepare object of Database class
  db.LoadIn7(sData);
  avY[0]= db.GetVar(sYVar);
  amX[0]= db.GetVar(asXVar);
  
  delete db;                    // Delete the object explicitly
      
  return (sizerc(avY[0]) > 0) && (sizerc(amX[0]) > 0);      
}

/*
**  AvgLnLiklRegr(const vP, const adLnPdf, const avScore, const amHess)
**
**  Purpose:
**    Compute the average log likelihood for a regression model
**
**  Inputs:
**    vP        iK+1 x 1 vector of parameters; last parameter is standard
**              dev
**    s_vY      iN x 1 vector of data
**    s_mX      iN x iK matrix of explanatory variables
**
**  Outputs:
**    adLnPdf   pointer to double with average loglikelihood
**
**  Return value:
**    1 if computation succeeded, zero otherwise
*/
AvgLnLiklRegr(const vP, const adLnPdf, const avScore, const amHess)
{
  decl vBeta, iK, iN, dSEps, vE;
  
  adLnPdf[0]= M_NAN;
  iK= columns(s_mX);
  iN= rows(s_vY);
  vBeta= vP[:iK-1];
  dSEps= vP[iK];
  if (dSEps < 0)
    return 0;
  
  vE= s_vY - s_mX*vBeta;
  adLnPdf[0]= -0.5*(log(M_2PI) + 2 * log(dSEps) + sumsqrc(vE)/(iN*sqr(dSEps)));
  
  return !ismissing(adLnPdf[0]);  
}

/*
**  Estimate(const avP, const avS, const adLL, const vY, const mX)
**
**  Purpose:
**    Estimate the model using one of the specified methods
**
**  Inputs:
**    ...
**
**  Return value:
**    ir    indicates type of convergence
*/
Estimate(const avP, const avS, const adLL, const vY, const mX)
{
  decl ir, mS2, iK, dS2, iN, mH, vBeta;
  
  iN= sizerc(vY);
  iK= columns(mX);
  mS2= constant(M_NAN, iK+1, iK+1);
  
  s_vY= vY;     // Prepare globals for AvgLnLiklRegr
  s_mX= mX;
  
  if (ismissing(avP[0]) || (sizerc(avP[0]) < iK+1))
    {
      println ("Initialising using OLS");
      olsc(vY, mX, &vBeta);
      avP[0]= vBeta|1;
    }  

  ir= MaxBFGS(AvgLnLiklRegr, avP, adLL, 0, TRUE);

  mS2= 0;
  if (Num2Derivative(AvgLnLiklRegr, avP[0], &mH))
    mS2= invertgen(-mH, 30)/iN, 
      
  avS[0]= sqrt(diagonal(mS2)');
  adLL[0]= double(adLL[0]);     // Make sure it's a double
  
  return ir;  
}

/*
**  Output(const asParNames, const vP, const vS, const dLL, const iN, const ir)
**
**  Purpose:
**    Provide output on screen
*/
Output(const asParNames, const vP, const vS, const dLL, const iN, const ir)
{
  print ("Estimation resulted in ");
  println (MaxConvergenceMsg(ir), 
           " using ML with LL= ", dLL*iN);
    
  println ("Parameter estimates: ", 
           "%c", {"p", "s"}, "%r", asParNames,
           vP~vS);
}

main()
{
  decl sYVar, asXVar, sData, ir, vP, vS, dLL, vY, mX;
  
  Initialise(&vP, &sYVar, &asXVar, &sData);         // Init settings
  InitData(&vY, &mX, sYVar, asXVar, sData);         // Read data
  
  ir= Estimate(&vP, &vS, &dLL, vY, mX);             // Estimate
  
  Output(asXVar~"sEps", vP, vS, dLL, sizerc(vY), ir);      // Give output
}
